
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include <assert.h>
#include <limits.h>
#include <sys/time.h>
#include <stdint.h>
#include <assert.h>

#include "HRTimer.hh"
#include "NbodyTypes.h"
#include "SerialPairwiseSimulator.h"
#include "SerialQTreeSimulator.h"
#include "QTree.h"

#define SEED 1025    // time(NULL) for true random


#define INITIAL_UNIVERSE_SIZE_X 40*149598000e3   // units: m width of the solar system
#define INITIAL_UNIVERSE_SIZE_Y 40*149598000e3   // units: m
#define MINIMUM_MASS (3.3e23)           // units: kg  
                                        // this minimum is the mass of mercury
#define MAXIMUM_MASS (1.9891e30)        // units: kg
                                        // this maximum is the mass of the sun
#define MAXIMUM_VELOCITY 50000          // units: m/s  (this is per x-y component) 
                                        // I set the maxiumim to reflect the orbital velocity of mercury

/* for the simulation output.  Defines the size of the printed universe 
 * Best accuracy when INITIAL_UNIVERSE_SIZE is a multiple of PRINTED_UNIVERSE_SIZE
 */
#define PRINTED_UNIVERSE_SIZE_X 20
#define PRINTED_UNIVERSE_SIZE_Y 20

int main(int argc, char* argv[])
{
  HRTimer timer;
  hrtime_t start_time, end_time;

  int num_bodies;
  double delta;
  int num_timesteps;
  
  if(argc == 4){
    num_bodies = atoi(argv[1]);
    delta = atof(argv[2]);
    num_timesteps = atoi(argv[3]);
  } else {
    printf("Usage: %s <# of particles> <delta t> <# of timesteps>\n", argv[0]);
    exit(1);
  }

  /* Phase 1: Universe generation */
  srand(SEED);  

#if defined(PAIRWISE)
  SerialPairwiseSimulator nbody_sim(0, INITIAL_UNIVERSE_SIZE_X, 0, INITIAL_UNIVERSE_SIZE_Y);
#elif defined(QTREE)
  SerialQTreeSimulator nbody_sim(0, INITIAL_UNIVERSE_SIZE_X, 0, INITIAL_UNIVERSE_SIZE_Y);
#endif
  nbody_sim.populate(num_bodies);

  printf("Initial state of universe:\n");
  nbody_sim.print(PRINTED_UNIVERSE_SIZE_X, PRINTED_UNIVERSE_SIZE_Y);
  printf("\n");

  start_time = timer.get_time_ns();
  nbody_sim.simulate(num_timesteps, delta);
  end_time = timer.get_time_ns();

  printf("Final State of Universe:\n");
  nbody_sim.print(PRINTED_UNIVERSE_SIZE_X, PRINTED_UNIVERSE_SIZE_Y);

  printf("Simulated time: ");
  double simulated_time = delta * num_timesteps;
  if( simulated_time >= 31536000 ) {
    int simulated_time_years = (int) (simulated_time / 31536000);
    printf("%i years ", simulated_time_years);
    simulated_time -= simulated_time_years * 31536000.0;
  }
  if(simulated_time >= 86400 ) {
    int simulated_time_days = (int) (simulated_time / 86400);
    printf("%i days ", simulated_time_days);
    simulated_time -= simulated_time_days * 86400.0;
  }
  if(simulated_time >= 3600 ) {
    int simulated_time_hours = (int) (simulated_time / 3600);
    printf("%i hours ", simulated_time_hours);
    simulated_time -= simulated_time_hours * 3600.0;
  }
  if(simulated_time >= 60 ) {
    int simulated_time_minutes = (int) (simulated_time / 60);
    printf("%i minutes ", simulated_time_minutes);
    simulated_time -= simulated_time_minutes * 60.0;
  }
  printf("%i seconds\n", (int) simulated_time);

  printf("simulation time: %lli\n", end_time - start_time);
}

