/* file "structure.cc" */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include <suif_copyright.h>

/* control-flow structuring for the porky program for SUIF */

#define RCS_BASE_FILE structure_cc

#include "porky.h"

RCS_BASE(
    "$Id: structure.cc,v 1.6 1995/12/12 06:30:25 cwilson Exp $")

/*----------------------------------------------------------------------*
    Begin Constant Declarations
 *----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*
    End Constant Declarations
 *----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*
    Begin Type Declarations
 *----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*
    End Type Declarations
 *----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*
    Begin Private Global Variables
 *----------------------------------------------------------------------*/

/*----------------------------------------------------------------------*
    End Private Global Variables
 *----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*
    Begin Private Function Declarations
 *----------------------------------------------------------------------*/

static void find_fors_on_node(tree_node *the_node, void *);

/*----------------------------------------------------------------------*
    End Private Function Declarations
 *----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*
    Begin Public Function Implementations
 *----------------------------------------------------------------------*/

extern void find_fors_on_proc(tree_proc *the_proc)
  {
    the_proc->map(&find_fors_on_node, NULL, FALSE);
  }

/*----------------------------------------------------------------------*
    End Public Function Implementations
 *----------------------------------------------------------------------*/
/*----------------------------------------------------------------------*
    Begin Private Function Implementations
 *----------------------------------------------------------------------*/

static void find_fors_on_node(tree_node *the_node, void *)
  {
    if (!the_node->is_loop())
        return;
    tree_loop *the_loop = (tree_loop *)the_node;

    tree_node *test_node = single_effect(the_loop->test());
    if (test_node == NULL)
        return;

    if (!test_node->is_instr())
        return;
    tree_instr *test_tree_instr = (tree_instr *)test_node;
    instruction *test_instr = test_tree_instr->instr();

    boolean test_reversed;
    switch (test_instr->opcode())
      {
        case io_btrue:
            test_reversed = FALSE;
            break;
        case io_bfalse:
            test_reversed = TRUE;
            break;
        default:
            return;
      }
    in_bj *test_bj = (in_bj *)test_instr;

    if (test_bj->target() != the_loop->toplab())
        return;

    operand test_op = test_bj->src_op();

    if (!test_op.is_expr())
        return;
    instruction *comp_instr = test_op.instr();

    if ((comp_instr->opcode() != io_sl) && (comp_instr->opcode() != io_sle))
        return;
    in_rrr *comp_rrr = (in_rrr *)comp_instr;

    var_sym *index;
    operand upper_bound;
    tree_for_test the_test;
    if (comp_rrr->src1_op().is_symbol() &&
        !might_modify(comp_rrr->src2_op(), the_loop->body()))
      {
        index = comp_rrr->src1_op().symbol();
        upper_bound = comp_rrr->src2_op();
        if (comp_instr->opcode() == io_sl)
            the_test = FOR_SLT;
        else
            the_test = FOR_SLTE;
      }
    else if (comp_rrr->src2_op().is_symbol() &&
             !might_modify(comp_rrr->src1_op(), the_loop->body()))
      {
        index = comp_rrr->src2_op().symbol();
        upper_bound = comp_rrr->src1_op();
        if (comp_instr->opcode() == io_sl)
            the_test = FOR_SGT;
        else
            the_test = FOR_SGTE;
      }
    else
      {
        return;
      }

    if (index->type()->is_volatile())
        return;

    if (test_reversed)
      {
        switch (the_test)
          {
            case FOR_SGT:
                the_test = FOR_SLTE;
                break;
            case FOR_SGTE:
                the_test = FOR_SLT;
                break;
            case FOR_SLT:
                the_test = FOR_SGTE;
                break;
            case FOR_SLTE:
                the_test = FOR_SGT;
                break;
            default:
                assert(FALSE);
          }
      }

    instruction *write_instr;
    int num_writes = possible_writes(the_loop->body(), index, &write_instr);
    if (num_writes != 1)
        return;

    if ((write_instr->opcode() != io_add) && (write_instr->opcode() != io_sub))
        return;
    in_rrr *write_rrr = (in_rrr *)write_instr;

    operand step;
    if ((write_rrr->src1_op() == operand(index)) &&
        !might_modify(write_rrr->src2_op(), the_loop->body()))
      {
        step = write_rrr->src2_op();
      }
    else if ((write_rrr->opcode() == io_add) &&
             (write_rrr->src2_op() == operand(index)) &&
             !might_modify(write_rrr->src1_op(), the_loop->body()))
      {
        step = write_rrr->src1_op();
      }
    else
      {
        return;
      }

    /* make sure the write is not in a tree_for */
    tree_node *follow_up = write_instr->owner();
    while (follow_up != the_loop)
      {
        assert(follow_up != NULL);
        assert(follow_up->parent() != NULL);
        if (follow_up->is_for())
            return;
        follow_up = follow_up->parent()->parent();
      }

    boolean is_signed;
    type_node *index_type = index->type()->unqual();
    switch (index_type->op())
      {
        case TYPE_INT:
        case TYPE_ENUM:
          {
            base_type *the_base = (base_type *)index_type;
            is_signed = the_base->is_signed();
            break;
          }
        case TYPE_FLOAT:
            is_signed = TRUE;
            break;
        case TYPE_PTR:
            is_signed = FALSE;
            break;
        default:
            return;
      }

    if (verbosity_level > 0)
      {
        printf("For detection: found for with index `%s' at line %d.\n",
               index->name(), source_line_num(the_loop));
      }

    upper_bound = upper_bound.clone();
    step = step.clone();

    if (write_instr->opcode() == io_sub)
      {
        step = operand(new in_rrr(io_neg, step.type()->unqual(), operand(),
                                  step));
      }

    tree_loop *new_loop = the_loop->clone();

    var_sym *lb_var =
            the_loop->scope()->new_unique_var(index->type()->unqual());

    operand lower_bound = operand(lb_var);

    tree_node_list *body = new_loop->body();
    new_loop->set_body(new tree_node_list);

    tree_node_list *then_part = new tree_node_list;

    in_rrr *lb_copy =
            new in_rrr(io_cpy, lb_var->type(), operand(lb_var),
                       operand(index));

    var_sym *new_index =
            the_loop->scope()->new_unique_var(step.type()->unqual());
    in_rrr *ind_set_instr =
            new in_rrr(io_add, index->type()->unqual(), operand(index),
                       lower_bound.clone(), operand(new_index));
    body->push(new tree_instr(ind_set_instr->clone()));
    then_part->append(new tree_instr(ind_set_instr));

    assert(new_index->type()->is_base());
    base_type *new_base = (base_type *)(new_index->type());
    is_signed = new_base->is_signed();

    upper_bound =
            fold_real_2op_rrr(io_sub, new_base, upper_bound, lower_bound);

    lower_bound = operand_int(new_base, 0);
    index = new_index;

    if (!is_signed)
      {
        switch (the_test)
          {
            case FOR_SGT:
                the_test = FOR_UGT;
                break;
            case FOR_SGTE:
                the_test = FOR_UGTE;
                break;
            case FOR_SLT:
                the_test = FOR_ULT;
                break;
            case FOR_SLTE:
                the_test = FOR_ULTE;
                break;
            default:
                assert(FALSE);
          }
      }

    tree_for *new_for =
            new tree_for(index, the_test, new_loop->contlab(),
                         new_loop->brklab(), body, lower_bound, upper_bound,
                         step, new tree_node_list);

    delete new_loop;

    then_part->push(new_for);

    then_part->push(new tree_instr(lb_copy));

    tree_node_list *else_part = new tree_node_list;

    assert(the_loop->scope()->is_block());
    block_symtab *block_scope = (block_symtab *)(the_loop->scope());
    label_sym *jumpto = block_scope->new_unique_label();

    tree_node_list *test_part = new tree_node_list;
    in_bj *bj_clone = (in_bj *)(test_bj->clone());
    if (bj_clone->opcode() == io_btrue)
        bj_clone->set_opcode(io_bfalse);
    else
        bj_clone->set_opcode(io_btrue);
    bj_clone->set_target(jumpto);
    test_part->append(new tree_instr(bj_clone));

    tree_if *the_if = new tree_if(jumpto, test_part, then_part, else_part);

    the_loop->parent()->insert_after(the_if, the_loop->list_e());

    the_loop->parent()->remove(the_loop->list_e());
    else_part->append(the_loop->list_e());
  }

/*----------------------------------------------------------------------*
    End Private Function Implementations
 *----------------------------------------------------------------------*/
