/*  Annotation Definitions */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#ifdef SUIFLIB
#include "suif_copyright.h"
#else
#include <suif/suif_copyright.h>
#endif

#ifndef ANNOTE_H
#define ANNOTE_H

#pragma interface

RCS_HEADER(annote_h,
    "$Id: annote.h,v 6.12 1995/10/17 23:20:41 cwilson Exp $")

class in_stream;
class out_stream;
class base_symtab;


/*
 *  Annotations are used to record arbitrary data associated with
 *  instructions and other suif_objects.  An annotation consists of
 *  a name and a void* value.  The annotation name must be an entry
 *  in the lexicon and is generally registered with the annotation
 *  manager.  There are two kinds of registered annotations: flat and
 *  structured.  For flat annotations the data field is a pointer to
 *  a list of immed values, whereas the data for structured annotations
 *  may be of any type.  The "immeds" and "set_immeds" methods only
 *  work with registered annotations.
 */

class annote {
private:
    char *nm;
    void *dt;

public:
    annote(in_stream *is, base_symtab *syms);
    annote(char *n, void *d = NULL);
    ~annote();
    
    char *name()			{ return nm; }
    void *data()			{ return dt; }
    immed_list *immeds();

    void set_name(char *n)		{ nm = n; }
    void set_data(void *d)		{ dt = d; }
    void set_immeds(immed_list *iml, suif_object *obj);

    boolean is_structured();

    void write(out_stream *os);
    void print(FILE *fp=stdout);
};


/*
 *  Annotation lists have added methods to retrieve annotations by
 *  name and to delete the contents of lists when they are destroyed.
 */

DECLARE_DLIST_CLASSES(annote_list_base, annote_list_e, annote_list_iter,
		      annote*);

class annote_list : public annote_list_base {
public:
    ~annote_list();

    annote *get_annote(char *name);
    annote *peek_annote(char *name);
};
    

/*
 *  Predefined annotations.  Most of these are only for the library's
 *  internal use, but a few of them are visible to users.
 */

void init_suif_annotes();

#ifdef DEFINE_ANNOTES
#define extern
#endif

/* annotations visible to users: */

/* initial data annotations */
extern char *k_fill;
extern char *k_repeat_init;
extern char *k_multi_init;

/* miscellaneous */
extern char *k_enable_exceptions;
extern char *k_line;
extern char *k_reg_num;
extern char *k_fields;
extern char *k_call_by_ref;
extern char *k_orig_type;
extern char *k_common_block;
extern char *k_history;
extern char *k_version_history;
extern char *k_source_file_name;

/* OBSOLETE!! The following are obsolete.  They are still provided for
   backward compatibility, but they will eventually be removed. */
extern char *k_common_field;
extern char *k_common_shape;


/* annotations for internal use: */

/* types */
extern char *k_int_type;
extern char *k_float_type;
extern char *k_void_type;
extern char *k_ptr_type;
extern char *k_array_type;
extern char *k_func_type;
extern char *k_group_type;
extern char *k_struct_type;
extern char *k_union_type;
extern char *k_enum_type;
extern char *k_const_type;
extern char *k_volatile_type;
extern char *k_call_by_ref_type;
extern char *k_null_type;

/* target machine parameters */
extern char *k_is_big_endian;
extern char *k_char_is_signed;
extern char *k_addressable_size;
extern char *k_C_type_size;
extern char *k_C_type_align;
extern char *k_array_align;
extern char *k_struct_align;
extern char *k_ptr_diff_type;

/* symtabs */
extern char *k_base_symtab;
extern char *k_block_symtab;
extern char *k_proc_symtab;
extern char *k_var_counter;
extern char *k_label_counter;
extern char *k_next_inum;
extern char *k_next_sym_id;
extern char *k_next_type_id;
extern char *k_param;
extern char *k_symbols;
extern char *k_types;
extern char *k_var_defs;
extern char *k_sym_annotes;
extern char *k_type_annotes;
extern char *k_var_def_annotes;

/* symbols */
extern char *k_labelsym;
extern char *k_procsym;
extern char *k_varsym;

/* variable definitions */
extern char *k_var_def;

/* tree nodes */
extern char *k_block;
extern char *k_for;
extern char *k_loop;
extern char *k_if;
extern char *k_list_end;

/* miscellaneous */
extern char *k_file_symtab_ptr;
extern char *k_proc_in_file;
extern char *k_file_num;
extern char *k_clone;

#ifdef DEFINE_ANNOTES
#undef extern
#endif

#endif /* ANNOTE_H */
