/*  Bit Set Implementation */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include "suif_copyright.h"

#define _MODULE_ "libsuif.a"

#pragma implementation "bitset.h"

#define RCS_BASE_FILE bitset_cc

#include "misc.h"
#include "bitset.h"

RCS_BASE(
    "$Id: bitset.cc,v 4.3 1994/05/02 23:13:52 cwilson Exp $")

/*  number of bits in a word */
const int w_size = sizeof(int) * suif_byte_size;


inline int
words (int i)
{
    return i / w_size;
}


/*
 *  This constructor builds a bit vector capable of representing sets
 *  with elements that are greater than or equal to "f" but strictly
 *  less than "l".  By default, it will clear the bit vector but this
 *  can be disabled using the no_clear flag.
 */

bit_set::bit_set (int f, int l, boolean no_clear)
{
    bits = NULL;
    expand(f, l, no_clear);
}


/*
 *  Create a new bitset and optionally clear all the bits.
 */

void
bit_set::expand (int f, int l, boolean no_clear)
{
    int sz;

    assert_msg(l >= f, ("bit_set::expand - length is negative"));
    delete bits;

    first = words(f) * w_size;
    sz = words(l - first - 1) + 1;
    last = first + (sz * w_size);

    bits = new int[sz];

    /* clear all the bits */
    if (!no_clear)
	clear();
}


void
bit_set::clear ()
{
    int i, *ip = bits;

    for (i = first; i < last; i += w_size) {
	*ip++ = 0;
    }
}


/*
 *  Set all bits to 1.
 */

void
bit_set::universal ()
{
    int i, *ip = bits;

    for (i = first; i < last; i += w_size) {
	*ip++ = ~0;
    }
}


/*
 *  Set one of the bits to 1.
 */

void
bit_set::add (int e)
{
    assert_msg(e >= first && e < last,
	       ("bit_set::add - bit %d not in range %d-%d", e, first, last));

    e -= first;
    bits[e / w_size] |= 1 << (e % w_size);
}


/*
 *  Clear one of the bits to 0.
 */

void
bit_set::remove (int e)
{
    assert_msg(e >= first && e < last,
	       ("bit_set::add - bit %d not in range %d-%d", e, first, last));

    e -= first;
    bits[e / w_size] &= ~(1 << (e % w_size));
}


/*
 *  Invert all the bits.
 */

void
bit_set::invert ()
{
    int i, *ip = bits;

    for (i = first; i < last; i += w_size) {
	*ip = ~*ip;
	ip++;
    }
}


/*
 *  Check if one of the bits is set.
 */

boolean
bit_set::contains (int e)
{
    if (e < first || e >= last)
	return FALSE;

    e -= first;
    return bits[e / w_size] & (1 << (e % w_size));
}


/*
 *  Create a bit-wise OR of two bitsets.
 */

void
bit_set::set_union (bit_set *l, bit_set *r)
{
    first = (l->first < r->first) ? l->first : r->first;
    last = (l->last > r->last) ? l->last : r->last;

    delete bits;
    bits = new int[words(last - first)];

    int i, *ip = bits;
    for (i = first; i < last; i += w_size) {
	*ip++ = (*l)[i] | (*r)[i];
    }
}


/*
 *  Create a bit-wise AND or two bitsets.
 */

void
bit_set::set_intersect (bit_set *l, bit_set *r)
{
    first = (r->first < l->first) ? l->first : r->first;
    last = (r->last > l->last) ? l->last : r->last;

    delete bits;
    bits = new int[words(last - first)];

    int i, *ip = bits;
    for (i = first; i < last; i += w_size) {
	*ip++ = (*l)[i] & (*r)[i];
    }
}


/*
 *  Copy a bit_set.
 */

void
bit_set::copy (bit_set *s)
{
    /* if not the same size, reallocate this bitset */
    if (first != s->first && last != s->last) {
	first = s->first;
	last = s->last;
	delete bits;
	bits = new int[words(last - first)];
    }

    /* copy the bits one word at a time */
    int i;
    int *ip = bits;
    int *jp = s->bits;
    for (i = first; i < last; i += w_size) {
	*ip++ = *jp++;
    }
}


/*
 *  Do a structure copy from a bit_set and optionally delete it.
 */

void
bit_set::transfer (bit_set *src, boolean del)
{
    if (bits)
	delete bits;
    *this = *src;
    src->bits = NULL;
    if (del) delete src;
}


/*
 *  Bit-wise OR with this bitset.
 */

void
bit_set::operator+= (bit_set &r)
{
    assert_msg(r.first >= first && last >= r.last,
	       ("bit_set::+= - bitset out of range"));

    int i, *ip = bits;
    for (i = first; i < last; i += w_size) {
	*ip++ |= r[i];
    }
}


/*
 *  Bit-wise AND with this bitset.
 */

void
bit_set::operator*= (bit_set &r)
{
    int i;
    if (first == r.first && last == r.last) {
	int *lp = bits;
	int *rp = r.bits;
	for (i = first; i < last; i += w_size) {
	    *lp++ &= *rp++;
	}
    } else {
	int *ip = bits;
	for (i = first; i < last; i += w_size) {
	    *ip++ &= r[i];
	}
    }
}


/*
 *  Bit-wise ANDNOT with this bitset.
 */

void
bit_set::operator-= (bit_set &r)
{
    int i, *ip = bits;
    for (i = first; i < last; i += w_size) {
	*ip++ &= ~r[i];
    }
}


/*
 *  Check if all the bits are the same.
 */

boolean
bit_set::operator== (bit_set &r)
{
    int start = (first < r.first) ? first : r.first;
    int end = (last > r.last) ? last : r.last;

    for ( ; start < end; start += w_size) {
	if ((*this)[start] != r[start]) return FALSE;
    }
    return TRUE;
}


/*
 *  Check if r contains at least the bits in this set.
 */

boolean
bit_set::operator<= (bit_set &r)
{
    int start = (first < r.first) ? first : r.first;
    int end = (last > r.last) ? last : r.last;

    for ( ; start < end; start += w_size) {
	int w = (*this)[start];
	if ((w & r[start]) != w) return FALSE;
    }
    return TRUE;
}


/*
 *  Test for non-empty intersection.
 */

boolean
bit_set::operator^ (bit_set &r)
{
    int i;
    if (first == r.first && last == r.last) {
	int *lp = bits;
	int *rp = r.bits;
	for (i = first; i < last; i += w_size) {
	    if (*lp++ & *rp++) return TRUE;
	}
    } else {
	int *ip = bits;
	for (i = first; i < last; i += w_size) {
	    if (*ip++ & r[i]) return TRUE;
	}
    }
    return FALSE;
}


/*
 *  Check if all bits are 0.
 */

boolean
bit_set::is_empty ()
{
    int *ip, sz;
    sz = (last - first) / w_size;
    for (ip = bits; sz > 0; sz--) {
	if (*ip++ != 0) return FALSE;
    }
    return TRUE;
}


/*
 *  Check if all bits are 1.
 */

boolean
bit_set::is_universal ()
{
    int *ip, sz;
    sz = (last - first) / w_size;
    for (ip = bits; sz > 0; sz--) {
	if (*ip++ != ~0) return FALSE;
    }
    return TRUE;
}


/*
 *  Count the number of one bits in this set.
 */

int
bit_set::count ()
{
    int cnt = 0;

    bit_set_iter bi(this);
    while (!bi.is_empty()) {
	bi.step();
	cnt++;
    }
    return cnt;
}


/*
 *  Print the bit_set.
 */

void
bit_set::print (FILE *fp, char *fmt)
{
    int i;
    fprintf(fp, "(%d:%d){", first, last);
    for (i = first; i < last; i++) {
	if (contains(i))
	    fprintf(fp, fmt, i);
    }
    putc('}', fp);
}


/*
 *  Retrieve an entire word of the bits.
 */

int
bit_set::operator[] (int i)
{
    if (i < first || i >= last)
	return 0;
    i -= first;
    return bits[i / w_size];
}


/*****************************************************************************/


/*
 *  Set up a bitset iterator.
 */

bit_set_iter::bit_set_iter (bit_set *st)
{
    s = st;
    reset();
}


void
bit_set_iter::reset ()
{
    e = 0;
    w = s->lb() - w_size;
}


/*
 *  Check if there are any more bits in this set.  Note: This function
 *  must be called once before each call to step().
 */

boolean
bit_set_iter::is_empty ()
{
    while (TRUE) {
	if (e != 0)
	    return FALSE;
	w += w_size;
	if (w >= s->ub())
	    break;
	e = (*s)[w];
    }
    return TRUE;
}


/*
 *  Return the position of the next bit.  This depends on is_empty() to
 *  load e with the word containing the next set bit.
 */

static int ffs_b(unsigned x);
static int ffs_w(unsigned x);

int
bit_set_iter::step ()
{
    int b = ffs_w(e);
    assert_msg(b >= 0, ("bit_set_iter::step - no more bits"));
    e &= ~(1 << b);
    return b + w;
}


/*
 *  The following functions are used to compute the position of the
 *  first bit that is set.  The ffs_n array contains the position of
 *  the first one for values between 0 and 15.  (When all of the bits
 *  are 0, we return -1.  Thus, ffs_n[0] = -5 so that when ffs_b() adds
 *  4, the result will still be -1.
 */

int ffs_n[] = {-5,0,1,0,2,0,1,0, 3,0,1,0,2,0,1,0 };

int
ffs_b (unsigned x)
{
    int b = ffs_n[x & 0xf];
    return (b >= 0) ? b : ffs_n[(x >> 4) & 0xf] + 4;
}


int
ffs_w (unsigned x)
{
    int y;
    if (y = (x & 0xff)) return ffs_b(y);
    if (y = ((x >> 8) & 0xff)) return ffs_b(y) + 8;
    if (y = ((x >> 16) & 0xff)) return ffs_b(y) + 16;
    if (y = ((x >> 24) & 0xff)) return ffs_b(y) + 24;
    return -1;
}


