/*  Bit Set Definitions */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#ifdef SUIFLIB
#include "suif_copyright.h"
#else
#include <suif/suif_copyright.h>
#endif

#ifndef BITSET_H
#define BITSET_H

#pragma interface

RCS_HEADER(bitset_h,
    "$Id: bitset.h,v 4.3 1995/02/26 13:11:22 cwilson Exp $")


/*
 *  The bit_set class is a bit vector representation for a set of
 *  integers.  The member "bits" is the bit vector that is used to
 *  represent integers greater than or equal to "first" but strictly
 *  less than "last".  To improve efficiency, first and last are always
 *  rounded down and up to the nearest multiple of bits in an integer,
 *  respectively.
 */

class bit_set {
    friend class bit_set_iter;

private:
    int first, last;
    int *bits;

    int operator[](int i);

public:
    bit_set()				{ first = 0; last = 0; bits = NULL; }
    bit_set(int f, int l, boolean no_clear = FALSE);
    ~bit_set()				{ delete bits; }

    void expand(int f, int l, boolean no_clear = FALSE);
    int lb()				{ return first; }
    int ub()				{ return last; }
    void clear();			/* clear all the bits to 0 */
    void universal();			/* set all bits to 1 */
    void add(int e);			/* set bit e to 1 */
    void remove(int e);			/* reset bit e to 0 */
    void invert();			/* invert all bits */
    boolean contains(int e);
    void set_union(bit_set *l, bit_set *r);
    void set_intersect(bit_set *l, bit_set *r);
    void copy(bit_set *s);
    void transfer(bit_set *src, boolean del = TRUE);
    void operator=(bit_set &b)		{ copy(&b); }
    void operator+=(bit_set &r);	/* bit-wise OR */
    void operator*=(bit_set &r);	/* bit-wise AND */
    void operator-=(bit_set &r);	/* bit-wise subtraction */
    boolean operator==(bit_set &r);	/* bit-wise comparison */
    boolean operator!=(bit_set &b)	{ return !(*this == b); }
    boolean operator<=(bit_set &r);	/* subset of the bits? */
    boolean operator^(bit_set &r);	/* test for non-empty intersection */
    boolean is_empty();			/* all zeros? */
    boolean is_universal();		/* all ones? */
    int count();			/* count the 1 bits */
    void print(FILE *fp = stdout, char *fmt = "%d,");
};


/*
 *  Bitset iterator.  This iterator works a little differently -- you
 *  must call is_empty() once before each call to step().
 */

class bit_set_iter {
    bit_set *s;
    int e, w;
public:
    bit_set_iter()			{ s = NULL; }
    bit_set_iter(bit_set *st);

    void reset(bit_set *st)		{ s = st; reset(); }
    void reset();
    int step();
    boolean is_empty();
};

#endif /* BITSET_H */
