/*  Library Initialization */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include "suif_copyright.h"

#define _MODULE_ "libsuif.a"

#pragma implementation "initsuif.h"

#define RCS_BASE_FILE initsuif_cc

#include "suif.h"
#include "suif_internal.h"
#include <string.h>

RCS_BASE(
    "$Id: initsuif.cc,v 1.12 1996/03/01 10:11:08 cwilson Exp $")


library_list *suif_libraries = NULL;

boolean warn_only_once = FALSE;


/*
 *  Options to be recognized by the command-line parser.
 */

static boolean version_only;
static cmd_line_option libsuif_options[] = {
    { CLO_NOARG, "-version",        "", &version_only },
    { CLO_NOARG, "-err-context",    "", &show_error_context },
    { CLO_NOARG, "-no-warn",        "", &ignore_warnings },
    { CLO_NOARG, "-warn-only-once", "", &warn_only_once },
    { CLO_NOARG, "-no-glob-check",  "", &disable_global_symtab_check }
};


/*  local function prototypes */
static void suif_new_handler();
static void init_suif_internal(int& argc, char *argv[]);
static void exit_suif_internal();


/*
 *  Add a library to the list of registered libraries.  This list records
 *  the library name, version, and compilation information along with
 *  pointers to the initialization and finalization functions for the library.
 */

void
register_library (char *nm, char *ver, char *info,
		  lib_init_f init_f, lib_exit_f exit_f)
{
    suif_library *new_lib = new suif_library;
    new_lib->name = nm;
    new_lib->version = ver;
    new_lib->compile_info = info;
    new_lib->initializer = init_f;
    new_lib->finalizer = exit_f;

    if (suif_libraries == NULL)
        suif_libraries = new library_list;
    suif_libraries->append(new_lib);
}


/*
 *  Initialize the SUIF library and any libraries that have been registered.
 */

void
init_suif (int& argc, char * argv[])
{
    int i;

    if (suif_libraries == NULL)
        suif_libraries = new library_list;

    /* record the entire command line */
    int cmd_line_len = 1;
    for (i = 0; i < argc; i++) {
	cmd_line_len += strlen(argv[i]) + 1;
    }
    _suif_command_line = new char[cmd_line_len];
    _suif_command_line[0] = '\0';
    if (argc > 0) {
	for (i = 0; i < argc - 1; i++) {
	    strcat(_suif_command_line, argv[i]);
	    strcat(_suif_command_line, " ");
	}
	strcat(_suif_command_line, argv[argc - 1]);
    }

    _suif_program_name = argv ? argv[0] : NULL;
    if (_suif_program_name != NULL) {
        _suif_prog_base_name = strrchr(_suif_program_name, '/');
        if (_suif_prog_base_name == NULL)
            _suif_prog_base_name = _suif_program_name;
        else
            ++_suif_prog_base_name;
    } else {
        _suif_prog_base_name = NULL;
    }

    set_new_handler(suif_new_handler);

    lexicon = new string_table;

    /* register the SUIF library itself */
    suif_library *new_lib = new suif_library;
    new_lib->name = "suif";
    new_lib->version = libsuif_ver_string;
    new_lib->compile_info = libsuif_who_string;
    new_lib->initializer = init_suif_internal;
    new_lib->finalizer = exit_suif_internal;
    suif_libraries->push(new_lib);

    parse_cmd_line(argc, argv, libsuif_options,
		   sizeof(libsuif_options) / sizeof(cmd_line_option));

    if (version_only) {
	fprintf(stderr, "%s %s %s\n", _suif_prog_base_name, prog_ver_string,
		prog_who_string);
	fprintf(stderr, "    Based on SUIF distribution %s\n",
                libsuif_suif_string);
	fprintf(stderr, "    Linked with:\n");
	library_list_iter libiter(suif_libraries);
	while (!libiter.is_empty()) {
	    suif_library *lib = libiter.step();
            fprintf(stderr, "  lib%s %s %s\n",
		    lib->name, lib->version, lib->compile_info);
	}
        exit(0);
    }

    /* call the initializers for the registered libraries */
    library_list_iter libiter(suif_libraries);
    while (!libiter.is_empty()) {
	suif_library *lib = libiter.step();
	if (lib->initializer) lib->initializer(argc, argv);
    }
}


void
suif_new_handler ()
{
    error_line(1, NULL, "Out of Memory");
}


/*
 *  This is the initialization function given when "init_suif" registers
 *  the SUIF library itself.  It initializes the annotation manager, adds
 *  the predefined annotations, and creates the file set.
 */

void
init_suif_internal (int& /* argc */, char * /* argv */ [])
{
    init_aman();
    init_suif_annotes();
    init_error_handler();

    fileset = new file_set;
}


/*
 *  SUIF library cleanup function.  Call the finalization functions for
 *  the registered libraries.  (This is not really necessary but it makes
 *  it easier to read the output from Purify.)
 */

void
exit_suif ()
{
    /* call the finalization functions in reverse order */
    while (!suif_libraries->is_empty()) {
	library_list_e *lib_e = suif_libraries->tail();
	suif_library *lib = lib_e->contents;
	delete suif_libraries->remove(lib_e);
	if (lib->finalizer) lib->finalizer();
	delete lib;
    }

    delete suif_libraries;
    suif_libraries = NULL;
}


/*
 *  This is the finalization function given when "init_suif" registers
 *  the SUIF library itself.  It deallocate all of the global data
 *  structures in the library.
 */

void
exit_suif_internal ()
{
    delete fileset;
    free_aman();
    delete lexicon;
    assert(clue_stack_empty());
}
