/*  SUIF Operands */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#ifdef SUIFLIB
#include "suif_copyright.h"
#else
#include <suif/suif_copyright.h>
#endif

#ifndef OPERAND_H
#define OPERAND_H

#pragma interface

RCS_HEADER(operand_h,
    "$Id: operand.h,v 5.14 1995/10/17 23:20:41 cwilson Exp $")

class in_stream;
class out_stream;
class base_symtab;
class var_sym;
class instruction;
class tree_node;
class type_node;
struct replacements;


/*
 *  Operands represent the sources and destinations of SUIF instructions.
 *  In the simplest cases, operands may either be null or refer directly to
 *  variables.  A source operand may also point to another instruction.  This
 *  is used to implement expression trees and, at lower levels of the system,
 *  to refer to the results of other instructions in a flat list.  An
 *  instruction pointer in a destination operand refers to the instruction
 *  that uses the result value.
 *
 *  The "is_expr" method tests if a source operand is a subexpression that is
 *  not contained in a separate tree_instr (i.e. it's not in a flat list).
 *  This method should not be used for destination operands.
 *
 *  Before changing an operand from an instruction pointer to some other
 *  value, the instruction must be removed.  The "remove" method checks if the
 *  operand is an instruction, and if so, calls the "instruction::remove"
 *  method for it.
 */

enum operand_kinds {
    OPER_NULL,				/* null operand */
    OPER_SYM,				/* variable symbol */
    OPER_INSTR				/* instruction */
};


class operand_dataonly {
friend class operand;

protected:
    operand_kinds k;
    union {
	var_sym *sym;
	instruction *i;
    } u;

public:
    operand_kinds kind() const		{ return k; }
    boolean is_null() const		{ return (k == OPER_NULL); }
    boolean is_symbol() const		{ return (k == OPER_SYM); }
    boolean is_instr() const		{ return (k == OPER_INSTR); }
    boolean is_expr() const;		/* is source operand a sub-expr? */

    var_sym *symbol() const;
    instruction *instr() const;

    boolean operator==(const operand_dataonly &r) const;
    boolean operator!=(const operand_dataonly &r) const
      { return !(*this == r); }
};


class operand : public operand_dataonly {
public:
    operand(const operand_dataonly &other)  { k = other.k; u = other.u; }
    operand(in_stream *is, base_symtab *syms, tree_node *t);
    operand(var_sym *s)			{ set_symbol(s); }
    operand(instruction *i)		{ set_instr(i); }
    operand()				{ u.sym = NULL; set_null(); }
    ~operand();
    
    void set_null()			{ k = OPER_NULL; }
    void set_symbol(var_sym *s)		{ k = OPER_SYM; u.sym = s; }
    void set_instr(instruction *i)	{ k = OPER_INSTR; u.i = i; }

    void remove();			/* remove src operand from expr tree */
    type_node *type();			/* find the type of the operand */
    boolean is_const_int(int *c=NULL) const;
					/* is this a constant integer? */

    operand& operator=(const operand &r) { k = r.k; u = r.u; return *this; }

    operand clone(base_symtab *dst_scope = NULL);
    operand clone_helper(replacements *r, boolean no_copy = FALSE);
    void find_exposed_refs(base_symtab *dst_scope, replacements *r);

    void write(out_stream *os) const;
    void print(instruction *i, FILE *f=stdout) const;
    void print(FILE *fp=stdout) const;
    void print_source(FILE *f=stdout) const	{ print(f); } /* obsolete */
};

DECLARE_DLIST_CLASS(operand_list, operand);

#endif /* OPERAND_H */
