/* file "init.cc" */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include <suif_copyright.h>

/*
 * This is the implementation of the routines dealing with
 * initialization annotations that go on vardefs.
 */

#define _MODULE_ "libuseful.a"

#pragma implementation "basic.h"

#define RCS_BASE_FILE libuseful_init_cc

#include "useful_internal.h"

RCS_BASE(
    "$Id: init.cc,v 1.8 1995/09/07 03:28:20 cwilson Exp $")

/*----------------------------------------------------------------------*
    Begin Documentation
 *----------------------------------------------------------------------*

                Summary
                -------

        This file is a place to put general routines to deal with the
        initialization annotations that go on vardefs.


 *----------------------------------------------------------------------*
    End Documentation
 *----------------------------------------------------------------------*/

/*
 *  Return the contents of a k_multi_init, k_repeat_init, or
 *  k_fill_init annote, respectively, in a convenient form.  The
 *  original data list is unchanged.
 */
static multi_init_struct translate_multi_init_annote(immed_list *data);
static repeat_init_struct translate_repeat_init_annote(immed_list *data);
static fill_init_struct translate_fill_init_annote(immed_list *data);

extern base_init_struct_list *read_init_data(var_def *the_def)
  {
    base_init_struct_list *new_list = new base_init_struct_list;

    annote_list_iter the_iter(the_def->annotes());
    while (!the_iter.is_empty())
      {
        annote *this_annote = the_iter.step();
        if (this_annote->name() == k_multi_init)
          {
            multi_init_struct *new_init_struct = new multi_init_struct;
            *new_init_struct =
                    translate_multi_init_annote(this_annote->immeds());
            new_list->append(new_init_struct);
          }
        else if (this_annote->name() == k_repeat_init)
          {
            repeat_init_struct *new_init_struct = new repeat_init_struct;
            *new_init_struct =
                    translate_repeat_init_annote(this_annote->immeds());
            new_list->append(new_init_struct);
          }
        else if (this_annote->name() == k_fill)
          {
            fill_init_struct *new_init_struct = new fill_init_struct;
            *new_init_struct =
                    translate_fill_init_annote(this_annote->immeds());
            new_list->append(new_init_struct);
          }
      }

    return new_list;
  }

extern void deallocate_init_data(base_init_struct_list *initializers)
  {
    while (!initializers->is_empty())
      {
        base_init_struct *this_init_struct = initializers->pop();
        if (this_init_struct->the_multi_init() != NULL)
            delete this_init_struct->the_multi_init()->data;
        delete this_init_struct;
      }
    delete initializers;
  }

extern void write_init_data(var_def *the_def, base_init_struct_list *data)
  {
    base_init_struct_list_iter data_iter(data);
    while (!data_iter.is_empty())
      {
        base_init_struct *this_init_struct = data_iter.step();
        if (this_init_struct->the_multi_init() != NULL)
          {
            multi_init_struct *the_multi_init =
                    this_init_struct->the_multi_init();

            immed_list *new_data = new immed_list;
            new_data->append(immed(the_multi_init->size));

            immed_list_iter the_immed_iter(the_multi_init->data);
            while (!the_immed_iter.is_empty())
              {
                immed this_immed = the_immed_iter.step();
                new_data->append(this_immed);
              }
            the_def->append_annote(k_multi_init, new_data);
          }
        else if (this_init_struct->the_repeat_init() != NULL)
          {
            repeat_init_struct *the_repeat_init =
                    this_init_struct->the_repeat_init();

            immed_list *new_data = new immed_list;
            new_data->append(immed(the_repeat_init->repetitions));
            new_data->append(immed(the_repeat_init->size));
            new_data->append(the_repeat_init->data);
            the_def->append_annote(k_repeat_init, new_data);
          }
        else if (this_init_struct->the_fill_init() != NULL)
          {
            fill_init_struct *the_fill_init =
                    this_init_struct->the_fill_init();

            immed_list *new_data = new immed_list;
            new_data->append(immed(the_fill_init->size));
            new_data->append(immed(the_fill_init->data));
            the_def->append_annote(k_fill, new_data);
          }
        else
          {
            assert(FALSE);
          }
      }
  }

extern boolean split_init_data(base_init_struct_list *original, int offset,
                               base_init_struct_list **chunk1,
                               base_init_struct_list **chunk2)
  {
    *chunk1 = new base_init_struct_list();

    int remaining_offset = offset;
    while (!original->is_empty())
      {
        base_init_struct *this_bis = original->head()->contents;
        int this_size = this_bis->total_size();
        if (this_size > remaining_offset)
            break;

        remaining_offset -= this_size;
        this_bis = original->pop();
        (*chunk1)->append(this_bis);
      }

    if (original->is_empty() || (remaining_offset == 0))
      {
        *chunk2 = original;
        return TRUE;
      }

    base_init_struct *this_bis = original->head()->contents;

    base_init_struct *new_init_struct = NULL;
    if (this_bis->the_multi_init() != NULL)
      {
        multi_init_struct *old_multi = this_bis->the_multi_init();
        if (remaining_offset % old_multi->size == 0)
          {
            remaining_offset /= old_multi->size;
            immed_list *new_list = new immed_list;
            while (remaining_offset > 0)
              {
                new_list->append(old_multi->data->pop());
                --remaining_offset;
              }
            new_init_struct = new multi_init_struct(old_multi->size, new_list);
          }
      }
    else if (this_bis->the_repeat_init() != NULL)
      {
        repeat_init_struct *old_repeat = this_bis->the_repeat_init();
        if (remaining_offset % old_repeat->size == 0)
          {
            remaining_offset /= old_repeat->size;
            new_init_struct =
                    new repeat_init_struct(remaining_offset, old_repeat->size,
                                           old_repeat->data);
            old_repeat->repetitions -= remaining_offset;
          }
      }
    else if (this_bis->the_fill_init() != NULL)
      {
        fill_init_struct *old_fill = this_bis->the_fill_init();
        new_init_struct =
                new fill_init_struct(remaining_offset, old_fill->data);
        old_fill->size -= remaining_offset;
      }
    else
      {
        assert(FALSE);
      }

    if (new_init_struct != NULL)
      {
        (*chunk1)->append(new_init_struct);
        *chunk2 = original;
        return TRUE;
      }
    else
      {
        original->push(*chunk1);
        delete *chunk1;
        *chunk1 = NULL;
        *chunk2 = NULL;
        return FALSE;
      }
  }

extern char *string_from_init_data(var_def *the_def)
  {
    base_init_struct_list *init_data = read_init_data(the_def);
    if (init_data == NULL)
        return NULL;
    if (init_data->is_empty())
      {
        deallocate_init_data(init_data);
        return NULL;
      }
    base_init_struct *head_init_struct = init_data->head()->contents;
    multi_init_struct *the_multi_init = head_init_struct->the_multi_init();
    if (the_multi_init == NULL)
      {
        deallocate_init_data(init_data);
        return NULL;
      }
    if (the_multi_init->size != target.size[C_char])
      {
        deallocate_init_data(init_data);
        return NULL;
      }
    immed_list *data = the_multi_init->data;
    assert(data != NULL);
    char *result = new char[data->count()];
    char *current_position = result;
    immed_list_iter data_iter(data);
    while (!data_iter.is_empty())
      {
        immed data_item = data_iter.step();
        if (!data_item.is_int_const())
          {
            deallocate_init_data(init_data);
            delete[] result;
            return NULL;
          }
        i_integer data_ii = immed_to_ii(data_item);
        if (!data_ii.is_c_char())
          {
            deallocate_init_data(init_data);
            delete[] result;
            return NULL;
          }
        char this_char = data_ii.c_char();
        *current_position = this_char;
        if (this_char == 0)
          {
            deallocate_init_data(init_data);
            return result;
          }
        ++current_position;
      }
    deallocate_init_data(init_data);
    delete[] result;
    return NULL;
  }


static multi_init_struct translate_multi_init_annote(immed_list *data)
  {
    if ((data->count() < 1) || (!((*data)[0].is_integer())))
        error_line(1, NULL, "bad format for \"%s\" annote", k_multi_init);

    immed_list_iter data_iter(data);

    immed first = data_iter.step();
    int size = first.integer();

    immed_list *new_data = new immed_list;
    while (!data_iter.is_empty())
      {
        immed this_data = data_iter.step();
        new_data->append(this_data);
      }

    return multi_init_struct(size, new_data);
  }

static repeat_init_struct translate_repeat_init_annote(immed_list *data)
  {
    if ((data->count() != 3) || (!((*data)[0].is_integer())) ||
        (!((*data)[1].is_integer())))
      {
        error_line(1, NULL, "bad format for \"%s\" annote", k_repeat_init);
      }

    int repetitions = (*data)[0].integer();
    int size = (*data)[1].integer();
    immed new_data = (*data)[2];

    return repeat_init_struct(repetitions, size, new_data);
  }

static fill_init_struct translate_fill_init_annote(immed_list *data)
  {
    if ((data->count() != 2) || (!((*data)[0].is_integer())) ||
        (!((*data)[1].is_integer())))
      {
        error_line(1, NULL, "bad format for \"%s\" annote", k_fill);
      }

    int size = (*data)[0].integer();
    int new_data = (*data)[1].integer();

    return fill_init_struct(size, new_data);
  }
