/*  Convert SUIF to conform to Simple SUIF conventions */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include <suif_copyright.h>

#define _MODULE_ "libsimple.a"

#include <suif.h>
#include "globals.h"

/*  local function declarations */
static operand convert_src(instruction *i, operand r, tree_node *tn);
static void convert_dst(instruction *i, tree_node *tn);
static in_ldc *add_src_ldc(instruction *i, operand r);
static void add_cpy(instruction *i, tree_node *tn);
static in_ldc *add_dst_ldc(instruction *i);



/*  This function is intented to be called by the tree_node "map" function to
    convert a procedure to conform to Simple SUIF's requirements.  Spilled
    variables can no longer be accessed directly, so we have to insert LOD/
    STR/MEMCPY instructions to access them.  Simple SUIF also requires that
    only CPY instructions can assign to pseudo registers (non-spilled
    variables), so we have to insert CPY instructions here, too.  */

void
convert_tree (tree_node *tn, void *)
{
    if (!tn->is_instr())
	simple_error("invalid tree_node -- not an instruction");

    tree_node_list *l = tn->parent();
    tree_instr *ti = (tree_instr *)tn;
    instruction *i = ti->instr();

    switch (i->opcode()) {

	case io_cpy: {
	    in_rrr *cpyi = (in_rrr *)i;

	    in_ldc *src_ldc = add_src_ldc(cpyi, cpyi->src_op());
	    in_ldc *dst_ldc = add_dst_ldc(cpyi);

	    if (src_ldc) {
		if (dst_ldc) {

		    l->insert_before(new tree_instr(src_ldc), tn->list_e());
		    l->insert_before(new tree_instr(dst_ldc), tn->list_e());

		    /* change the cpy instruction to a memcpy */
		    cpyi->set_opcode(io_memcpy);
		    cpyi->set_result_type(type_void);
		    cpyi->set_dst(operand());
		    cpyi->set_src_addr_op(operand(src_ldc));
		    cpyi->set_dst_addr_op(operand(dst_ldc));

		} else {

		    l->insert_before(new tree_instr(src_ldc), tn->list_e());

		    /* change the cpy to a lod */
		    cpyi->set_opcode(io_lod);
		    cpyi->set_src_addr_op(operand(src_ldc));

		    add_cpy(cpyi, tn);
		}

	    } else if (dst_ldc) {

		/* change copy instruction to a store */
		operand src = cpyi->src_op();
		src.remove();

		(void)l->insert_before(new tree_instr(dst_ldc), tn->list_e());

		cpyi->set_opcode(io_str);
		cpyi->set_result_type(type_void);
		cpyi->set_dst(operand());
		cpyi->set_src(src);
		cpyi->set_dst_addr_op(operand(dst_ldc));
	    }

	    break;
	}

	case io_lod: {
	    in_rrr *lodi = (in_rrr *)i;

	    /* first convert the source address as usual */
	    operand src_addr = convert_src(lodi, lodi->src_addr_op(), tn);

	    in_ldc *dst_ldc = add_dst_ldc(lodi);
	    if (dst_ldc) {

		(void)l->insert_before(new tree_instr(dst_ldc), tn->list_e());

		/* change the load to a memcpy */
		src_addr.remove();
		lodi->set_opcode(io_memcpy);
		lodi->set_result_type(type_void);
		lodi->set_dst(operand());
		lodi->set_src_addr_op(src_addr);
		lodi->set_dst_addr_op(operand(dst_ldc));

	    } else {

		lodi->set_src_addr_op(src_addr);
		add_cpy(lodi, tn);
	    }

	    break;
	}

	case io_str: {
	    in_rrr *stri = (in_rrr *)i;

	    /* first convert the source address as usual */
	    operand dst_addr = convert_src(stri, stri->dst_addr_op(), tn);

	    in_ldc *src_ldc = add_src_ldc(stri, stri->src_op());
	    if (src_ldc) {

		(void)l->insert_before(new tree_instr(src_ldc), tn->list_e());

		/* change the str to a memcpy */
		dst_addr.remove();
		stri->set_opcode(io_memcpy);
		stri->set_result_type(type_void);
		stri->set_dst(operand());
		stri->set_src_addr_op(operand(src_ldc));
		stri->set_dst_addr_op(dst_addr);

	    } else {

		stri->set_dst_addr_op(dst_addr);
	    }

	    break;
	}

	default: {

	    /* convert each source operand */
	    for (unsigned n = 0; n < i->num_srcs(); n++) {
		i->set_src_op(n, convert_src(i, i->src_op(n), tn));
	    }

	    /* convert the destination operand */
	    convert_dst(i, tn);
	}
    }
}



/*  Check if a source operand is a spilled variable, and if so, insert a LOD
    instruction to access the variable.  */

operand
convert_src (instruction *i, operand r, tree_node *tn)
{
    in_ldc *ldci = add_src_ldc(i, r);
    if (ldci) {
	/* generate a load instruction */
	in_rrr *lodi = new in_rrr(io_lod, r.type());
	(void)tn->parent()->insert_before(new tree_instr(ldci), tn->list_e());
	(void)tn->parent()->insert_before(new tree_instr(lodi), tn->list_e());
	lodi->set_src_addr_op(operand(ldci));

	/* check if the simple_reg number for the lod is specified */
	if (i->are_annotations()) {
	    annote *an = i->annotes()->get_annote(k_simple_reg);
	    if (an) lodi->annotes()->push(an);
	}

	return operand(lodi);
    }
    return r;
}



/*  Check if a source operand is a spilled variable, and if so, generate an
    LDC instruction to get the address of the variable.  This function is used
    by "convert_src" and also by the code to handle CPY/LOD/STR instructions
    in "convert_tree".  */

in_ldc *
add_src_ldc (instruction *i, operand r)
{
    if (!r.is_symbol() || !r.symbol()->is_spilled()) return NULL;

    /*  In Simple SUIF only non-addressed scalar local variables may be used
	as source operands.  Other operands have to be explicitly loaded
	before use, so here we have to add instructions to load spilled
	variables.  */

    /* generate an ldc instruction to get the address */
    type_node *pt = new_ptr_type(r.symbol()->type());
    in_ldc *ldci = new in_ldc(pt, operand(), immed(r.symbol()));

    /* check if the simple_reg number for the ldc is specified */
    if (i->are_annotations()) {
	annote *an = i->annotes()->get_annote(k_simple_reg);
	if (an) ldci->annotes()->push(an);
    }

    return ldci;
}



/*  Check if the destination operand needs to be explicitly stored,
    and if so, generate the instructions to store it.  */

void
convert_dst (instruction *i, tree_node *tn)
{
    in_ldc *ldci = add_dst_ldc(i);
    if (ldci) {
	/* make the dst be a new store instruction */
	in_rrr *stri = new in_rrr(io_str);
	(void)tn->parent()->insert_after(new tree_instr(stri), tn->list_e());
	(void)tn->parent()->insert_after(new tree_instr(ldci), tn->list_e());
	stri->set_dst_addr_op(operand(ldci));
	stri->set_src(operand(i));
    } else {
	add_cpy(i, tn);
    }
}



/*  Check if the destination operand is a spilled variable, and if so,
    generate an LDC instruction to get its address.  This function is used
    by the "convert_dst" function and also by the code to handle CPY/LOD/STR
    instruction in "convert_tree".  */

in_ldc *
add_dst_ldc (instruction *i)
{
    operand dst = i->dst_op();
    if (!dst.is_symbol() || !dst.symbol()->is_spilled()) return NULL;

    /* generate an ldc instruction to get the address */
    type_node *pt = new_ptr_type(dst.symbol()->type());
    in_ldc *ldci = new in_ldc(pt, operand(), immed(dst.symbol()));

    /* check if the simple_reg number for the ldc is specified */
    if (i->are_annotations()) {
	annote *an = i->annotes()->get_annote(k_simple_reg);
	if (an) ldci->annotes()->push(an);
    }

    return ldci;
}



/*  Check if the instruction is not a CPY and assigns to a variable.  If so,
    Simple SUIF requires that we insert a CPY to assign to the variable.  */

void
add_cpy (instruction *i, tree_node *tn)
{
    /* only CPY instructions can assign to symbols in Simple SUIF */
    if (i->opcode() == io_cpy) return;

    operand dst = i->dst_op();
    if (dst.is_symbol() && !dst.symbol()->is_spilled()) {

	in_rrr *cpyi = new in_rrr(io_cpy, i->result_type(), dst, operand(i));
	(void)tn->parent()->insert_after(new tree_instr(cpyi), tn->list_e());
    }
}


