/*  Functions to get rid of unnecessary instructions from Simple SUIF */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include <suif_copyright.h>

#define _MODULE_ "libsimple.a"

#include <suif.h>
#include "globals.h"

/*  local function declarations */
static var_sym *eliminate_ldc(operand op, type_node *op_type);
static operand optimize_src(instruction *i, operand r);



/*  This function is intended to be called by the tree_node "map" method to
    optimize away unnecessary instructions that were required for Simple
    SUIF.  It is important that the "map" be done in postorder (that makes
    it much easier to delete instructions).  */

void
optimize_tree (tree_node *tn, void *)
{
    if (!tn->is_instr()) return;
    tree_instr *ti = (tree_instr *)tn;
    instruction *i = ti->instr();

    switch (i->opcode()) {

	case io_lod: {
	    in_rrr *lodi = (in_rrr *)i;
	    var_sym *v = eliminate_ldc(lodi->src_addr_op(),
				       lodi->result_type());
	    if (v) {
		/* replace the load with a copy -- this may be optimized
		   away when the result of the copy is used */
		lodi->set_opcode(io_cpy);
		lodi->set_src(operand(v));
	    }
	    break;
	}

	case io_str: {
	    in_rrr *stri = (in_rrr *)i;
	    operand src = stri->src_op();
	    var_sym *v = eliminate_ldc(stri->dst_addr_op(), src.type());
	    if (v) {

		if (src.is_instr() && within_expr(stri, src.instr())) {
		    instruction *si = src.instr();

		    /* make source instr assign directly to the symbol */
		    src.remove();
		    si->set_dst(operand(v));

		    /* insert the simple_reg annotations from the store */
		    si->annotes()->insert_before(stri->annotes(),
						 si->annotes()->tail());

		    /* delete the str instruction */
		    delete ti->parent()->remove(ti->list_e());
		    delete ti;
		    i = NULL;

		} else {

		    /* change the store to a copy */
		    stri->set_opcode(io_cpy);
		    stri->set_result_type(src.type()->unqual());
		    stri->set_dst(operand(v));
		    stri->set_src1(src);
		    stri->set_src2(operand());
		}
	    }
	    break;
	}

	case io_memcpy: {
	    in_rrr *mcpyi = (in_rrr *)i;
	    operand src_addr = mcpyi->src_addr_op();
	    operand dst_addr = mcpyi->dst_addr_op();
	    ptr_type *ptyp = (ptr_type *)mcpyi->src_addr_op().type();
	    assert_msg(ptyp->is_ptr(), ("optimize_tree - memcpy src operand "
					"is not a pointer"));
	    var_sym *dstv = eliminate_ldc(dst_addr, ptyp->ref_type());
	    var_sym *srcv = eliminate_ldc(src_addr, ptyp->ref_type());
	    if (srcv) {
		if (dstv) {

		    /* change the memcpy to a cpy */
		    mcpyi->set_opcode(io_cpy);
		    mcpyi->set_result_type(dstv->type()->unqual());
		    mcpyi->set_dst(operand(dstv));
		    mcpyi->set_src1(operand(srcv));
		    mcpyi->set_src2(operand());

		} else {

		    /* change the memcpy to a store */
		    dst_addr.remove();
		    mcpyi->set_opcode(io_str);
		    mcpyi->set_dst_addr_op(dst_addr);
		    mcpyi->set_src(operand(srcv));
		}

	    } else if (dstv) {

		/* change the memcpy to a load */
		src_addr.remove();
		mcpyi->set_src1(operand());
		mcpyi->set_src2(operand());
		mcpyi->set_opcode(io_lod);
		mcpyi->set_result_type(dstv->type()->unqual());
		mcpyi->set_dst(operand(dstv));
		mcpyi->set_src_addr_op(src_addr);
	    }
	    break;
	}

	case io_cpy: {
	    /* get rid of extra copies inserted for Simple SUIF */
	    in_rrr *cpyi = (in_rrr *)i;
	    if (!cpyi->dst_op().is_instr() &&
		cpyi->src_op().is_instr() &&
		within_expr(cpyi, cpyi->src_op().instr())) {

		/* make the src instruction assign directly to the dst */
		instruction *srci = cpyi->src_op().instr();
		srci->remove();
		srci->set_dst(cpyi->dst_op());

		/* leave the "simple_reg" annotation on the src instr even
		   though it no longer assigns to a node -- the cpy will
		   be added back if the instruction is reconverted to Simple
		   SUIF */

		/* delete the copy instruction */
		delete ti->parent()->remove(ti->list_e());
		delete ti;
		i = NULL;
	    }
	    break;
	}

	default: {
	    break;
	}
    }

    /* check the source operands to remove extra cpys */
    if (i) {
	/* go in reverse order because of the simple_reg annotations */
	for (unsigned n = i->num_srcs(); n > 0; n--) {
	    i->set_src_op(n - 1, optimize_src(i, i->src_op(n - 1)));
	}
    }
}



/*  Check if an operand is an LDC instruction (presumably for a LOD/STR/MEMCPY
    instruction) that can be replaced by a direct reference to a variable.
    If so, remove the LDC and return the variable.  The annotation recording
    the simple_reg number used by the LDC is transferred to the destination
    instruction.  */

var_sym *
eliminate_ldc (operand op, type_node *op_type)
{
    if (op.is_instr() && (op.instr()->opcode() == io_ldc)) {

	in_ldc *ldci = (in_ldc *)op.instr();
	sym_addr addr = ldci->value().addr();

	/*  a direct reference to a symbol is only allowed if it is a variable
	    and the offset is zero and it is accessed using the correct type */

	if (addr.symbol()->is_var() &&
	    (addr.offset() == 0) &&
	    (op_type == ((var_sym *)addr.symbol())->type())) {

	    /* record the ldc's node number */
	    ldci->dst_op().instr()->annotes()->push(ldci->annotes());

	    /* remove the ldc */
	    tree_instr *ldcti = ldci->parent();
	    assert(ldcti->instr() == ldci);
	    ldci->remove();
	    delete ldcti->parent()->remove(ldcti->list_e());
	    delete ldcti;

	    return (var_sym *)addr.symbol();
	}
    }
    return NULL;
}



/*  Check if a source operand is a CPY instruction left over from a LOD that
    was removed.  If so, get rid of the CPY and return its source operand.
    The annotation recording the simple_reg number used by the CPY is
    transferred to the current instruction.  */

operand
optimize_src (instruction *i, operand r)
{
    in_rrr *cpyi;

    /* check if the operand is a cpy produced from a lod that was removed */
    if (!r.is_instr() ||
	((cpyi = (in_rrr *)r.instr())->opcode() != io_cpy) ||
	!cpyi->src_op().is_symbol() ||
	!cpyi->src_op().symbol()->is_spilled()) {
	return r;
    }

    if (within_expr(i, cpyi)) {

	operand cpy_src = cpyi->src_op();
	cpy_src.remove();

	/* record the cpy's node number(s) */
	i->annotes()->push(cpyi->annotes());

	/* remove the copy instruction */
	tree_instr *ti = cpyi->parent();
	delete ti->parent()->remove(ti->list_e());
	delete ti;

	return cpy_src;
    }

    return r;
}


