/*  Register Map Implementation */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include <suif_copyright.h>

#define _MODULE_ "libsimple.a"

#pragma implementation "regmap.h"

#include <stdlib.h>
#include <suif.h>
#include "globals.h"



/*  machine-specific register stuff */
#ifdef __mips
/*  register information for MIPS architectures */
int num_machine_regs[] = { 8, 6, 6, 1 };
simple_reg ** machine_regs[MACHINE_REG_KINDS];
static int callee_gpr_regs[] = { 16, 17, 18, 19, 20, 21, 22, 23 };
static int caller_gpr_regs[] = { 8, 9, 10, 11, 12, 13 };
static int callee_fpr_regs[] = { 52, 54, 56, 58, 60, 62 };
static int caller_fpr_regs[] = { 36 };
#else
/*  register allocation is currently not supported on other machines */
int num_machine_regs[] = { 0, 0, 0, 0 };
simple_reg ** machine_regs[MACHINE_REG_KINDS];
static int callee_gpr_regs[] = { 0 };
static int caller_gpr_regs[] = { 0 };
static int callee_fpr_regs[] = { 0 };
static int caller_fpr_regs[] = { 0 };
#endif /* __mips */



/*  Create all of the hard registers and enter them in the machine_regs
    array.  The suif_sym fields may be filled in later if the input file
    contains references to machine registers.  */

static void assign_reg_nums(int reg_type, int *regs, simple_type *t);

void
init_machine_regs ()
{
    simple_type *t;

    /* assume all GPRs have signed 32-bit types */
    t = simple_type_signed;
    assign_reg_nums(CALLEE_GPRS, callee_gpr_regs, t);
    assign_reg_nums(CALLER_GPRS, caller_gpr_regs, t);

    /* assume all FPRs have float 64-bit types */
    t = simple_type_double;
    assign_reg_nums(CALLEE_FPRS, callee_fpr_regs, t);
    assign_reg_nums(CALLER_FPRS, caller_fpr_regs, t);
}



void
assign_reg_nums (int reg_type, int *regs, simple_type *t)
{
    /* allocate the array of machine registers */
    machine_regs[reg_type] = new simple_reg*[num_machine_regs[reg_type]];

    for (int j = 0; j < num_machine_regs[reg_type]; j++) {
	simple_reg *reg = new_reg(t, MACHINE_REG, regs[j]);
	machine_regs[reg_type][j] = reg;
    }
}



/*****************************************************************************/



reg_map::reg_map (int sz)
{
    size = sz;
    bucket = new alist[sz];
    nxt = 1;
}



reg_map::~reg_map ()
{
    empty();
    delete[] bucket;
}



/*  Enter a simple_reg structure into the table for the specified register.
    An error occurs if the register was already in the table.  */

void
reg_map::set_reg (int r, simple_reg *reg)
{
    int ndx = hash(r);

    /* make sure that this register is not already in the table */
    if (bucket[ndx].search((void *)r)) {
	simple_error("duplicate register in register table");
    }

    (void)bucket[ndx].enter((void *)r, (void *)reg);

    /* adjust the next number counter (the input file may specify register
       numbers for some of the variables) */
    if (r >= nxt) nxt = r + 1;
}



/*  Check if a simple_reg structure has been entered for the given register,
    and if so, return it.  */

simple_reg *
reg_map::get_reg (int r)
{
    int ndx = hash(r);
    alist_e *a = bucket[ndx].search((void *)r);
    if (!a) return NULL;
    return (simple_reg *)(a->info);
}



/*  Empty out all the buckets and delete their contents.  */

void
reg_map::empty ()
{
    alist *current_bucket;
    int ndx = 0;

    while (ndx < size) {
	current_bucket = &bucket[ndx];
	while (!current_bucket->is_empty()) {

	    /* delete the entry */
	    alist_e *a = current_bucket->pop();
	    simple_reg *r = (simple_reg *)(a->info);
	    delete r;
	    delete a;
	}
	ndx += 1;
    }

    /* reset the next number counter */
    nxt = 1;
}



