/*  Simple SUIF Driver and Miscellaneous Functions */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include <suif_copyright.h>

#define _MODULE_ "libsimple.a"

#include <stdlib.h>
#include <suif.h>
#include "globals.h"

/*  global variables */
char *k_simple_reg;
char *k_simple_machine_reg;
char *k_simple_sym;
char *k_simple_type;
simple_type *simple_type_void;
simple_type *simple_type_char;
simple_type *simple_type_signed;
simple_type *simple_type_unsigned;
simple_type *simple_type_float;
simple_type *simple_type_double;
simple_type *simple_type_address;

/*  local variables */
static reg_map *regs;
static op_map *ops;

static void cleanup_symtab(base_symtab *st);
static void cleanup_machine_regs();



/*  Read an input file, convert each procedure to Simple SUIF, call the user's
    function for each, convert the results back to SUIF, and write the output.
    This is the main function used to access the Simple SUIF library.  */

void
compile (char *infile, char *outfile)
{
    ops = new op_map;
    regs = new reg_map(REG_MAP_SIZE);
    init_machine_regs();

    /* register annotations */
    ANNOTE(k_simple_reg, "simple_reg", TRUE);
    k_simple_machine_reg = lexicon->enter("simple_machine_reg")->sp;
    k_simple_sym = lexicon->enter("simple_sym")->sp;
    k_simple_type = lexicon->enter("simple_type")->sp;

    fileset->add_file(infile, outfile);

    fileset->reset_iter();
    file_set_entry *fse = fileset->next_file();
    
    /* convert the global symbol tables */
    simplify_globals(fileset->globals(), fse->symtab());

    proc_sym *ps;
    fse->reset_proc_iter();
    while (ps = fse->next_proc()) {
	ps->read_proc(FALSE, FALSE);
            
	/* convert the procedure to Simple SUIF */
	annote *line_annote = NULL;
	simple_instr *slist = simplify_proc(ps, regs, ops, &line_annote);

	/* call the user's procedure */
	slist = do_procedure(slist, ps->name());

	/* translate back to a SUIF procedure */
	translate_proc(slist, ps, ops, line_annote);

	regs->empty();
	cleanup_symtab(ps->block()->symtab());

	ps->write_proc(fse);
	ps->flush_proc();
    }

    delete regs;

    cleanup_machine_regs();
    cleanup_symtab(fse->symtab());
    cleanup_symtab(fileset->globals());

    delete ops;
    delete fileset;
}



/*  Delete the simple_sym and simple_type structures and remove their
    annotations from the symbol table. */

void
cleanup_symtab (base_symtab *st)
{
    sym_node_list_iter snli(st->symbols());
    while (!snli.is_empty()) {
	sym_node *sn = snli.step();
	simple_sym *sym = (simple_sym *)sn->get_annote(k_simple_sym);
	if (sym) delete sym;
    }

    type_node_list_iter tnli(st->types());
    while (!tnli.is_empty()) {
	type_node *tn = tnli.step();
	simple_type *typ = (simple_type *)tn->get_annote(k_simple_type);
	if (typ) delete typ;
    }
}



/*  Delete the simple_reg and simple_sym structures for the machine
    registers.  */

void
cleanup_machine_regs ()
{
    for (int i = 0; i < MACHINE_REG_KINDS; i++) {
	for (int j = 0; j < num_machine_regs[i]; j++) {
	    simple_reg *r = machine_regs[i][j];
	    simple_sym *sym = r->var;
	    assert_msg(sym, ("cleanup_machine_regs - missing variable"));
	    delete sym;
	    delete r;
	    machine_regs[i][j] = NULL;
	}
	num_machine_regs[i] = 0;
    }
}



void
_simple_error (char *msg, char *file, int line)
{
    fprintf(stderr, "Error: %s (%s:%d)\n", msg, file, line);
    exit(-1);
}



void
_simple_warning (char *msg, char *file, int line)
{
    fprintf(stderr, "Warning: %s (%s:%d)\n", msg, file, line);
}


