/*  Translate SUIF Instructions to Simple Instructions */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include <suif_copyright.h>

#define _MODULE_ "libsimple.a"

#include <stdlib.h>
#include <string.h>
#include <suif.h>
#include "globals.h"


/*  The simplify_proc function defined here converts a SUIF procedure to
    a list of Simple SUIF instructions.  Any macros or array instructions in
    the procedure must have already been expanded; these are not allowed in
    Simple SUIF.

    To avoid name conflicts between the library and the users, all of the
    global objects are passed as parameters and then stored into static
    variables for local use.  */

/*  functions to translate the various SUIF objects */
static void simplify_types(type_node_list *tnl);
static void simplify_symbols(sym_node_list *snl, reg_map *regs);
static void simplify_instr(tree_node *tn, void *);
static simple_reg *simplify_dest(operand r, type_node *t, instruction *i);
static simple_reg *simplify_operand(operand r);
static simple_sym *simplify_symbol(sym_node *sn);
static simple_type *simplify_type(type_node *t);
static void simplify_immed(immed i, simple_immed *v);

/*  miscellaneous helper functions */
static void emit_instr(simple_instr *s);

/*  local variables */
static proc_sym *current_proc;		/* the current procedure */
static reg_map *regs;			/* if_reg -> simple_reg map */
static op_map *ops;			/* opcode conversion tables */
static simple_instr *tail;		/* ptr to tail of instruction list */
static annote **line_annote_ptr;	/* first line number annotation */



/*  Create Simple SUIF equivalents for the symbols and types in the global
    and file symbol tables.  Also set the predefined simple_type pointers.  */

void
simplify_globals (global_symtab *gsyms, file_symtab *fsyms)
{
    simplify_types(gsyms->types());
    simplify_symbols(gsyms->symbols(), NULL);

    simplify_types(fsyms->types());
    simplify_symbols(fsyms->symbols(), NULL);

    /* translate the predefined types */
    simple_type_void = simplify_type(type_void);
    simple_type_char = simplify_type(type_char);
    simple_type_signed = simplify_type(type_signed);
    simple_type_unsigned = simplify_type(type_unsigned);
    simple_type_float = simplify_type(type_float);
    simple_type_double = simplify_type(type_double);
    simple_type_address = simplify_type(type_ptr);
}



/*  Create simple_type structures for the given list of type_nodes.
    A "simple_type" annotation on each type_node records the simple_type
    pointer.  Modifier types (and function types) are not translated, so
    we have to be careful to use unqualified types everywhere.  */

void
simplify_types (type_node_list *tnl)
{
    type_node_list_iter tnli(tnl);
    while (!tnli.is_empty()) {
	type_node *tn = tnli.step();
	simple_type_base b;
	switch (tn->op()) {

	    case TYPE_VOID:	{ b = VOID_TYPE; break; }
	    case TYPE_FLOAT:	{ b = FLOAT_TYPE; break; }
	    case TYPE_PTR:	{ b = ADDRESS_TYPE; break; }
	    case TYPE_STRUCT:	{ b = RECORD_TYPE; break; }
	    case TYPE_UNION:	{ b = RECORD_TYPE; break; }
	    case TYPE_ARRAY:	{ b = RECORD_TYPE; break; }

	    case TYPE_INT: {
		base_type *bt = (base_type *)tn;
		if (bt->is_signed()) {
		    b = SIGNED_TYPE;
		} else {
		    b = UNSIGNED_TYPE;
		}
		break;
	    }

	    case TYPE_ENUM: {
		enum_type *et = (enum_type *)tn;
		if (et->is_signed()) {
		    b = SIGNED_TYPE;
		} else {
		    b = UNSIGNED_TYPE;
		}
		break;
	    }

	    default: {
		/* other types are not used in Simple SUIF */
		continue;
	    }
	}

	/* create the new simple_type structure */
	simple_type *st = new simple_type;
	st->base = b;
	st->len = tn->size();
	st->suif_type = (void *)tn;

	/* attach it to the type_node as an annotation */
	annote *an = new annote(k_simple_type, (void *)st);
	tn->annotes()->push(an);
    }
}



/*  Create a simple_sym structure for every symbol in the given list that has
    not already been translated.  For procedure symbol tables, this function
    should first be called with the parameter list so that those variables are
    given the first register numbers; after that it can be called again to
    translate the rest of the symbols, ignoring the parameters.  */

void
simplify_symbols (sym_node_list *snl, reg_map *regs)
{
    sym_node_list_iter snli(snl);
    while (!snli.is_empty()) {
	sym_node *sn = snli.step();

	/* skip symbols that have already been translated (this should only
	   happen with parameter variables) */
	if (sn->peek_annote(k_simple_sym)) continue;

	/* special case for machine registers */
	if (sn->is_var() && ((var_sym *)sn)->is_reg()) {

	    /* get the register number from the annotations */
	    annote *an = sn->annotes()->peek_annote(k_reg_num);
	    if (!an) simple_error("missing machine register number");
	    int mr = (*an->immeds())[0].integer();

	    /* search through the machine_registers array */
	    simple_reg *r = NULL;
	    for (int i = 0; i < MACHINE_REG_KINDS; i++) {
		for (int j = 0; j < num_machine_regs[i]; j++) {
		    if (machine_regs[i][j]->num == mr) {
			r = machine_regs[i][j];
			break;
		    }
		}
	    }
	    if (!r) simple_error("machine register does not exist");

	    /* record the sym_node */
	    assert(r->var);
	    if (r->var->suif_sym) {
		simple_error("multiple suif_syms for machine register");
	    }
	    r->var->suif_sym = (void *)sn;

	    /* record the simple_reg on an annotation */
	    annote *a = new annote(k_simple_machine_reg, (void *)r);
	    sn->annotes()->push(a);

	    continue;
	}

	/* create a new simple_sym */
	simple_sym *sym = new simple_sym;
	sym->suif_sym = (void *)sn;
	sym->name = sn->name();

	switch (sn->kind()) {

	    case SYM_PROC: {
		sym->kind = PROC_SYM;
		sym->type = simple_type_void;
		break;
	    }

	    case SYM_LABEL: {
		sym->kind = LABEL_SYM;
		sym->type = simple_type_void;
		break;
	    }

	    case SYM_VAR: {
		var_sym *vs = (var_sym *)sn;
		sym->kind = VAR_SYM;
		sym->type = simplify_type(vs->type());

		if (!vs->is_spilled()) {

		    /* non-addressed, scalar locals go into simple_regs */
		    assert_msg(regs, ("simplify_symbols - "
				      "missing register map"));
		    int rnum;

		    /* check if the symbol already has a register number */
		    annote *a = sn->annotes()->peek_annote(k_simple_reg);
		    if (a) {
			rnum = (*a->immeds())[0].integer();
		    } else  {
			/* get the next unused register number */
			rnum = regs->next_num();
		    }

		    /* create the new register */
		    simple_reg *reg = new simple_reg;
		    reg->num = rnum;
		    reg->kind = PSEUDO_REG;
		    reg->var = sym;

		    /* put it in the register table */
		    regs->set_reg(rnum, reg);

		    /* record the number in an annotation on the variable */
		    if (!a) {
			a = new annote(k_simple_reg);
			a->immeds()->push(immed(rnum));
			sn->annotes()->push(a);
		    }
		}
		break;
	    }
	}

	/* attach it to the sym_node as an annotation */
	annote *an = new annote(k_simple_sym, (void *)sym);
	sn->annotes()->push(an);
    }
}



/*  Convert a SUIF procedure to a list of simple_instrs.  In the process,
    all of the SUIF instructions are removed from the procedure and deleted. */

simple_instr *
simplify_proc (proc_sym *p, reg_map *r, op_map *o, annote **lp)
{
    /* record the parameters */
    current_proc = p;
    regs = r;
    ops = o;
    line_annote_ptr = lp;

    proc_symtab *psyms = p->block()->proc_syms();
    if (!psyms->children()->is_empty())
	simple_error("procedure contains nested blocks");

    /* translate the local types and symbols */
    simplify_types(psyms->types());
    simplify_symbols(psyms->params(), regs);
    simplify_symbols(psyms->symbols(), regs);

    /* insert lods, strs, and cpys required by Simple SUIF */
    p->block()->body()->map(convert_tree, NULL);

    /* create a temporary simple_instr node to hold the output list */
    simple_instr *head = new simple_instr;
    head->prev = NULL;
    tail = head;

    /* convert the body of the procedure */
    p->block()->body()->map(simplify_instr, NULL);
    tail->next = NULL;

    /* remove the dummy node from the head of the output list */
    tail = head; 
    head = head->next;
    head->prev = NULL;
    delete tail;

    /* delete the original SUIF code */
    tree_node_list *tnl = p->block()->body();
    while (!tnl->is_empty()) {
	delete tnl->pop();
    }

    return head;
}



/*  Convert a SUIF instruction. */

void
simplify_instr (tree_node *tn, void *)
{
    if (!tn->is_instr())
	simple_error("invalid tree_node -- not an instruction");

    tree_instr *ti = (tree_instr *)tn;
    instruction *in = ti->instr();

    simple_instr *s = NULL;
    simple_op op = ops->simplify_opcode(in->opcode());

    switch (in->opcode()) {

	case io_mrk: {
	    /* ignore MRKs but save the first line number annotation */
	    if (!(*line_annote_ptr) && in->are_annotations()) {
		*line_annote_ptr = in->annotes()->get_annote(k_line);
	    }
	    break;
	}

	case io_ldc: {
	    in_ldc *ldci = (in_ldc *)in;
	    s = new_instr(op, simplify_type(ldci->result_type()));
	    s->u.ldc.dst =
		simplify_dest(ldci->dst_op(), ldci->result_type(), in);
	    simplify_immed(ldci->value(), &s->u.ldc.value);
	    break;
	}

	case io_btrue:
	case io_bfalse:
	case io_jmp: {
	    in_bj *bji = (in_bj *)in;
	    s = new_instr(op, simple_type_void);
	    s->u.bj.src = simplify_operand(bji->src_op());
	    s->u.bj.target = simplify_symbol(bji->target());
	    break;
	}

	case io_cal: {
	    in_cal *cali = (in_cal *)in;
	    s = new_instr(op, simplify_type(cali->result_type()));
	    s->u.call.dst =
		simplify_dest(cali->dst_op(), cali->result_type(), in);
	    s->u.call.proc = simplify_operand(cali->addr_op());

	    unsigned nargs = cali->num_args();
	    s->u.call.nargs = nargs;
	    s->u.call.args = new simple_reg*[nargs];
	    for (unsigned arg = 0; arg < nargs; arg++) {
		s->u.call.args[arg] = simplify_operand(cali->argument(arg));
	    }
	    break;
	}

	case io_mbr: {
	    in_mbr *mbri = (in_mbr *)in;
	    s = new_instr(op, simple_type_void);
	    s->u.mbr.deflab = simplify_symbol(mbri->default_lab());
	    s->u.mbr.src = simplify_operand(mbri->src_op());
	    s->u.mbr.offset = mbri->lower();

	    unsigned ntargets = mbri->num_labs();
	    s->u.mbr.ntargets = ntargets;
	    s->u.mbr.targets = new simple_sym*[ntargets];
	    for (unsigned t = 0; t < ntargets; t++) {
		s->u.mbr.targets[t] = simplify_symbol(mbri->label(t));
	    }
	    break;
	}

	case io_lab: {
	    in_lab *labi = (in_lab *)in;
	    s = new_instr(op, simple_type_void);
	    s->u.label.lab = simplify_symbol(labi->label());
	    break;
	}

	case io_min:
	case io_max:
	case io_abs:
	case io_divfloor:
	case io_divceil:
	case io_array: {
	    simple_error("unexpected opcode");
	    break;
	}

	default: {
	    /* make sure that the instruction is the right format */
	    if (in->format() != inf_rrr) {
		simple_error("unexpected instruction format");
	    }
	    in_rrr *rrri = (in_rrr *)in;
	    s = new_instr(op, simplify_type(rrri->result_type()));
	    s->u.base.dst =
		simplify_dest(rrri->dst_op(), rrri->result_type(), in);
	    s->u.base.src1 = simplify_operand(rrri->src1_op());
	    s->u.base.src2 = simplify_operand(rrri->src2_op());
	}
    }

    if (s) emit_instr(s);
}



/*  Because node registers do not have any type information associated with
    them, the type must be explicitly specified when a node register is
    first referenced.  Since node registers must be defined before they are
    used, this function is used to convert destination registers.  If the
    operand is a node, it is converted with the specified type; otherwise,
    it is converted as usual.  */

simple_reg *
simplify_dest (operand r, type_node *t, instruction *i)
{
    if (r.is_instr()) {

	int rnum;
	annote *an = i->annotes()->peek_annote(k_simple_reg);
	if (an) {
	    rnum = (*an->immeds())[0].integer();
	} else {
	    rnum = regs->next_num();
	}

	/* create a new temporary register */
	simple_reg *node = new_reg(simplify_type(t), TEMP_REG, rnum);
	regs->set_reg(rnum, node);

	/* record the register number on an annotation */
	if (!an) {
	    an = new annote(k_simple_reg);
	    an->immeds()->push(immed(rnum));
	    i->annotes()->push(an);
	}

	return node;
    }

    return simplify_operand(r);
}



/*  Translate a SUIF operand to a simple_reg.  This is just a matter of
    reading annotations on var_syms or instructions.  */

simple_reg *
simplify_operand (operand r)
{
    annote *an;
    suif_object *obj;

    switch (r.kind()) {
	case OPER_NULL: {
	    return NO_REGISTER;
	}
	case OPER_SYM: {
	    if (r.symbol()->is_reg()) {

		/* get the register number from the var_sym's annotations */
		simple_reg *mr = (simple_reg *)r.symbol()->
		    peek_annote(k_simple_machine_reg);
		assert_msg(mr, ("simplify_operand - cannot find "
				"machine register annotation"));
		return mr;
	    }
	    obj = r.symbol();
	    break;
	}
	case OPER_INSTR: {
	    obj = r.instr();
	    break;
	}
    }

    an = obj->annotes()->peek_annote(k_simple_reg);
    assert_msg(an, ("simplify_operand - cannot find register annotation"));
    int rnum = (*an->immeds())[0].integer();
    simple_reg *result = regs->get_reg(rnum);
    assert_msg(result,("simplify_operand - cannot find register %d",rnum));
    return result;
}



simple_sym *
simplify_symbol (sym_node *sn)
{
    simple_sym *sym = (simple_sym *)sn->peek_annote(k_simple_sym);
    assert_msg(sym, ("simplify_symbol - cannot find symbol annotation"));
    return sym;
}



simple_type *
simplify_type (type_node *t)
{
    t = t->unqual();
    simple_type *st = (simple_type *)t->peek_annote(k_simple_type);
    assert_msg(st, ("simplify_type - cannot find type annotation"));
    return st;
}



/*  Convert a SUIF immed to a simple_immed.  Because this function only
    handles immeds that appear in LDC instructions, only integers, floats,
    and symbols are accepted; other kinds of immeds will cause errors.  */

void
simplify_immed (immed i, simple_immed *v)
{
    switch (i.kind()) {

	case im_int:
	case im_extended_int: {
	    v->format = IMMED_INT;
	    if (i.is_integer())
		v->u.ival = i.integer();
	    else if (i.is_unsigned_int())
		v->u.ival = (int)(i.unsigned_int());
	    else
		simple_error("overflow in integer immed");
	    break;
	}

	case im_float:
	case im_extended_float: {
	    v->format = IMMED_FLOAT;
	    v->u.fval = i.force_double();
	    break;
	}

	case im_symbol: {
	    v->format = IMMED_SYMBOL;
	    v->u.s.symbol = simplify_symbol(i.symbol());
	    v->u.s.offset = i.offset();
	    break;
	}

	default: {
	    simple_error("unexpected immed kind");
	}
    }
}



/*  Append an instruction to the end of the current list.  The pointer to
    the tail of the list is kept in a global variable so we don't have to
    pass it all over the place.  */

void
emit_instr (simple_instr *s)
{
    tail->next = s;
    s->prev = tail;
    tail = s;
}



/*  Create a new simple_reg.  This function is only used internally;
    it is not directly accessible by users.  */

simple_reg *
new_reg (simple_type *t, reg_kind knd, int rnum)
{
    /* create a name for the new variable */
    char *name = new char[10];
    switch (knd) {
	case PSEUDO_REG: {
	    sprintf(name, "_preg%d", rnum);
	    break;
	}
	case TEMP_REG: {
	    sprintf(name, "_node%d", rnum);
	    break;
	}
	case MACHINE_REG: {
	    sprintf(name, "_hard%d", rnum);
	    break;
	}
    }

    /* create a simple_sym for the new register */
    simple_sym *rsym = new simple_sym;
    rsym->suif_sym = NULL;
    rsym->kind = VAR_SYM;
    rsym->type = t;
    rsym->name = name;

    /* create the new register */
    simple_reg *reg = new simple_reg;
    reg->num = rnum;
    reg->kind = knd;
    reg->var = rsym;

    return reg;
}



/*  Install a ptr_type.  Because we've already converted the types, if the
    type did not already exist we also have to create a simple_type for the
    new ptr_type.  */

type_node *
new_ptr_type (type_node *t)
{
    type_node *pt = t->parent()->install_type(new ptr_type(t));

    /* check if the ptr_type already existed */
    if (!pt->peek_annote(k_simple_type)) {

	/* create the new simple_type structure */
	simple_type *st = new simple_type;
	st->base = ADDRESS_TYPE;
	st->len = pt->size();
	st->suif_type = (void *)pt;

	/* attach it to the type_node as an annotation */
	annote *an = new annote(k_simple_type, (void *)st);
	pt->annotes()->push(an);
    }

    return pt;
}



/*****************************************************************************/



/*  Constructor for simple_instrs.  */

extern "C" simple_instr *
new_instr (simple_op op, simple_type *t)
{
    simple_instr *s = new simple_instr;
    s->opcode = op;
    s->type = t;
    s->next = NULL;

    return s;
}



/*  Deallocate a simple instruction.  The call and mbr instructions both
    contain data in separate arrays which must also be deallocated when the
    instructions are deleted.  */

extern "C" void
free_instr (simple_instr *s)
{
    switch (s->opcode) {
	case CALL_OP: {
	    delete[] s->u.call.args;
	    break;
	}
	case MBR_OP: {
	    delete[] s->u.mbr.targets;
	    break;
	}
	default: {
	    break;
	}
    }

    delete s;
}



/*  Create a new simple_reg.  This function is provided to the user to
    allocate new registers.  It will not create new machine registers.  */

extern "C" simple_reg *
new_register (simple_type *t, reg_kind knd)
{
    /* get the next unused register number */
    int rnum = regs->next_num();

    if (knd == MACHINE_REG)
	simple_error("cannot create new MACHINE_REGs");

    simple_reg *reg = new_reg(t, knd, rnum);
    regs->set_reg(rnum, reg);
    return reg;
}



/*  Create a new label and return a simple_sym structure for it.  This
    function should be called by the user's C program whenever a new label
    is needed.  */

extern "C" simple_sym *
new_label ()
{
    block_symtab *symtab = current_proc->block()->symtab();
    sym_node *lab = symtab->new_unique_label();

    /* create a new simple_sym */
    simple_sym *sym = new simple_sym;
    sym->suif_sym = (void *)lab;
    sym->name = lab->name();
    sym->kind = LABEL_SYM;
    sym->type = simple_type_void;

    /* attach it to the sym_node as an annotation */
    annote *an = new annote(k_simple_sym, (void *)sym);
    lab->annotes()->push(an);

    return sym;
}



/*  Get a type that is a pointer to the specified type.  This function is
    available to users to create new pointer types.  */

extern "C" simple_type *
get_ptr_type (simple_type *t)
{
    type_node *st = (type_node *)t->suif_type;
    type_node *pt = new_ptr_type(st);
    return simplify_type(pt);
}


