/*  Translate Simple Instructions Back to SUIF Instructions */

/*  Copyright (c) 1994 Stanford University

    All rights reserved.

    This software is provided under the terms described in
    the "suif_copyright.h" include file. */

#include <suif_copyright.h>

#define _MODULE_ "libsimple.a"

#include <suif.h>
#include "globals.h"


/*  The translate_proc function defined here converts a list of Simple SUIF
    instructions to a SUIF procedure.  This basically consists of undoing
    the changes made by the simplify_proc function.  Rather than just blindly
    translating the instructions, the function also checks for several common
    errors so that they will not cause subsequent compiler passes to crash. */

/*  local functions declarations */
static void translate_instr(simple_instr *s);
static operand translate_source(simple_reg *r);
static operand translate_register(simple_reg *r, instruction *srci = NULL);
static label_sym *translate_label(simple_sym *s);
static sym_node *translate_symbol(simple_sym *s);
static type_node *translate_type(simple_type *t);
static immed translate_immed(simple_immed *i);

/*  local variables */
static proc_sym *current_proc;
static op_map *ops;
static tree_node_list *tnl;



/*  The nr_stat list records each node (temporary) register defined in the
    current basic block along with the defining instruction for each.  */

static alist nr_stat;



/*  Translate a list of simple instructions back to a SUIF procedure.
    The results are appended to the procedure's tree_node_list which
    is stored in a global variable so it doesn't have to be passed around
    between all of the translation functions.  */

void
translate_proc (simple_instr *s, proc_sym *p, op_map *o, annote *line_annote)
{
    /* record the parameters */
    current_proc = p;
    ops = o;

    /* record the current output list */
    tnl = p->block()->body();

    /* insert the line number annotation at the beginning */
    if (line_annote) {
	instruction *mrk = new in_rrr(io_mrk);
	mrk->annotes()->push(line_annote);
	tnl->push(new tree_instr(mrk));
    }

    /* traverse the input list */
    while (s) {

	translate_instr(s);

	simple_instr *stmp = s;
	s = s->next;
	free_instr(stmp);
    }

    /* get rid of unnecessary instructions */
    tnl->map(optimize_tree, NULL, FALSE);
}



/*  Translate one simple instruction. */

void
translate_instr (simple_instr *s)
{
    boolean end_block = FALSE;
    simple_reg *dst = NO_REGISTER;
    instruction *i = NULL;

    switch (s->opcode) {

	case LDC_OP: {
	    dst = s->u.ldc.dst;
	    i = new in_ldc(translate_type(s->type),
			   translate_register(s->u.ldc.dst),
			   translate_immed(&s->u.ldc.value));
	    break;
	}

	case BTRUE_OP:
	case BFALSE_OP:
	case JMP_OP: {
	    i = new in_bj(ops->translate_opcode(s->opcode),
			  translate_label(s->u.bj.target),
			  translate_source(s->u.bj.src));
	    end_block = TRUE;
	    break;
	}

	case CALL_OP: {
	    dst = s->u.call.dst;
	    in_cal *cali = new in_cal(translate_type(s->type),
				      translate_register(s->u.call.dst),
				      translate_source(s->u.call.proc),
				      s->u.call.nargs);
	    for (unsigned n = 0; n < cali->num_args(); n++) {
		cali->set_argument(n, translate_source(s->u.call.args[n]));
	    }
	    i = cali;
	    end_block = TRUE;
	    break;
	}

	case MBR_OP: {
	    in_mbr *mbri = new in_mbr(translate_source(s->u.mbr.src),
				      s->u.mbr.offset,
				      s->u.mbr.ntargets,
				      translate_label(s->u.mbr.deflab));
	    for (unsigned n = 0; n < mbri->num_labs(); n++) {
		mbri->set_label(n, translate_label(s->u.mbr.targets[n]));
	    }
	    i = mbri;
	    end_block = TRUE;
	    break;
	}

	case LABEL_OP: {
	    i = new in_lab(translate_label(s->u.label.lab));
	    end_block = TRUE;
	    break;
	}

	/*  The following opcodes are all BASE_FORM instructions.  Instead
	    of just handling them all in the default case, we split them up
	    so that we can force unused register fields to be empty.  */

	case NOP_OP: {
	    i = new in_rrr(ops->translate_opcode(s->opcode));
	    break;
	}

	case RET_OP: {
	    i = new in_rrr(ops->translate_opcode(s->opcode),
			   translate_type(s->type),
			   operand(),
			   translate_source(s->u.base.src1));
	    end_block = TRUE;
	    break;
	}

	case STR_OP: {
	    i = new in_rrr(ops->translate_opcode(s->opcode),
			   translate_type(s->type),
			   operand(),
			   translate_source(s->u.base.src1),
			   translate_source(s->u.base.src2));
	    break;
	}

	case CPY_OP:
	case NEG_OP:
	case NOT_OP:
	case LOAD_OP: {
	    dst = s->u.base.dst;
	    i = new in_rrr(ops->translate_opcode(s->opcode),
			   translate_type(s->type),
			   translate_register(s->u.base.dst),
			   translate_source(s->u.base.src1));
	    break;
	}

	default: {
	    /* handle all of the other BASE_FORM instructions */
	    dst = s->u.base.dst;
	    assert(simple_op_format(s->opcode) == BASE_FORM);
	    i = new in_rrr(ops->translate_opcode(s->opcode),
			   translate_type(s->type),
			   translate_register(s->u.base.dst),
			   translate_source(s->u.base.src1),
			   translate_source(s->u.base.src2));
	}
    }
    tnl->append(new tree_instr(i));


    /*  Node registers can only be used within blocks so empty out the nr_stat
	list at the end of each basic block. */

    if (end_block) {
	while (!nr_stat.is_empty()) {
	    delete nr_stat.pop();
	}
    }


    /*  Check if the destination is a temporary register, and if so, record
	the register number and indicate that it has not yet been used.  */

    if ((dst != NO_REGISTER) && (dst->kind == TEMP_REG)) {

	/* make sure the node register is not redefined */
	if (nr_stat.search((void *)dst->num)) {
	    simple_error("temporary register has multiple definitions");
	}

	/* record that the register is defined but not yet used */
	(void)nr_stat.enter((void *)dst->num, (void *)i);

	/* save the register number as an annotation on the instruction */
	annote *an = new annote(k_simple_reg);
	an->immeds()->push(immed(dst->num));
	i->annotes()->push(an);
    }
}



/*  If the register is a temporary, make sure that it has been defined but
    not yet used.  Translate the register.  */

operand
translate_source (simple_reg *r)
{
    instruction *srci = NULL;

    if ((r != NO_REGISTER) && (r->kind == TEMP_REG)) {

	/* make sure that the register has been defined */
	alist_e *a = nr_stat.search((void *)r->num);
	if (!a) simple_error("temporary register used before defined");

	/* make sure that it hasn't already been used */
	srci = (instruction *)a->info;
	if (!srci) simple_error("temporary register used more than once");

	/* indicate that the register has now been used */
	a->info = NULL;
    }

    return translate_register(r, srci);
}



/*  Translate a simple_reg to SUIF.  If the register is a temporary and the
    "srci" parameter is available, then it is treated as a source operand;
    otherwise, if the "srci" parameter is NULL, it is handled as a destination
    operand.  If the register is a newly allocated machine or pseudo register,
    the sym_node has to be created.  */

operand
translate_register (simple_reg *r, instruction *srci)
{
    if (r == NO_REGISTER) return operand();

    if (!r->var) simple_error("invalid register");

    if (r->kind == TEMP_REG) {
	if (srci) return operand(srci);

	/* otherwise, the register is a destination operand which will
	   be automatically filled it later */
	return operand();
    }

    /* check if we need to create a new variable */
    if (!r->var->suif_sym) {

	if (!r->var->type)
	    simple_error("missing type for register");

	base_symtab *syms;
	annote *an;
	if (r->kind == MACHINE_REG) {
	    syms = fileset->globals();
	    an = NULL;
	} else {
	    syms = current_proc->block()->symtab();
	    an = new annote(k_simple_sym, (void *)r->var);
	}

	/* create a SUIF var_sym */
	type_node *t = translate_type(r->var->type);
	var_sym *v = syms->new_var(t, r->var->name);
	r->var->suif_sym = (void *)v;

	/* get rid of the old name (it's not in the lexicon) */
	delete[] r->var->name;
	r->var->name = v->name();

	/* record the simple_sym and register as annotations on the var_sym */
	if (an) {
	    v->annotes()->push(an);

	    an = new annote(k_simple_reg);
	    an->immeds()->push(immed(r->num));
	    v->annotes()->push(an);
	}
    }
    assert(r->var->suif_sym);

    return operand((var_sym *)r->var->suif_sym);
}



label_sym *
translate_label (simple_sym *s)
{
    label_sym *lab = (label_sym *)s->suif_sym;
    if (!lab->is_label())
	simple_error("symbol is not a label");
    return lab;
}



sym_node *
translate_symbol (simple_sym *s)
{
    return (sym_node *)s->suif_sym;
}



type_node *
translate_type (simple_type *t)
{
    return (type_node *)t->suif_type;
}



immed
translate_immed (simple_immed *i)
{
    switch (i->format) {

	case IMMED_INT: {
	    return immed(i->u.ival);
	}

	case IMMED_FLOAT: {
	    return immed(i->u.fval);
	}

	case IMMED_SYMBOL: {
	    return immed(translate_symbol(i->u.s.symbol), i->u.s.offset);
	}
    }

    simple_error("unexpected immed_type");
    return immed(0);	/* just to keep the compiler happy */
}


