"""
script to generate n_step perturbed photos
"""
from __future__ import division, print_function, absolute_import
from argparse import ArgumentParser
import numpy as np
from copy import deepcopy


def user_inputs():
    """
    the user inputs
    :return string, path to natural photo to perturb
            string, path to perturbation file
            float, the amount of perturbation
            string, path to output file to save perturbed photo
    """
    parser = ArgumentParser()
    parser.add_argument('-x', type=str, required=True,
                        help='Path to natural photo to perturb')
    parser.add_argument('-v', type=str, required=True,
                        help='Path to perturbation file')
    parser.add_argument('-a', type=float, default=128,
                        help='The amoung of perturbation')
    parser.add_argument('-o', type=str, required=True,
                        help='Path to output file to save perturbed photo')
    args = parser.parse_args()

    print('\n\nArguments')
    print('Natural image path: {}'.format(args.x))
    print('Perturbation file path: {}'.format(args.v))
    print('a: {}'.format(args.a))
    print('Output path: {}'.format(args.o))
    print('\n\n')

    return args.x, args.v, args.a, args.o


def load_photo(fname):
    """
    load a photo
    :param fname: string, path to photo in txt file
    :return numpy 1D array of floats in {0, 1, ..., 255}
    """
    photo = np.genfromtxt(fname)
    return photo.astype('float')


def load_perturbation(fname):
    """
    load a perturbation
    :param fname: string, path to perturbation
    :return numpy 1D array of floats
    """
    v = np.genfromtxt(fname)
    return v.astype('float')


def project_photo(x):
    """
    project a continuous photo to a natural photo space
    :param x: numpy 3D array of floats, the continuous photo
    :return numpy 1D array of uint8s, the projected photo
    """
    rgb_max = 255
    t = deepcopy(x)
    t[t > rgb_max] = rgb_max
    t[t < 0] = 0
    t = np.round(t)

    return t.astype('uint8')


def main():
    """the main method"""
    x_path, v_path, a, o_path = user_inputs()

    # load data
    x = load_photo(x_path)
    v = load_perturbation(v_path)

    if not np.array_equal(x.shape, v.shape):
        print('Photo and perturbation dimensions do not match')
        exit()

    # generate perturbation and project
    xp = x + a * v
    xp = project_photo(xp)

    # save
    np.savetxt(o_path, xp, fmt='%d')


if __name__ == '__main__':
    main()
