"""
utility methods to draw plots
"""
import matplotlib.pyplot as plt
import numpy as np


def draw_box_plot(conditions, data_list, xlabel='', ylabel='',
                  xrotation='horizontal', title=None, ylim=None):
    """
    draw box plots
    :param conditions: list(strings), the conditions to put on the x-axis
    :param data_list: list(1D list), the input lists
    :param xlabel: string, the xlabel, default ''
    :param ylabel: string, the ylabel, default ''
    :param xrotation: string or float, the rotation of the xlabels, default
    horizontal
    :param title: string, the title of the plot, default None
    :param ylim: [float, float], the y_lim, default None
    :return matplotlib.pyplot, the plotted bars
    """
    import warnings
    warnings.filterwarnings("ignore")
    index = np.arange(len(conditions)) + 1
    plt.boxplot(data_list)
    plt.xlabel(xlabel)
    plt.ylabel(ylabel)
    plt.xticks(index, conditions)
    plt.xticks(rotation=xrotation)
    plt.grid(linestyle='dashed')

    if title is not None:
        plt.title(title)

    if ylim is not None:
        if type(ylim) == list:
            plt.ylim(ylim)
        else:
            plt.ylim(top=ylim)

    return plt


def draw_box_plot_with_twin_axis(conditions, data_list1, data_list2, xlabel='',
                                 ylabel1='', ylabel2='',
                                 xrotation='horizontal', title=None,
                                 ylim1=None, ylim2=None, yticks2=None):
    """
    draw box plot with another axis data
    :param conditions: list(strings), the conditions to put on the x-axis
    :param data_list1: list(1D list), the input lists for boxplot
    :param data_list2: list(float), the inputs for the second y-axis
    :param xlabel: string, the xlabel, default ''
    :param ylabel1: string, the ylabel for the first y axis, default ''
    :param ylabel2: string, the ylabel for the second y axis, default ''
    :param xrotation: string or float, the rotation of the xlabels, default
    horizontal
    :param title: string, the title of the plot, default None
    :param ylim1: [float, float], the ylim for the first y axis, default None
    :param ylim2: [float, float], the ylim for the second y axis, default None
    :param yticks2: list(obj), the yticks for the second y axis
    :return matplotlib.pyplot, the plotted bars
    """

    import warnings
    warnings.filterwarnings("ignore")
    index = np.arange(len(conditions)) + 1

    fig, ax1 = plt.subplots()
    plt.grid(linestyle='dashed')

    ax1.set_ylabel(ylabel1)
    plt.xlabel(xlabel)
    ax1.boxplot(data_list1)
    plt.xticks(index, conditions)
    plt.xticks(rotation=xrotation)

    ax2 = ax1.twinx()
    color = 'blue'
    ax2.set_ylabel(ylabel2, color=color)
    for i2, count in enumerate(data_list2):
        x = i2 + 1
        ax2.bar(x, count, color='b', align='edge', width=0.2)

    if ylim1 is not None:
        ax1.set_ylim(ylim1)

    if ylim2 is not None:
        ax2.set_ylim(ylim2)

    if yticks2 is not None:
        ax2.set_yticks(yticks2)

    fig.tight_layout()
    return plt
