
#include <time.h>
#include <sys/time.h>
#include <stdlib.h>
#include <stdio.h>

#include "NbodySimulator.h"

NbodySimulator::NbodySimulator(double universe_min_x, double universe_max_x,
                               double universe_min_y, double universe_max_y)
  : m_universe_min_x(universe_min_x), m_universe_max_x(universe_max_x),
    m_universe_min_y(universe_min_y), m_universe_max_y(universe_max_y)
{
  m_bodies = NULL;
}

NbodySimulator::~NbodySimulator()
{
  if (m_bodies != NULL) {
    delete [] m_bodies;
    m_bodies = NULL;
  }
}

void NbodySimulator::populate(int num_bodies)
{
  double x_coord, y_coord;

  m_bodies = new Body[num_bodies];

  for(int i=0; i<num_bodies; i++){
    // a simple placement of bodies is not sufficient, as locations could conflict 
    // because the placement algorithm is not on the critical path, I use a dumb O(n^2) algorithm                                                           
    
    x_coord = (rand() % (int) m_universe_max_x) * (m_universe_max_x / RAND_MAX);
    y_coord = (rand() % (int) m_universe_max_y) * (m_universe_max_y / RAND_MAX);
    for(int j=0;j<i;j++){
      if(m_bodies[j].point_mass.center_of_mass.x == x_coord && m_bodies[j].point_mass.center_of_mass.y == y_coord){
        // must pick new coordinates
        x_coord = (rand() % (int) m_universe_max_x) * (m_universe_max_x / RAND_MAX);
        y_coord = (rand() % (int) m_universe_max_y) * (m_universe_max_y / RAND_MAX);
        j = 0;
      }
    }
    m_bodies[i].point_mass.center_of_mass.x = x_coord;
    m_bodies[i].point_mass.center_of_mass.y = y_coord;                                                                                                        
    
    do{
      m_bodies[i].point_mass.mass = MINIMUM_MASS + (MAXIMUM_MASS - MINIMUM_MASS)/((double) rand()+1.0);
    } while(m_bodies[i].point_mass.mass > MAXIMUM_MASS);                                                                                                      
    
    m_bodies[i].veloc.y = (double) (rand() % MAXIMUM_VELOCITY);
    if(rand() % 1) m_bodies[i].veloc.y = 0 - m_bodies[i].veloc.y;
    m_bodies[i].veloc.x = (double) (rand() % MAXIMUM_VELOCITY);
    if(rand() % 1) m_bodies[i].veloc.x = 0 - m_bodies[i].veloc.x;
  }
  m_num_bodies = num_bodies;
}


/* since the universe is probably too large to fit on the screen,
 *  I shrink it at the expense of accuracy.
 *  It separates the universe into grids and represents each grid with an integer
 *  if there are any bodies in the grid
 */
void NbodySimulator::print( int matrix_size_x, int matrix_size_y )
{
  int i, j, k;

  double x_offset = 0.0, y_offset = 0.0;
  if (m_universe_min_x < 0) {
    m_universe_max_x += -m_universe_min_x;
    x_offset = - m_universe_min_x;
    m_universe_min_x = 0;
  }
  if (m_universe_min_y < 0) {
    m_universe_min_y += -m_universe_min_y;
    y_offset = - m_universe_min_y;
    m_universe_min_y = 0;
  }

  if (m_universe_max_x < matrix_size_x) {
    matrix_size_x = m_universe_max_x+1;
  }
  if (m_universe_max_y < matrix_size_y) {
    matrix_size_y = m_universe_max_y+1;
  }

  double scaling_factor_x = (m_universe_max_x + 1.0) / (double)matrix_size_x;
  double scaling_factor_y = (m_universe_max_y + 1.0) / (double) matrix_size_y;
  
  int* matrix = new int[matrix_size_x * matrix_size_y];
  for(i=0; i<matrix_size_y; i++){
    for(j=0; j<matrix_size_x; j++){
      matrix[i*matrix_size_x + j] = 0;
    }
  }
  for(k=0;k<m_num_bodies;k++){
    int x_coord = (int) ((m_bodies[k].point_mass.center_of_mass.x + x_offset)/scaling_factor_x);
    int y_coord = (int) ((m_bodies[k].point_mass.center_of_mass.y + y_offset)/scaling_factor_y);
    if (x_coord >= matrix_size_x) x_coord = matrix_size_x - 1;  // rounding can lead to overflow...fix here
    if (y_coord >= matrix_size_y) y_coord = matrix_size_y - 1;
    ++matrix[ x_coord + y_coord*matrix_size_x ]; 
  }
  
  for(i=0; i<matrix_size_y; i++){
    for(j=0; j<matrix_size_x; j++){
      if (matrix[i*matrix_size_x + j] == 0) printf("* ");
      else printf("%i ", matrix[i*matrix_size_x + j]);
    }
    printf("\n");
  }
  
  printf("Universe size: %f x %f\n", m_universe_max_x, m_universe_max_y);
  
  delete [] matrix;
}


