/**
	Various utilities using STL-based data structures
**/

#ifndef STL_UTILITIES_H
#define STL_UTILITIES_H

/*
0       10        20        30        40        50        60        70        80
12345678901234567890123456789012345678901234567890123456789012345678901234567890
*/

#include <vector>
#include <list>
#include <map>

#include <algorithm>
#include <iterator>
#include <stdlib.h>
#include <iostream>
using namespace std;

/** Randomize the order of a list of POD (USES COPY CONSTRUCTOR) */
template <typename _T_Iterator> 
void randomize(_T_Iterator begin, _T_Iterator end);

/**
 * Class for randomizing a list. 
 * Contains a random number (on which the 
 *	elements will be sorted)
 */
template <typename _T_Iterator>
class STL_UTIL_KEY { 

  public:
  
  	typedef typename iterator_traits<_T_Iterator>::value_type value_type;
  
  	int key;
	value_type original_value;	// copy of original value
	_T_Iterator itty;
	
	STL_UTIL_KEY(int k, _T_Iterator i, const value_type &t) 
		: key(k), itty(i), original_value(t) { }

	// sort by key
	bool operator< (const STL_UTIL_KEY &LHS) const { return key <  LHS.key; }
	bool operator<=(const STL_UTIL_KEY &LHS) const { return key <= LHS.key; }
	bool operator> (const STL_UTIL_KEY &LHS) const { return key >  LHS.key; }
	bool operator>=(const STL_UTIL_KEY &LHS) const { return key >= LHS.key; }
};


/** 
	Randomize a collection by assigning a random "key" to each element, then
	sorting on that key.  This will copy POD from the collection twice, once
	to store the original value (while other elements from the collection 
	are being overwritten), and once to copy values from another places in 
	the collection.
*/
template <typename _T_Iterator>
void randomize(_T_Iterator begin, _T_Iterator end) { 

	// create a list of 
	//	{ random keys, original data, pointers-to-data locations }
	vector<STL_UTIL_KEY<_T_Iterator> > keys;
	typedef typename vector<STL_UTIL_KEY<_T_Iterator> >::iterator keys_iterator;

	// populate the list
	for (_T_Iterator i = begin; i != end; i++) { 
		// copy POD once
		keys.push_back(STL_UTIL_KEY<_T_Iterator>(rand(), i, *i));	
	}

	// sort by random key
	std::sort(keys.begin(), keys.end());

	// assign original values back to the original collection
	keys_iterator key_itty = keys.begin();
	for (_T_Iterator i = begin; i != end; i++) { 

		*i = key_itty++ -> original_value;	// copy POD again

	}

	return;

} // randomize
	

/** Print a collection of things (for which operator<< is defined) */
template <typename _T_Iterator> 
ostream& stl_print(ostream &out, 
               _T_Iterator begin, _T_Iterator end,
               string delim_front = "<", 
			   string separator = " ",
			   string delim_back = ">") {

	out << delim_front;
	for (_T_Iterator i=begin; i!=end;) { // inc. inside loop
		out << (*i);
		i++;
		if (i != end) { out << separator; }
	}
	out << delim_back;

	return out;
}

/** Customized operator<< for VECTORS */
template <typename T> 
ostream& operator<<(ostream &out, const vector<T> &data) { 
	return stl_print(out, data.begin(), data.end(), "<", " ", ">");
}
/** Customized operator<< for LISTS */
template <typename T>
ostream& operator<<(ostream &out, const list<T> &data) { 
	return stl_print(out, data.begin(), data.end(), "", " -> ", "");
}
/** Customized operator<< for PAIRS */
template <typename F, typename S>
ostream& operator<<(ostream &out, const pair<F,S> &data) { 
	out << "(" << data.first << "," << data.second << ")";
}
/** Customized operator<< for MAPS */
template <typename F, typename S>
ostream& operator<<(ostream &out, const map<F,S> &data) { 
	out << "{";
	for (typename map<F,S>::const_iterator i=data.begin(); i!=data.end();) { 
		out << i->first << ": " << i->second;
		i++;
		if (i != data.end()) { out << ", "; }
	}
	out << "}";
	return out;
}

/** normalize a vector of type N */
template <typename N, typename _Iterator> 
void normalize(_Iterator begin, _Iterator end) { 
	N sum = 0.0;
	for (_Iterator i=begin; i!=end; i++) { sum += *i; }
	for (_Iterator i=begin; i!=end; i++) { *i /= sum; }
	return;
}

/** Normalize a VECTOR */
template <typename T>
void normalize(vector<T> &data) { 
	normalize<T>( data.begin(), data.end() );
	return;
}

/** Normalize a LIST */
template <typename T>
void normalize(list<T> &data) { 
	normalize<T>( data.begin(), data.end() );
	return;
}

#endif
