
#include "gaussian.h"



void Gaussian::generate(double &z0, double &z1) const { 

	// http://en.wikipedia.org/wiki/Box-Muller_transform

	double r = 0.0;
	double x,y;
	while (r==0.0 || r > 1.0) { 
		// generate two uniform randoms [1,1]
		x = ((double)rand()/RAND_MAX) * 2.0 - 1.0;
		y = ((double)rand()/RAND_MAX) * 2.0 - 1.0;
		r = (x*x)+(y*y);
	}
	double W = (r==0.0) ? 0.0 : sqrt( (-2.0*log(r)) / r );	// not to be confused with weight, "w"
	
	// scale by standard distribution and offset by mu
	z0 = (x * W) * sigma + mu;	
	z1 = (y * W) * sigma + mu; 
	return;
}

double Gaussian::generate() const { 
	
	double z0,z1;
	generate(z0,z1);
	return z0;
}


	
/** make Gaussian comparable */
bool Gaussian::operator == (const Gaussian &g) const { 
	return (g.mu==mu && g.sigma==sigma); 
}
bool Gaussian::operator != (const Gaussian &g) const { 
	return (g.mu!=mu || g.sigma!=sigma); 
}

/** for inequality, compare by mu */
bool Gaussian::operator <= (const Gaussian &g) const { return mu <= g.mu; }
bool Gaussian::operator >= (const Gaussian &g) const { return mu >= g.mu; }
bool Gaussian::operator <  (const Gaussian &g) const { return mu <  g.mu; }
bool Gaussian::operator >  (const Gaussian &g) const { return mu >  g.mu; }



/** make Gaussian printable 
 *	this prints "gaussian(x, u, v, w)" where u=mu, v=sigma_check, w=w.
 *	this output can be used by gnuplot if the gaussian function is defined:
 *
 *		gaussian(x,u,v,w) = (w/sqrt(2.0*3.1415926*v)) * exp( (-(x-u) * (x-u))/(2.0*v));
 *
 */
ostream& operator << (ostream& s, const Gaussian& g) { 
	s << g.w << "*gaussian(x," << g.mu << "," << g.sigma << ")";
	return s;
}
 







double GaussianMixture::density(double x) const { 
	double ans = 0.0;
	for (const_iterator i=begin(); i!=end(); i++) { 
		ans += i->density(x);
	}
	return ans;
}

vector<double> GaussianMixture::prob_dist(double x) const { 

	vector<double> ans;
	
	double sum = 0.0;
	
	for (const_iterator i=begin(); i!=end(); i++) { 
		ans.push_back(i->density(x));
		sum += ans.back();
	}
	for (int i=0; i<ans.size(); i++) { 
		ans[i] /= sum;
	}
	return ans;
}


void GaussianMixture::flatten() {

	for (iterator itty=begin(); itty!=end(); itty++) { 
		itty->w = (1.0 / size());
	}
}

void GaussianMixture::normalize() { 

	double sum = 0.0;
	for (iterator itty=begin(); itty!=end(); itty++) { sum += itty->w; }
	for (iterator itty=begin(); itty!=end(); itty++) { itty->w /= sum; }
}


double GaussianMixture::generate() const { 

	// choose a random Gaussian and return generated points

	// first get a total amount of weight in this mixture
	double weight = 0.0;
	for (const_iterator itty=begin(); itty!=end(); itty++) { weight += itty->w; }

	for (const_iterator i=begin(); i!=end(); i++) { 
		double r = (double)rand()/RAND_MAX;
		if ( (r*weight) < i->w ) { 
			// if rnd < "weighted" weight of this mixture member...
			return i->generate();
		} else {
			weight -= i->w;	// less "weight" to consider now...
		}
	}
	// should have picked a mixture member by now...
	cerr << "GaussianMixture::generate():  Error:  No Gaussian Selected." << endl;
	return 0.0;
}

/** make Gaussian Mixture Model comparable */
bool GaussianMixture::operator == (const GaussianMixture &gmm) const { 
	if (gmm.size() != this->size()) { return false; }
	for (int i=0; i<size(); i++) { 
		if ( this->at(i) != gmm[i] ) { return false; }
	}
	return true;
}
bool GaussianMixture::operator != (const GaussianMixture &gmm) const { 
	if (gmm.size() != this->size()) { return true; }
	for (int i=0; i<size(); i++) { 
		if ( this->at(i) != gmm[i] ) { return true; }
	}
	return false;
}



ostream& GaussianMixture::print_sum(ostream &out) const { 

	if (empty()) { 
		out << "0";
	} else {

		out << front();
		for (const_iterator i = begin()+1; i!=end(); i++) { 
			out << " + " << *i;
		}
	}
	return out;
}
ostream& GaussianMixture::print_list(ostream &out) const { 

	if (empty()) { 
		out << "0";
	} else {

		out << front();
		for (const_iterator i = begin()+1; i!=end(); i++) { 
			out << ", " << *i;
		}
	}
	return out;
}

ostream& GaussianMixture::operator<< (ostream &s) const { return this->print_sum(s); }
ostream& operator<< (ostream& s, const GaussianMixture &gm) { 
	if (gm.print_sum_of_gaussians) { return gm.print_sum(s); }
	else { return gm.print_list(s); }
}



