/*
12345678901234567890123456789012345678901234567890123456789012345678901234567890
*/

/**

    OptionParser:  a poorly-written command line parser for use with STL and C++

    This is not meant to be super-flexable.  It's meant to easily handle the
    most common types of command-line parameters in a familiar style.

    Parses popt-style parameters of type string, double, int and bool
	(and that's all!)

	Boolean options are by default false and become true if the option is
	supplied (although there is a version for which the *variable* can be
	made true by default if you don't want to use the negation).

	Also supports required and optional parameters (via an optional boolean
	in the Option class constructor.  Note that the default value of a 
	required option is meaningless, but it's still part of the constructor).

	Options are preceeded by flags ('-'), and there are two "names" for 
	each option, a "long" option name (preceeded by '--') and a "short",
	single character, name (preceeded by '-').
    
    If 'p' is the "short" name, and "parameter" is the "long" name, it 
    handles parameters like:

    a.out -p 42        # parameter value is 42
    a.out -p42   
    a.out -p=42   
    a.out --parameter 42
    a.out --parameter=42

    Reserved options are [--help, -?] and [--usage, -/], which print
    messages and exit (!).

    Errors are:
        1) a missing parameter value, 
            e.g. "a.out -p"    # no value given for -p! 
            (assuming p is not boolean; they don't have values given)
        2) missing a required parameter

    On errors, the program will print an error message and exit (!)

    HOW TO USE IT:

        OptionParser parser(synopsis);	// constructor, give it a short 
                                      	// description of the program
                                      	// for use with the help file

		// add options of type INT, DOUBLE, or STRING (std::string)
        parser.add(Option(longname, shortname, pointer-to-var, default-value, 
                   help-description [,required]));
				  
		// add boolean option (can't be required)
        parser.add(Option(longname, shortname, pointer-to-var, default-value,
                   help-description)); 
		// NOTE:  if the default value of a boolean is TRUE, supplying that
		//	option will make it FALSE.  This need not be visible to the 
		//	user, but make sure you describe the long/short names and 
		//	help message appropriately.

        vector<string> rest_of_args = parser.parse(argc, argv, 
                                                   rest_of_args_description);


        parser.help(cerr);  // print help message (perhaps in case of
                            //    an illegal parameter).  call help() 
                            //    after parse(...)

**/

#ifndef OPTIONPARSER_H
#define OPTIONPARSER_H 0

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <stdlib.h>

#include <vector>
#include <string>
#include <sstream>
#include <iostream>
using namespace std;


class Option;
class OptionParser;

class Option { 

  friend class OptionParser;

  private:

    enum Option_ARG_TYPES {OPTION_INT, OPTION_BOOL, OPTION_STRING,
                            OPTION_DOUBLE, NUM_OPTION_ARG_TYPES};

    string longname;
    char shortname;
    int type;    
    bool required;
    string description;
    
	// pointers to various possible parameter types
    int *int_ptr;
    string *string_ptr;
    double *double_ptr;
    bool *bool_ptr;

	// default values for possible parameter types
    int int_defv;            
    string string_defv;
    double double_defv;
	bool bool_defv;

    bool processed;   // has this parameter been processed? 
                      // (if so, it can be ignored during further parsing)
    
    void set(string value);  // set the parameters value (depends on type)

    /** set current value to default, etc */
    void init();



  public:

    // a constructor for each type.  For example,
	//	Option(long-name, short-name, pointer-to-option-variable,
	//         default-value, help-description, option-is-required=false)

    /** INTEGER */
    Option(string ln, char sn, int *p, int dv, string ds, bool r=false) :
        type(OPTION_INT), int_ptr(p), int_defv(dv),
        longname(ln), shortname(sn), description(ds), required(r) {init();}

    /** DOUBLE */
    Option(string ln, char sn, double *p, double dv, string ds, bool r=false) :
        type(OPTION_DOUBLE), double_ptr(p), double_defv(dv),
        longname(ln), shortname(sn), description(ds), required(r) {init();}

    /** STRING */
    Option(string ln, char sn, string *p, string dv, string ds, bool r=false) :
        type(OPTION_STRING), string_ptr(p), string_defv(dv),
        longname(ln), shortname(sn), description(ds), required(r) {init();}

    /** BOOL (no default version) */
    Option(string ln, char sn, bool *p, string ds) :
        type(OPTION_BOOL), bool_ptr(p), bool_defv(false),
        longname(ln), shortname(sn), description(ds), required(false) {init();}

    /** BOOL (with default version) */
    Option(string ln, char sn, bool *p, bool dv, string ds) :
        type(OPTION_BOOL), bool_ptr(p), bool_defv(dv),
        longname(ln), shortname(sn), description(ds), required(false) {init();}

    /** convert option to how-to-use string, e.g. "[-N|--number]" */
    string tostring() const;
 

};


class OptionParser {

  private:

    vector<Option> arguments;
    bool reserved_help_param;    // arguments[0]
    bool reserved_usage_param;    // arguments[1]

    // a few positions in the list of options are automatic
    static const int RESERVED_HELP = 0;
    static const int RESERVED_USAGE = 1;
    static const int NUM_RESERVED = 2;

    const string synopsis;  // synopsis for help message
    string program;         // program name
    string rest;            // description of what non-option arguments are for

    // util
    string justify(const string&, int, int lpad=0, int rpad=0,
                    bool padfirstline=true);
    string toupper(const string&);

    // print usage messages (abbrev <=> don't list all options)
    void usage(ostream &out, const string &program, 
               const vector<Option> &arguments, 
			   const string &rest, bool abbrev=false);
               
    // print help message
    void help(ostream &out, const string &program, 
                  const vector<Option> &arguments,
                  const string &rest, string synopsis);

    // initialize (with 2 reserved optional parameters)
    void init() { 
        this->add(Option("help", '?', &reserved_help_param,
                  "print this help message and exit"));
        this->add(Option("usage", '/', &reserved_usage_param, 
                  "print usage and exit"));
        program = "<this-program>";
        rest = "<remaining-arguments>";
    }

  public:

    OptionParser() : synopsis("None given.") { init(); }
    OptionParser(string brief_synopsis) : synopsis(brief_synopsis) { init(); }

    /** add a command-line argument to the list */
    void add(const Option &arg) { arguments.push_back(arg); /*copy*/ }
    
    /** set values and return the rest of the arguments */
    vector<string> parse(int argc, const char **argv, 
                         string help_description_for_rest="");

    vector<string> parse(int argc, char **argv, 
                         string help_description_for_rest="");

    void usage(ostream &out=cerr) 
		{ usage(out, program, arguments, rest); }
    void help(ostream &out=cerr) 
		{ help(out, program, arguments, rest, synopsis); }

};

#endif
