#ifndef BRANCH_H
#define BRANCH_H 0

#include <vector>
#include <list>
#include <map>
#include <iostream>
#include <fstream>
#include <stdlib.h>
#include <string>
#include <sstream>
#include <math.h>
#include <assert.h>
using namespace std;

#include "definitions.h"

struct Branch;
struct Tree;


/** A branch is a DISJUNCTION (basically a list of qualifying motifs)
 	
 *	also bookeeping data over sequence data
 *		hits:  a list of (sequence-id, list of (position,strand))

 *	also a strand constraint (1=TEMPLATE, 2=TRANSCRIBED, 3=EITHER), which
 *	matches the bookeeping info.
 
 *	Many trees may point to the same branch, so I'll keep a count (pins) of the
 *	number of objects pointing to each branch
 
 */




struct Branch {
  friend struct Tree;
  friend ostream& operator<<(ostream &out, const Branch&);
  friend ostream& operator<<(ostream &out, const Tree&);
  private:
  	int pins;
	vector<list<pair<int,int> > > hits;	// hits[sequence] -> list of (position, strand) where motif(s) occur.
	
  public:
  	list<int> motifs;
	int strand;
  
  	// create a null conjunct
	Branch() {
		#if DEBUG_MALLOC_CHECK
		branch_alloc++; 
		#endif
		pins = 1;
		motifs.clear();
		strand = EITHER_STRAND;
	}

	/** Create a (new) copy of the branch */
	explicit Branch(const Branch &copy) { 
		#if DEBUG_MALLOC_CHECK
		branch_alloc++; 
		#endif
		pins = 1;
		motifs = copy.motifs;	// copy list
		strand = copy.strand;
		hits = copy.hits;		// copy all
	}

	#if DEBUG_MALLOC_CHECK 
	~Branch() { branch_free++; }
	#endif

	bool operator==(const Branch &b) const { 
		return (b.motifs == this->motifs && b.strand == this->strand); // recall motifs are in sorted order
	}
	bool operator!=(const Branch &b) const { return !((*this)==b); }

	/** File I/O */
	void Branch::write(ostream &out) const;
	void Branch::read(istream &in);

	/** Add a new motif to the disjunction and update bookkeeping data 
	 *  Return false iff motif is already in list
	 */
	bool add(int motif, int S, const SMMap &smmap);

	/** set this->strand and update bookeeping data (remove hits that do not match) */
	void constrain_strand(int strand);

	/** Reset bookkeeping data based on (possibly new) sequence data */
	void reset(int S, const SMMap &smmap);

	
	/** Remove one of the motifs and reset data */
	void remove(int index, int S, const SMMap &smmap);
	
};	// Branch


#endif
