/**

	Option:  a poorly-written command-line option parser and usage/help message
	generator.  
	
	You supply a pointer to an option variable, and this an OptionParser object
	will parse the command line (argv) for instances of "--optname" or "-f",
	where `optname' is a string associated with the option (the option's "name"
	hereafter) and `f' is a single-character unique to the option that we'll
	call the option's "flag."
	
	OptionParser uses template classes.  Hereafter, let T be the type of an
	option variable.
	
	To use this class, T must be able to READ FROM std::istream, that is, 
		
		istream& operator>>(istream&, T&) 

	must be defined.  Also, T must be able to WRITE TO std::ostream to use the
	OptionParser::dump(ostream&) method (which dumps the value of option
	variables).

	OptionParser uses partial specialization for the special case of a boolean
	option variable, which doesn't need to have a value supplied (just
	mentioning the option's name or flag set the boolean variable to the
	negation of its default).  

	Illustration:

		OptionParser.parser("program synopsis");

		parser.add("optname", 'f', &var, default_value, "help-context-string");

			# where `default_value' is a value that can be converted to a T.

		parser.add("optname", 'f', &var, "help-context string");	

			# This version of `add' does not have a default value, so it is a
			# "Required" option.

			# Note:  If the given flag is a const char called NULL_FLAG, it
			# will not be printed by help/usage messages as having a flag at
			# all (say you want this option to only be indicated by a
			# --optname).  (If NULL_FLAG is really given--it's '\0'--it will
			# still work.)

		parser.parse(argc, argv, "description of command-line arguments");

		parser.help(ostream);	# print help
		parser.usage(ostream);	# print usage string


	OptionParser will parse the command line arguments as:

		--optname=VALUE
		--optname VALUE
		-f=VALUE
		-fVALUE		(but not `--optnameVALUE'; that's interpreted as one long `--optnamevalue')
		-f VALUE
	
	A special case is BOOLEAN variables, which do not require a value
	(simply mentioning them toggles their default value):

		--optname
		-f
		-fghij		(where `g', `h', etc. are other boolean option flags)

	That is, multiple boolean variables can be set with their flag characters
	bunched together.  Note that I'm saying "toggled" or "set"--if a boolean's
	default is TRUE, it will be set to FALSE if it's name or flag is mentioned.
	(If a boolean has no default, it will be a "required" option and so
	necessarily the opposite of its default which doesn't seem too useful.)

	OptionParser will handle error cases by throwing an instance of type
	"OptionException" (derived from std::exception).  It will also throw
	instances of type "OptionWarning" (derived from OptionException).

		Errors:
			Missing value, e.g. `a.out -L' where -L is non-boolean and requires a value
			Unrecognized option, e.g. `a.out -x' where `x' is not an option added to the parser
				Note:  anything starting with a '-' is an option, unless it looks like a negative number
			Missing required option, e.g. `a.out' where --optname|-f was added with no default value
			Multiple values for an option, e.g. `a.out -f7 -f=7'
		Warnings:
			Multiple options with identical names or flags, e.g.
				parser.add("--optname1, 'j', &var1, def1, "help");
				parser.add("--optname2, 'j', &var2, def2, "help");

	OptionParser automatically adds two options, --help and --usage.  These are
	booleans.  If either (or both) are given, OptionParser will print the
	corresponding usage/help messages and EXIT THE PROGRAM.

	NOTE:  Technically, all data are assigned to their default values (if
	given) via `T::operator=(const default_type&)' and then re-assigned to
	given values via `Option<T>::read(istream&)' if read from the command line.
	Any side effects of the assignment operator apply.

	NOTE:  Special consideration is taken for char* types (they are treated
	somewhat like strings), although using std::string is highly recommended.
	Remember that char* is a POINTER, so the default would be to read and
	assign pointers.  Since it doesn't make sense to read a pointer from the
	command line, the Option<char*>::read(istream&) function will read
	character data from the stream into the pointer currently pointed to, which
	is the default POINTER (if given).  Hence, this should be a memory location
	and not a const char * created by a quoted string literal.  If no default
	is given, Option<char*>::read(istream&) will still read into the location
	pointed to by the char* variable, so it should be given a memory location
	before the parse(...) function is called. 
		
**/

#ifndef OPTION_PARSER_H
#define OPTION_PARSER_H

#include <vector>
#include <string>
#include <sstream>
#include <iostream>
#include <exception>
using namespace std;

#ifndef uint
typedef unsigned int uint;
#endif

const char NULL_FLAG = '\0';

class OptionException;
class OptionWarning;

class UntypedOption;
template <typename T> class Option;
class OptionParser;

class UntypedOption {

  friend class OptionParser;

  protected:

  	string name;
	char flag;
	bool required;
	string help;
  
  	virtual ~UntypedOption() { }

	/** read data into an option variable */
	virtual void read(istream&) = 0;

	/** return whether the option's flag requires a value (probably true for all but Option<bool> */
	virtual bool requires_value() const = 0;

	/** if a value is not required, this function is called instead of `read' */
	virtual void set() = 0;

  public:


	/** print a description of the variable's input type (e.g. "Integer" or "String") */
	virtual void type(ostream&) const = 0;


	/** print the variable's current value (e.g. 7 or "Nantuckett") */
	virtual void value(ostream&) const = 0;

	/** print the variable's default value (e.g. 7 or "Nantuckett") */
	virtual void default_value(ostream&) const = 0;

	/** print option usage (e.g. ``[-n|--name=String]'' or ``[--verbose]'') */
	void usage(ostream &out) const { 
		if (!required) { out << "["; }
		if (flag != NULL_FLAG) { out << "-" << flag << "|"; }
		out << "--" << name;
		if (requires_value()) { 
			out << "=";
			type(out);
		}
		if (!required) { out << "]"; }
		return;
	}

	/** string versions */
	string type() { ostringstream oss; this->type(oss); return oss.str(); }
	string usage() const { ostringstream oss; this->usage(oss); return oss.str(); }
	string value() const { ostringstream oss; this->value(oss); return oss.str(); }
	string default_value() const { ostringstream oss; this->default_value(oss); return oss.str(); }
	
};

template <typename T>
class Option : public UntypedOption {

  private:

  	T *ptr;	
	T def;

	// IMPORTANT:  On some compilers, I can't override Option<T>::method(...)
	// with Option<specific-type>::method(...) if I define the method here.
	// So, I DON'T do `void set() { }', I do `void set();' and define below.
	// Not sure why this is the case.
  	
	void read(istream&);
	bool requires_value() const;
	void set();	

  public:

	template <typename F>
	Option<T>(string name, char flag, T *pointer, const F &default_value,
				const string &help, bool required) {
				
		this->name = name;
		this->flag = flag;
		this->ptr = pointer;
		this->def = (T)(default_value);	// copy constructor
		this->help = help;
		this->required = required;

		(*ptr) = def;		// initialize

	}

	~Option<T>() { }

	void type(ostream&) const;
	void value(ostream&) const;
	void default_value(ostream&) const;



};

class OptionParser {

  private:

    vector<UntypedOption*> options;

    // a few positions in the list of options are automatic
    static const int RESERVED_HELP = 0;
    static const int RESERVED_USAGE = 1;
    static const int NUM_RESERVED = 2;

	bool reserved_help_param;
	bool reserved_usage_param;

    const string synopsis;  // synopsis for help message
    string program;         // program name
    string rest;            // description of what non-option options are for

	bool duplicate(); // check last-added option for duplicate option names/flags (throw exception)

  public:

    OptionParser(string synopsis="");
	~OptionParser();

	template <typename T, typename F>
	void add(const string &longkey, char shortkey, T *pointer, 
			 const F &default_value, const string &help_context) {

		// default given <=> optional option
		options.push_back(new Option<T>(longkey, shortkey, pointer,
		                                default_value, help_context, false));
		duplicate();	// check, throw exception if duplicated option

	}

	template <typename T>
	void add(const string &longkey, char shortkey, T *pointer, 
			 const string &help_context) {
	
		// no default <=> required option
		options.push_back(new Option<T>(longkey, shortkey, pointer,
		                                *pointer, help_context, true));
		duplicate();	// check, throw exception if duplicated option
	}

   	/** Parse command line and set variables.  Return a list of the non-option arguments */
    vector<string> parse(int argc, const char **argv, string rest="");
    vector<string> parse(int argc, char **argv, string rest="");

	/** Print usage */
    void usage(ostream &out=cerr) const;

	/** Print help (synopsis, usage, option descriptions) */
    void help(ostream &out=cerr) const; 

	/** Dump all options and their values (terse== use flags or names) */
	void dump(ostream &out=cerr, string separator=":  ", string delim="\n", bool terse=false) const;

};




// NOTE:  Partial specialization of Option member functions are implemented in .cpp file
template <typename T>
void Option<T>::type(ostream &out) const { out << "Value"; }

template <typename T>
void Option<T>::value(ostream &out) const { out << (*ptr); }

template <typename T>
void Option<T>::default_value(ostream &out) const { out << def; }

template <typename T>
bool Option<T>::requires_value() const { return true; }

template <typename T>
void Option<T>::set() {  }	// only non-requires_value options need this.

template <typename T>
void Option<T>::read(istream &in) { in >> (*ptr);  }



/** Thrown when parse fails */
class OptionException : public std::exception {
  public:
  	std::string msg;
	OptionException(const std::string &m) : exception(), msg(m) { }
	virtual ~OptionException() throw() { }
	virtual const char* what() const throw() { return this->msg.c_str(); }
};

/** Thrown when a non-fatal error occurs */
class OptionWarning : public OptionException { public: OptionWarning(const string &m) : OptionException(m) { } };


#endif
