#include <iterator>
using namespace std;

/**
 
 *	A Tree is a list of Branches and constraints between them.

 *	Trees have the ability to score over a data set

 */

#include "Tree.h"

int Tree::compare(const Tree &t) const {

	if (this->branches.size() < t.branches.size()) { return -1; }	// ME < IT:  I have fewer branches
	if (this->branches.size() > t.branches.size()) { return  1; }
	
	// number of branches the same
	
	if (this->tss < t.tss) { return -1; }
	if (this->tss > t.tss) { return  1; }
	if (this->negup < t.negup) { return -1; }
	if (this->negup > t.negup) { return  1; }
	if (this->negdown < t.negdown) { return -1; }
	if (this->negdown > t.negdown) { return  1; }
	
	// global constraints the same
	
	vector<int> me2it(this->size());	// mapping from my branch index to t's branch index
	
	for (int b=0; b<branches.size(); b++) { 

		// find t's corresponding branch.
		for (int j=0; j<branches.size(); j++) { 
			
		}

		if (this->strand(b) < t.strand(b)) { return -1; }
		if (this->strand(b) > t.strand(b)) { return  1; }

		list<int>::const_iterator me=branches[b]->motifs.begin();
		list<int>::const_iterator it=t.branches[b]->motifs.begin();

		while (true) {
			if (me == branches[b]->motifs.end()) { return -1; }		// me < it, me has fewer motifs
			if (it == t.branches[b]->motifs.end()) { return  1; }	// me > it, it has fewer motifs

			if (*me < *it) { return -1; }
			if (*it < *me) { return  1; }
			me++; 
			it++;
		}

	} // next branch

	// structure the same, check interbranch constraints
	for (int b1=0; b1<branches.size(); b1++) { 
	for (int b2=b1+1; b2<branches.size(); b2++) { 
		
		if (this->constraints[b1][b2].oc < t.constraints[b1][b2].oc) { return -1; }
		if (this->constraints[b1][b2].oc > t.constraints[b1][b2].oc) { return  1; }
		
		if (this->constraints[b1][b2].dc < t.constraints[b1][b2].dc) { return -1; }
		if (this->constraints[b1][b2].dc > t.constraints[b1][b2].dc) { return  1; }

		list<int>::const_iterator n1 = this->constraints[b1][b2].negated.begin();
		list<int>::const_iterator n2 = t    .constraints[b1][b2].negated.begin();
		
		if (   n1 == this->constraints[b1][b2].negated.end()
			&& n2 != t.constraints[b1][b2].negated.end()) { return -1; }	// this has shorter list
		if ( n2 ==    t.constraints[b1][b2].negated.end()) { return  1; }	// this has longer list

		if (*n1 < *n2) { return -1; }
		if (*n1 > *n2) { return  1; }
		
	}}
		
	return 0;	// equal!
} // compare


/** remove pin, last one to leave takes out the garbage */
void Tree::delete_branch(Branch *d) { 
	d->pins--;
	if (!d->pins) { delete d; }
}

/** Add a new conjunct (branch) to the tree (will go in the last position) */
void Tree::add_conjunct(int motif, int S, const SMMap &smmap) {

	// NOTE:  if I were INSERTING a branch (as opposed to adding to the end of the list),
	//	I would need to INSERT into the constraints structures

	for (int b=0; b<branches.size(); b++) { 
		constraints[b].push_back(Constraint());
	}

	branches.push_back(new Branch());
	branches.back()->add(motif, S, smmap);
	
	constraints.push_back(vector<Constraint>(branches.size()));
	/* no longer necessary-> default constructor used default params
	for (int b=0; b<branches.size(); b++) { 
		constraints[branches.size()-1][b].first = EITHER_ORDER;
		constraints[branches.size()-1][b].second = INFINITY;
	}
	*/

}

/** Add a new disjunct (motif to a branch) to the given branch */
bool Tree::add_disjunct(int position, int motif, int S, const SMMap &smmap) { 

	vector<Branch*>::iterator itr = branches.begin() + position;
	
	// check for motif already there
	for (list<int>::const_iterator d=(*itr)->motifs.begin(); d!=(*itr)->motifs.end(); d++) { 
		if (*d == motif) { return false; }
	}

	Branch *new_branch = new Branch(**itr);	// copy branch (new pointer)
	new_branch->add(motif, S, smmap);		// add given motif
	delete_branch(*itr);					// delete old pointer
	*itr = new_branch;						// insert new branch
	return true;	
}

/** Constrain a given branch to the given strand */
void Tree::constrain_strand(int position, int strand) { 
	
	vector<Branch*>::iterator itr = branches.begin() + position;
	
	Branch *new_branch = new Branch(**itr);	// copy branch (new pointer)
	new_branch->constrain_strand(strand);	// constrain the strand now
	delete_branch(*itr);					// delete old pointer
	*itr = new_branch;						// insert new branch
	this->squal.clear();	// 2004-12-30
}

/** reset internal bookkeeping data based on (possibly new) sequence data */
void Tree::reset(int S, const SMMap &smmap) {
	for (vector<Branch*>::iterator b=branches.begin(); b!=branches.end(); b++) { 
		(*b)->reset(S, smmap);
	}
	this->squal.clear();	// 2004-12-30
}



/** constrain the order by supplying branch indicies (starting at 0),
 * branch 'upstream' must be upstream of branch 'downstream'.  return false
 * if model is already constrained */
void Tree::constrain_order(int upstream, int downstream) { 

	constraints[upstream][downstream].oc = UPSTREAM_OF;	// upstream UPSTREAM_OF downstream
	constraints[downstream][upstream].oc = DOWNSTREAM_OF;
	return;
}

void Tree::constrain_distance(int branch1, int branch2, double distance) { 
	constraints[branch1][branch2].dc = constraints[branch2][branch1].dc = distance;
}

void Tree::constrain_distance(double distance) { tss = distance; }



bool Tree::negate(int branch1, int branch2, int motif) { 
	
	for(list<int>::iterator i=constraints[branch1][branch2].negated.begin(); ; ) {
		if (i == constraints[branch1][branch2].negated.end() || motif > *i) { 
			i = constraints[branch1][branch2].negated.insert(i, motif);
			return true;
		} else if (motif == *i) { 
			return false;
		} else {
			i++;
		}
	}

	for(list<int>::iterator i=constraints[branch2][branch1].negated.begin(); ; i++) {

		if (i == constraints[branch2][branch1].negated.end() || motif > *i) { 
			i = constraints[branch2][branch1].negated.insert(i, motif);
			return true;
		}
	}


}

bool Tree::negate(bool upstream, int motif) { 
	
	if (upstream) {
		for (list<int>::iterator i=negup.begin(); i!=negup.end(); i++) { 
			if (motif == *i) { return false; }
			if (motif > *i) { i = negup.insert(i, motif);  return true; }
		}
		negup.push_back(motif);
		return true;
	} else {
		for (list<int>::iterator i=negdown.begin(); i!=negdown.end(); i++) { 
			if (motif == *i) { return false; }
			if (motif > *i) { i = negdown.insert(i, motif);  return true; }
		}
		negdown.push_back(motif);
		return true;
	}
}








/** remove a branch and update data structures */
void Tree::remove(int index) { 

	for (int b=0; b<branches.size(); b++) { 
		constraints[b].erase(constraints[b].begin() + index);
	}
	constraints.erase(constraints.begin() + index);
	
	delete_branch(branches[index]);
	
	branches.erase(branches.begin() + index);
}


/** remove a motif and update data structures */
void Tree::remove(int branch_index, int motif_index, int S, const SMMap &smmap) { 

	Branch *original_branch_ptr = branches[branch_index];
	branches[branch_index] = new Branch(*(branches[branch_index]));
	delete_branch(original_branch_ptr);
	branches[branch_index]->remove(motif_index, S, smmap);
}





/** create a new Results structure for the model. 
 *	hint:  model is being constrained/relaxed.  The idea is if we're constraining a 
 *			model, a sequence that didn't qualify before still won't and there's no need
 *			to evaluate the sequence.
 *  hint = 1:  model is being strictly constrained compared to current 'squal' structure
 *  hint = 2:  model is being strictly relaxed compared to current 'squal' structure
 *  hint = 0:  unknown, so evaluate all sequences
 *
 *	04/15/2005:  bug fix:  was incrementing iterators INSIDE pairwise-constraint-check.
 */	
Results* Tree::evaluate(int S, int P, const SMMap &smmap, const vector<double> &sw,
                        int hint, vector<list<vector<int> > > *passlist) {

	if (squal.size() != S) { 
		hint = 0; // must re-evaluate everything
		squal.resize(S);
		std::fill(squal.begin(), squal.end(), true);
	}

	double tp=0.0,fp=0.0,tn=0.0,fn=0.0;

	for (int s=0; s<S; s++) {

		if (passlist) { (*passlist)[s].clear(); }

		bool Q;				// sequence qualifies according to model?
		bool T = (s<P);		// sequence represents a promoter (positive example)?

		if (hint == 1 && !squal[s]) { 
			// constraining a model that didn't qualify before
			Q = false;
		} else if (hint == 2 && squal[s]) { 
			// relaxing a model that already qualified
			Q = true;
		} else {
			// we have to actually look at the sequences
			bool have_req_motifs = true; // tentative assignment

			// make sure all branches have *some* motifs (and close enough to TSS)
			for (int b=0; b<branches.size(); b++) { 
				if (   branches[b]->hits.size() <= s	// no such sequence
					|| branches[b]->hits[s].empty())	// no motif instances
				{ 
					have_req_motifs = false; break;
				}

			}

			Q = false;	// tentative assignment
			
			// check constraints (we know all branches have hits in this sequences)
			if (have_req_motifs) {

				// As far as we know, this sequence qualifies...

				// create a list of positions for each branch
				//	
				// for each branch (let's say 3):
				// instances[0] = { -100                         -32                -5 }
				// instances[1] = {       -85               -45       -28              }
				// instances[2] = {               -60  -50                   -12       }
				//                     ^    ^       ^
				//                     |    |       |
				// iterators[0] -------     |       |   (Each iterator pointing to the first instance)
				// iterators[1] ------------        |
				// iterators[2] --------------------

				
				vector<list<int> > instances(branches.size());
				vector<list<int>::iterator> iterators(branches.size());
				
				for (int b=0; b<branches.size(); b++) {
					for (list<pair<int,int> >::const_iterator i=branches[b]->hits[s].begin(); 
															  i!=branches[b]->hits[s].end(); i++) {
						instances[b].push_back(i->first);	// i->first == position
					}
					iterators[b] = instances[b].begin();
				}

				// Now if these instances pass the tests (constraints), we're done.
				//	otherwise, move to a new combination with iterators[2]++
				//	and so on...

				for (bool done=false; !done; ) {

					// not done <=> still haven't found a combo that works, or gone through all of them.

					// right now, we're considering each branch B (0..branches.size()-1)
					//	to be represented by the motif at instances[BRANCH][iterators[BRANCH]];
					
					bool pass = true;	// tentative

					// this combination doesn't pass if any instances are equal
					for (int b1=0; b1<branches.size(); b1++) { 
					for (int b2=b1+1; b2<branches.size(); b2++) { 
						if (*(iterators[b1]) == *(iterators[b2])) {
							pass = false; break; 
						}
					} if (!pass) { break; } }

					// MAKE SURE THIS SET OF INSTANCES PASSES GLOBAL CONSTRAINTS
					//	(only need to go through this or any more tests if ``pass'' still equals true)
					if (pass) {

						int up_pos = *(iterators[0]);
						int down_pos = *(iterators[0]);
						
						for (int b=1; b<branches.size(); b++) { 
							if (*(iterators[b]) > down_pos) { down_pos = *(iterators[b]); }
							if (*(iterators[b]) < up_pos) { up_pos = *(iterators[b]); }
						}
					
						if (-down_pos > tss) { 
							
							pass = false; 	// CRM too far away

						} else {

							// check for negated regions upstream or downstream of CRM
							for (list<int>::const_iterator nu=negup.begin(); nu!=negup.end(); nu++) { 
								for (list<pair<int,int> >::const_iterator np=smmap[s][*nu].begin();
																		  np!=smmap[s][*nu].end(); np++) { 
									if (np->first < up_pos) { pass = false; break; }	// neg. upstream
								}
								if (!pass) { break; }
							}
							if (pass) {
							for (list<int>::const_iterator nd=negdown.begin(); nd!=negdown.end(); nd++) { 
								for (list<pair<int,int> >::const_iterator np=smmap[s][*nd].begin();
																		  np!=smmap[s][*nd].end(); np++) { 
									if (np->first > down_pos) { pass = false; break; }	// neg. downstream
								}
								if (!pass) { break; }
							}
							}
						}
					}


					// MAKE SURE THIS SET OF INSTANCES PASSES PAIRWISE CONSTRAINTS
					if (pass) {

						for (int b1=0; b1<branches.size(); b1++) {
						for (int b2=b1+1; b2<branches.size(); b2++) {

							int i1 = *(iterators[b1]);
							int i2 = *(iterators[b2]);	// shorthand

							if (    constraints[b1][b2].oc == EITHER_ORDER
								 && constraints[b1][b2].dc == INFINITY
								 && constraints[b1][b2].negated.empty() ) {

								continue;	// common case, unconstrained pair
							}

							// check order and distance
							if (       constraints[b1][b2].oc == UPSTREAM_OF && i2 <= i1) {
								// b1 should be UPSTREAM, but it's downstream
								pass = false;
							}
							
							if (pass && constraints[b1][b2].oc == DOWNSTREAM_OF && i1 <= i2) {
								 pass = false;
							}

							if (pass && (abs(i1 - i2) > constraints[b1][b2].dc)) {
								 pass = false;
							}

							// check negated regions
							if (pass) {
								for (list<int>::const_iterator n=constraints[b1][b2].negated.begin();
															   n!=constraints[b1][b2].negated.end(); n++) { 

									for (list<pair<int,int> >::const_iterator np=smmap[s][*n].begin(); 
																			  np!=smmap[s][*n].end(); np++) { 

										if (i1 < np->first && np->first < i2) { pass=false; break; }
										if (i2 < np->first && np->first < i1) { pass=false; break; }
									}
									if (!pass) { break; }
								}
							}

							// IF we found a pair of branches that DO NOT PASS, this 
							//	combination of motif instances DO NOT PASS, and we can
							//	stop checking them.
							if (!pass) { break; }

						} if (!pass) { break; } } // next pair of branches

					} // if (pass)

					// if pass still equals true, this combination passes all constraints
					//	so, we can exit the loop.
					if (pass) {
						Q = true;
						if (passlist) { 
							(*passlist)[s].push_back(vector<int>(this->branches.size()));
							for (uint b=0; b<this->branches.size(); b++) { 
								(*passlist)[s].back()[b] = *(iterators[b]);
							}
						} else {
							done = true;
						}
					}


					// UPDATE ITERATORS FOR NEXT LOOP
					for (int b=branches.size()-1; b>=0; b--) { 
						iterators[b]++;
						if (iterators[b] != instances[b].end()) { break; }
						if (b==0) { 
							// first iterator at the end <=> we've tried all combinations 
							done = true;  
							break; 
						}
						iterators[b] = instances[b].begin();
					}

				} // while !done

			} // if (have_req_motifs)

		} // if using hints.


		if (Q && passlist) { 
			assert(passlist->size() > s);
			assert((*passlist)[s].size() > 0);
			assert((*passlist)[s].begin()->size() == this->branches.size());
			assert((*passlist)[s].begin()->size() > 0);
		}

		squal[s] = Q;

		// increment appropriate count
	
			 if ( Q &&  T) { tp += sw[s]; }
		else if ( Q && !T) { fp += sw[s]; }
		else if (!Q &&  T) { fn += sw[s]; }
		else               { tn += sw[s]; }

	} // next sequence

	return new Results(tp,fp,tn,fn);

} // evaluate



/** 
 * Copy contents of Tree::squal to the given location.  Valid after a call to Tree::evaluate
 */
void Tree::copy_squal(vector<bool> *copy) { 

	copy->resize(squal.size());
	std::copy(squal.begin(), squal.end(), copy->begin());
	return;

}






/**
 *	Among sequences that QUALIFY, get a list of RELEVANT DISTANCES between 
 *	EACH PAIR OF BRANCHES
 *
 *	That is, return a list of in-between branch distances such that the 
 *	sequence qualifies given the tree, but it WOULDN'T if the distance were
 *	smaller than that.
 *
 *	return value:  pairwise[smaller-branch-index][larger-branch-index] -> vector of (distances, sequence ID)
 *				   global -> vector of (distances, sequence ID)
 *
 *	NOTE:  based on the ``evaluate'' function above.  Since I'm only interested
 *	in the examples that QUALIFY.
 *
 *
 */
void Tree::get_distances(int S, const SMMap &smmap, 
	vector<vector<vector<pair<int, int> > > > &pairwise, vector<pair<int, int> > &global) {
	
	pairwise.resize(branches.size());
	for (int b1=0; b1<branches.size(); b1++) {
		pairwise[b1].resize(branches.size());
		for (int b2=0; b2<branches.size(); b2++) { 
			pairwise[b1][b2].clear();
		}
	}

	if (squal.size() != S) {
		squal.resize(S);
		std::fill(squal.begin(), squal.end(), true);
	}

	for (int s=0; s<S; s++) {

		if (squal[s]) {

			// if/when this sequence ends up passing the tests, keep the
			//	MINIMUM distances you see for each 
			int min_used_global = -1;
			vector<vector<int> > min_used_pairwise(branches.size());
			for (int b=0; b<branches.size(); b++) { 
				min_used_pairwise[b].resize(branches.size()); 
				for (int j=0; j<branches.size(); j++) { 
					min_used_pairwise[b][j] = -1;	// let -1 mean unset as of yet
				}
			}

			// NOTE:  we *should* KNOW that this sequence passes
			// (because squal[s]), but we don't know WHICH motif instances
			// lead to a pass, only that ONE OR MORE do.  plus, ``squal''
			// might not have been initialized.

			// make sure all branches have *some* motifs 
			for (int b=0; b<branches.size(); b++) { 
				if (   branches[b]->hits.size() <= s	// no such sequence
					|| branches[b]->hits[s].empty())	// no motif instances
				{ 
					squal[s] = false;
				}
			}

			// check constraints (we know we have at least one instance per branch)
			if (squal[s]) {

				// As far as we know, this sequence qualifies...

				// create a list of positions for each branch
				//	
				// for each branch (let's say 3):
				// instances[0] = { -100                         -32                -5 }
				// instances[1] = {       -85               -45       -28              }
				// instances[2] = {               -60  -50                   -12       }
				//
				//                     ^    ^       ^
				//                     |    |       |
				// iterators[0] -------     |       |   (Each iterator pointing to the first instance)
				// iterators[1] ------------        |
				// iterators[2] --------------------

				
				vector<list<int> > instances(branches.size());
				vector<list<int>::iterator> iterators(branches.size());
				
				for (int b=0; b<branches.size(); b++) {
					for (list<pair<int,int> >::const_iterator i=branches[b]->hits[s].begin(); 
															  i!=branches[b]->hits[s].end(); i++) {
						instances[b].push_back(i->first);	// i->first == position
					}
					iterators[b] = instances[b].begin();
				}

				// Now if these instances pass the tests (constraints), we're done.
				//	otherwise, move to a new combination with iterators[2]++
				//	and so on...

				for (bool done=false; !done; ) {

					// not done <=> still haven't found a combo that works, or gone through all of them.

					// right now, we're considering each branch B (0..branches.size()-1)
					//	to be represented by the motif at instances[BRANCH][iterators[BRANCH]];
					
					bool pass = true;	// tentative

					// create an array for *tentative* inter-branch distances
					//	(which only end up mattering if the combination passes)
					vector<vector<int> > tentative_pairwise(branches.size());
					for (int i=0; i<branches.size(); i++) 
						{ tentative_pairwise[i].resize(branches.size()); }
					int tentative_global; 

					// this combination doesn't pass if any instances are equal
					for (int b1=0; b1<branches.size(); b1++) { 
					for (int b2=b1+1; b2<branches.size(); b2++) { 
						if (*(iterators[b1]) == *(iterators[b2])) {
							pass = false; break; 
						}
					} if (!pass) { break; } }

					// MAKE SURE THIS SET OF INSTANCES PASSES GLOBAL CONSTRAINTS
					//	(only need to go through this or any more tests if ``pass'' still equals true)
					if (pass) {

						int up_pos = *(iterators[0]);
						int down_pos = *(iterators[0]);
						
						for (int b=1; b<branches.size(); b++) { 
							if (*(iterators[b]) > down_pos) { down_pos = *(iterators[b]); }
							if (*(iterators[b]) < up_pos) { up_pos = *(iterators[b]); }
						}
						
						if (-down_pos > tss) { 
						
							pass = false; 	// CRM too far away

						} else {

							// passed the global distance constraint
							tentative_global = -down_pos;

							// check for negated regions upstream or downstream of CRM
							for (list<int>::const_iterator nu=negup.begin(); nu!=negup.end(); nu++) { 
								for (list<pair<int,int> >::const_iterator np=smmap[s][*nu].begin();
																		  np!=smmap[s][*nu].end(); np++) { 
									if (np->first < up_pos) { pass = false; break; }	// neg. upstream
								}
								if (!pass) { break; }
							}
							if (pass) {
							for (list<int>::const_iterator nd=negdown.begin(); nd!=negdown.end(); nd++) { 
								for (list<pair<int,int> >::const_iterator np=smmap[s][*nd].begin();
																		  np!=smmap[s][*nd].end(); np++) { 
									if (np->first > down_pos) { pass = false; break; }	// neg. downstream
								}
								if (!pass) { break; }
							}
							}
						}
					}

					// MAKE SURE THIS SET OF INSTANCES PASSES PAIRWISE CONSTRAINTS
					if (pass) {

						for (int b1=0; b1<branches.size(); b1++) {
						for (int b2=b1+1; b2<branches.size(); b2++) {

							int i1 = *(iterators[b1]);
							int i2 = *(iterators[b2]);	// shorthand


							// check order and distance
							if (       constraints[b1][b2].oc == UPSTREAM_OF && i2 <= i1) {
								// b1 should be UPSTREAM, but it's downstream
								pass = false;
							}
							
							if (pass && constraints[b1][b2].oc == DOWNSTREAM_OF && i1 <= i2) {
								 pass = false;
							}

							if (pass) { 
								if (abs(i1 - i2) > constraints[b1][b2].dc) {
									pass = false;
								} else {

									// we are within distance constraints.
									//	if this ends up being a pass, we'll have a possibly
									//	used distance.
									tentative_pairwise[b1][b2] = abs(i1-i2);

								}
							}

							// check negated regions
							if (pass) {
								for (list<int>::const_iterator n=constraints[b1][b2].negated.begin();
															   n!=constraints[b1][b2].negated.end(); n++) { 

									for (list<pair<int,int> >::const_iterator np=smmap[s][*n].begin(); 
																			  np!=smmap[s][*n].end(); np++) { 

										if (i1 < np->first && np->first < i2) { pass=false; break; }
										if (i2 < np->first && np->first < i1) { pass=false; break; }
									}
									if (!pass) { break; }
								}
							}

							// IF we found a pair of branches that DO NOT PASS, this 
							//	combination of motif instances DO NOT PASS, and we can
							//	stop checking them.
							if (!pass) { break; }

						} if (!pass) { break; } } // next pair of branches

					} // if (pass)

					// if ``pass'' still == true, we have a combination that works
					if (pass) {

						for (int b1=0; b1<branches.size(); b1++) { 
						for (int b2=b1+1; b2<branches.size(); b2++) { 
							if (min_used_pairwise[b1][b2] == -1 || tentative_pairwise[b1][b2] < min_used_pairwise[b1][b2]) { 
								min_used_pairwise[b1][b2] = tentative_pairwise[b1][b2];
							}
						}}

						if (min_used_global == -1 || tentative_global < min_used_global) { 
							min_used_global = tentative_global;
						}
					}

					// UPDATE ITERATORS FOR NEXT LOOP
					for (int b=branches.size()-1; b>=0; b--) { 
						iterators[b]++;
						if (iterators[b] != instances[b].end()) { break; }
						if (b==0) { 
							// first iterator at the end <=> we've tried all combinations 
							done = true;  	// only way to finish -- tried all combinations
							break; 
						}
						iterators[b] = instances[b].begin();
					}

					
				} // while !done

				// add used global distance
				global.push_back(pair<int,int>(min_used_global, s));
				// TODO:  following is debug:
				if (min_used_global == -1) { cerr << "squal[" << s << "], but I didn't find min. used global in tree " << *this << "." << endl; }
				

				// add used pairwise distance for each pair
				for (int b1=0; b1<branches.size(); b1++) { 
				for (int b2=b1+1; b2<branches.size(); b2++) { 

					pairwise[b1][b2].push_back(pair<int,int>(min_used_pairwise[b1][b2], s));

					// TODO:  following is debug:
					if (min_used_pairwise[b1][b2] == -1) {
						cerr << "squal[" << s << "], but I didn't find min. used pairwise[" 
							 << b1 << "][" << b2 << "] in tree " << *this << "." << endl;

					}

				}}
			
			} // if some instances exist

		} // squal[s]

	} // next sequence

	return; 

} // get_distances


/** print complete tree */
void Tree::print(ostream &out, const vector<string> &motif_id) { 
	
	int branch_id = 0;

	if (!(negup.empty())) { out << "-------------\nNegated regions upstream:" << endl; }
	for (list<int>::const_iterator nu=negup.begin(); nu!=negup.end(); nu++) { 
		out << motif_id[*nu] << endl;
	}
	
	for (vector<Branch*>::const_iterator b=branches.begin(); b!=branches.end(); b++) {
		out << "-------------" << endl;
		out << "Binding site (" << branch_id++ << ") on ";
		switch((*b)->strand) {
			case(TEMPLATE_STRAND):  out << "template strand"; break;
			case(TRANSCRIBED_STRAND):  out << "transcribed strand"; break;
			case(EITHER_STRAND):  out << "either strand"; break;
			default:  out << "strand value=" << ((*b)->strand);
		}
		out << ":" << endl;
		for (list<int>::const_iterator m=(*b)->motifs.begin(); m!=(*b)->motifs.end(); m++) { 
			out << motif_id[*m] << endl;
		}
	}
	
	out << "-------------" << endl;
	for (int b=0; b<branches.size(); b++) { 
		for (int b2=b+1; b2<branches.size(); b2++) { 
			if (constraints[b][b2].oc == UPSTREAM_OF) { 
				out << b << " < Upstream of < " << b2 << endl;
			} else if (constraints[b][b2].oc == DOWNSTREAM_OF) { 
				out << b2 << " < Upstream of < " << b << endl;
			} else {
				// Need this?
				out << b << ", " << b2 << " in either order" << endl;
			}

			if (constraints[b][b2].dc != INFINITY) { 
				out << b << ", " << b2 << " within " << constraints[b][b2].dc << " bp" << endl;
			} else {
				// Need this?
				out << b << ", " << b2 << " any distance apart" << endl;
			}

			
		}
	}

	if (tss != INFINITY) { out << "-------------\nCRM within " << tss << " bp of TSS." << endl; }

	if (!(negdown.empty())) { out << "-------------\nNegated regions downstream:" << endl; }
	for (list<int>::const_iterator nd=negdown.begin(); nd!=negdown.end(); nd++) { 
		out << motif_id[*nd] << endl;
	}
	
	return;
}



double read_double(istream &in) { 
	#ifndef INFINITY
	const double INFINITY = -log(0.0);
	#endif
	double ans;
	string s;
	in >> s >> ws;
	if (s == "inf") { 
		ans = INFINITY;
	} else {
		istringstream iss(s);
		iss >> ans >> ws;
	}
	return ans;
}
	

/**
 	number-of-branches, B
	branch 0
	branch 1
	...
	branch B-1
	tss
	(length) negated UP
	(length) negated DOWN
	(lenth)
		(length) oc dc oc dc ... oc dc

*/
void Tree::write(ostream &out) const { 
	out << branches.size() << endl;
	for (vector<Branch*>::const_iterator branch=branches.begin(); branch!=branches.end(); branch++) { 
		(*branch)->write(out);
	}
	out << tss << endl;
	out << negup.size() << endl;
	std::copy(negup.begin(), negup.end(), ostream_iterator<int>(out, " "));
	out << endl;
	out << negdown.size() << endl;
	std::copy(negdown.begin(), negdown.end(), ostream_iterator<int>(out, " "));
	out << endl;
	out << constraints.size() << endl;
	for (uint i=0; i<constraints.size(); i++) { 
		out << constraints[i].size() << endl;
		for (uint j=0; j<constraints[i].size(); j++) { 
			out << constraints[i][j].oc << " ";
			out << constraints[i][j].dc << " ";
		}
		out << endl;
	}
		
}

void Tree::read(istream &in) {
	uint B;
	in >> B >> ws;
	for (uint b=0; b<B; b++) { 
		Branch *branch = new Branch();
		branch->read(in);
		this->branches.push_back(branch);
	}

	tss = read_double(in);
	
	uint U;
	in >> U >> ws;
	for (uint i=0; i<U; i++) { int m; in >> m >> ws; negup.push_back(m); }
	
	uint D;
	in >> D >> ws;
	for (uint i=0; i<D; i++) { int m; in >> m >> ws; negdown.push_back(m); }

	uint Cx;
	in >> Cx >> ws;
	constraints.resize(Cx);
	for (uint i=0; i<Cx; i++) { 
		uint Cy;
		in >> Cy >> ws;
		constraints[i].resize(Cy);
		for (uint j=0; j<Cy; j++) { 
			in >> constraints[i][j].oc >> ws;
			constraints[i][j].dc = read_double(in);
		}
	}

}
	



