#ifndef TREE_H
#define TREE_H

#include "definitions.h"

#include "Branch.h"
#include "Results.h"

struct Constraint {
	int oc;
	double dc;
	list<int> negated;
	Constraint() : oc(EITHER_ORDER), dc(INFINITY) { }
	bool operator==(const Constraint &C) const { return (oc == C.oc && dc == C.dc && negated == C.negated); }
	bool operator!=(const Constraint &C) const { return (!(*this == C)); }
};

/**
 
 *	A Tree is a list of Branches and constraints between them.

 *	Trees have the ability to score over a data set

 */


struct Tree {

  friend ostream& operator<<(ostream&, const Tree&);
  
  private:
  	vector<Branch*> branches;

	// constraints[branch index i][j] -> (order constraint, distance constraint)
	vector<vector<Constraint> > constraints;
	double tss;	// global distance constraint between CRM and TSS
	list<int> negup, negdown; // negated regions up and downstream
	
	vector<bool> squal;	// current bitmap of which sequences qualify


  public:

	/** create null tree */
	Tree() { 
		#if DEBUG_MALLOC_CHECK
		tree_alloc++;  
		#endif
		tss = INFINITY;
	}

	~Tree() {
		#if DEBUG_MALLOC_CHECK
		tree_free++;
		#endif
		for (vector<Branch*>::iterator b=branches.begin(); b!=branches.end(); b++) { delete_branch(*b); }
	}

	explicit Tree(const Tree &copy) {
		#if DEBUG_MALLOC_CHECK
		tree_alloc++;  
		#endif
		branches = copy.branches;	// copy pointer list
		for (vector<Branch*>::iterator b=branches.begin(); b!=branches.end(); b++) { (*b)->pins++; }
		constraints = copy.constraints;
		tss = copy.tss;
		negup = copy.negup;
		negdown = copy.negdown;

		//squal = copy.squal;
		squal.resize(copy.squal.size());
		for (int i=0; i<squal.size(); i++) { squal[i] = copy.squal[i]; }
	}

	Tree(istream &in) { this->read(in); }

	/** File I/O */
	void Tree::write(ostream &out) const;
	void Tree::read(istream &in);

	/** size of tree -> number of branches (conjuncts) */
	inline int size() const { return branches.size(); } 

	/** size of tree[branch index] -> number of motifs (disjuncts) */
	int size(int index) const { return branches[index]->motifs.size(); }

	/** strand of tree[branch index] -> strand of said branch */
	inline int strand(int index) const { return branches[index]->strand; }

	inline const list<int>::const_iterator motif_begin(int branch) const
		{ return branches[branch]->motifs.begin(); }
	inline const list<int>::const_iterator motif_end(int branch) const
		{ return branches[branch]->motifs.end(); }
		
	inline const list<int>::const_iterator negated_begin(bool upstream) const
		{ return (upstream ? negup.begin() : negdown.begin()); }
	inline const list<int>::const_iterator negated_end(bool upstream) const
		{ return (upstream ? negup.end() : negdown.end()); }
	inline const list<int>::const_iterator negated_begin(int b1, int b2) const
		{ return constraints[b1][b2].negated.begin(); }
	inline const list<int>::const_iterator negated_end(int b1, int b2) const
		{ return constraints[b1][b2].negated.end(); }

	int compare(const Tree &t) const;
	
	bool operator==(const Tree &t) const { return compare(t) == 0; }
	bool operator!=(const Tree &t) const { return compare(t) != 0; }

	bool operator< (const Tree &t) const { return compare(t) <  0; }
	bool operator<=(const Tree &t) const { return compare(t) <= 0; }
	bool operator> (const Tree &t) const { return compare(t) >  0; }
	bool operator>=(const Tree &t) const { return compare(t) >= 0; }

	/** remove pin, last one to leave takes out the garbage */
	void delete_branch(Branch *d);
	
	/** Add a new conjunct (branch) to the tree (will go in the last position) */
	void add_conjunct(int motif, int S, const SMMap &smmap);

	/** Add a new disjunct (motif to a branch) to the given branch */
	bool add_disjunct(int position, int motif, int S, const SMMap &smmap);
	
	/** Constrain a given branch to the given strand */
	void constrain_strand(int position, int strand);

	/** reset internal bookkeeping data based on (possibly new) sequence data */
	void reset(int S, const SMMap &smmap);



	inline int order_constraint(int branch1, int branch2) const { return constraints[branch1][branch2].oc; } 
	inline double distance_constraint(int branch1, int branch2) const { return constraints[branch1][branch2].dc; } 
	inline double distance_constraint() const { return tss; }
	inline int num_negated_regions(int branch1, int branch2) const { return constraints[branch1][branch2].negated.size(); }
	inline int num_negated_regions(bool upstream) const { return ( upstream ? negup.size() : negdown.size() ); }

	/** constrain the order by supplying branch indicies (starting at 0),
	 * branch 'upstream' must be upstream of branch 'downstream'.  return false
	 * if model is already constrained */
	void constrain_order(int upstream, int downstream);
	void constrain_distance(int branch1, int branch2, double distance);
	void constrain_distance(double distance);


	/** add a negated region between two branches */
	bool negate(int branch1, int branch2, int motif);

	/** add a negated region upstream or downstream of the CRM */
	bool negate(bool upstream, int motif);


	/** remove a branch and update data structures */
	void remove(int index);
	
	/** remove a motif and update data structures */
	void remove(int branch_index, int motif_index, int S, const SMMap &smmap);

	
	/** create a new Results structure for the model. 
	 *	hint:  model is being constrained/relaxed.  The idea is if we're constraining a 
	 *			model, a sequence that didn't qualify before still won't and there's no need
	 *			to evaluate the sequence.
	 *  hint = 1:  model is being strictly constrained compared to current 'squal' structure
	 *  hint = 2:  model is being strictly relaxed compared to current 'squal' structure
	 *  hint = 0:  unknown, so evaluate all sequences
	 *
	 *	04/15/2005:  bug fix:  was incrementing iterators INSIDE pairwise-constraint-check.
	 */	
	Results* evaluate(int S, int P, const SMMap &smmap, const vector<double> &sw, int hint=0, vector<list<vector<int> > > *passlist = NULL); 


	/** 
	 * Copy contents of Tree::squal to the given location.  
	 * Valid after a call to Tree::evaluate
	 * NOT GUARANTEED TO BE VALID EXCEPT AFTER A CALL TO Tree::evaluate
	 */
	void copy_squal(vector<bool> *copy);

	/**
	 *	Among sequences that QUALIFY, get a list of RELEVANT DISTANCES between 
	 *	EACH PAIR OF BRANCHES
	 *
	 *	That is, return a list of in-between branch distances such that the 
	 *	sequence qualifies given the tree, but it WOULDN'T if the distance were
	 *	smaller than that.
	 *
	 *	return value:  pairwise[smaller-branch-index][larger-branch-index] -> vector of (distances, sequence ID)
	 *				   global -> vector of (distances, sequence ID)
	 *
	 *	NOTE:  based on the ``evaluate'' function above.  Since I'm only interested
	 *	in the examples that QUALIFY.
	 *
	 *
	 */
	void get_distances(int S, const SMMap &smmap, 
		vector<vector<vector<pair<int, int> > > > &pairwise, vector<pair<int, int> > &global);


	/** print complete tree */
	void print(ostream &out, const vector<string> &motif_id);
		
};	// struct Tree


#endif
