#include <iomanip>
using namespace std;

#include "CRM.h"
#include "PWM.h"

const char CRM_FILE_COMMENT = '#';

const string CRM_FILE_VERSION = "1.0.2";
	// VERSION:
	//	1.0:  original
	//	1.0.1:  added motif preference parameters
	//	1.0.2:	added negated motifs




void CRM::write(ostream &out) const {


	const char COMMENT[] = { CRM_FILE_COMMENT, '\t', '\0' };
	const char TAB = '\t';
	
	out << setprecision(CRM::FPP);

	out << COMMENT << "scrm2 CRM data file version " << this->file_version() << endl;

	out << size() << TAB << COMMENT << "CRM size (" << size() << " binding sites)" << endl;
	for (uint i=0; i<size(); i++) { 
		out << COMMENT << "Binding Site #" << i << " (" << (i+1) << " of " << size() << ")" << endl;
		out << (site(i).strand_preference(BindingSite::TMPL))
			<< TAB << COMMENT << "P(site on template strand) (normalized)" << endl;
		out << (site(i).negated ? "1" : "0") << TAB << COMMENT << "Whether site is negated" << endl;
		out << site(i).multiplicity() << TAB << COMMENT << "Number of motifs for site " << i << endl;
		for (uint m=0; m<site(i).multiplicity(); m++) { 
			out << COMMENT << "Binding Site #" << i << ", Motif #" << m
				<< " (motif " << (m+1) << " of " << site(i).multiplicity() << ")" << endl;
			out << site(i).motif_preference(m)
				<< TAB << COMMENT << "The probability (binding site preference) of motif #" << m << endl;
			out << site(i).motif(m).width()
				<< TAB << COMMENT << "The width of motif #" << m << endl;
			out << COMMENT
				<< "PWM entries with respect to " << DNA_ALPHABET 
				<< ", consensus is " << site(i).motif(m).consensus() 
				<< ", MLE is " << site(i).motif(m).mle() 
				<< endl;
			for (uint s=0; s<site(i).motif(m).width(); s++) { 
				for (uint b=0; b<DNA_ABLEN; b++) { 
					out << (site(i).motif(m)(s,b)) << "\t";
				}
				out << endl;
			}
		}
	}

	// PAIRWISE RELATIONSHIPS
	
	out << endl << COMMENT << "DISTANCE PREFERENCES:" << endl << endl;
	for (uint i=0; i<size(); i++) {
	for (uint j=i; j<size(); j++) {
		const Distance *distance;
		if (i == j) { 
			distance = &(this->distance(i));
			out << COMMENT << "Distance between site #" << i << " and (downstream, right) end of sequence" << endl;
		} else if (i < j) { 
			distance = &(this->distance(i,j));
			out << COMMENT << "Distance between sites #" << i << " and #" << j << endl;
		} else { continue; }
		
		out << distance->size() << TAB << COMMENT << "size (max. distance represented--set to zero for uniform) " << endl;
		out << distance->binw() << TAB << COMMENT << "bin width" << endl;
		
		for (uint x=0; x<distance->size(); x += distance->binw()) { 
			out << (distance->prob(x)) 
				<< TAB << COMMENT << "P(distance=" << x << ")" << endl;
		}
		out << endl;
	}}

	out << endl << COMMENT << "ORDER PREFERENCES:" << endl << endl;
	for (uint i=0; i<size(); i++) {
	for (uint j=0; j<size(); j++) { 

		if (i != j) { 	// i==j could be NaN or some value that can't be read properly.

			out << (order(i,j))
				<< TAB << COMMENT << "P(site #" << i << " is upstream of #" << j << ")" << endl;
		}
	}}

	return;
}

void read_to_eol(istream &in) { 
	char c='\0';
	while (in.good() && c!='\n') {
		c = in.get();
	}
}

/** scan for a T (ignoring comments) */
template <typename T>
T scan(istream &in) {

	string data;
	T ans;
	in >> ws;

	char c = in.peek();
	while (c == CRM_FILE_COMMENT) { 
		read_to_eol(in);
		in >> ws;
		c = in.peek();
	}

	
	in >> ans >> ws;

	return ans;
}



bool CRM::read(istream &in) {

	#if CRM_DEBUG
	if (this->size()) { 
		cerr << "WARNING:  Call to CRM::read(istream&) on non-empty CRM:" 
			 << endl << *this << endl;
	}
	#endif

	while (this->size()) { remove_site(0); }	// clear CRM

	try {

		const uint CRM_SIZE = scan<uint>(in);
		for (uint i=0; i<CRM_SIZE; i++) { 

			if (!in.good()) { return false; }

			BindingSite bs;
			bs.strand_preference(BindingSite::TMPL) = scan<probability>(in);
			bs.strand_preference(BindingSite::TCX) = 1 - (bs.strand_preference(BindingSite::TMPL));
			bs.negated = scan<bool>(in);
			const uint MULTIPLICITY = scan<uint>(in);

			for (uint m=0; m<MULTIPLICITY; m++) { 
				probability p = scan<probability>(in);
				const uint WIDTH = scan<uint>(in);
				PWM<probability> pwm(WIDTH);
				for (uint s=0; s<WIDTH; s++) { 
					for (uint b=0; b<DNA_ABLEN; b++) { 
						if (!in.good()) { return false; }
						pwm(s, b) = scan<probability>(in);
					}
				}
				bs.insert(bs.size(), pwm);
				bs.motif_preference(m) = p;	// as read above.
			}
			this->add_site(bs);	// add to end of list
		}

		// PAIRWISE RELATIONSHIPS

		for (uint i=0; i<size(); i++) {
		for (uint j=i; j<size(); j++) {
			Distance *distance;
			if (i == j) { 
				distance = &(this->distance(i));
			} else if (i < j) { 
				distance = &(this->distance(i,j));
			} else { continue; }
			
			const uint DISTANCE_SIZE = scan<uint>(in);
			const uint BIN_WIDTH = scan<uint>(in);
			CRM::BINW = BIN_WIDTH;	// for evaluation purposes.
			distance->resize(DISTANCE_SIZE, BIN_WIDTH);
			distance->fill(0.0);
			for (uint x=0; x<DISTANCE_SIZE; x += BIN_WIDTH) { 
				if (!in.good()) { return false; }
				probability p = scan<probability>(in);
				distance->add(x, p);
			}

			// 2006-07-12, if distance size doesn't match CRM::MAXL, resize it.
			if (DISTANCE_SIZE != CRM::MAXL+1) { 
				cerr << "WARNING:  changing distance from " 
					 << DISTANCE_SIZE << " (given in file) to " 
					 << (CRM::MAXL+1) << " (model size)." << endl;
				distance->resize(CRM::MAXL+1, CRM::BINW);
			}
			
			distance->normalize();	// won't change distribution, but this calculates CDF.
		}}

		for (uint i=0; i<size(); i++) {
		for (uint j=0; j<size(); j++) { 
			if (i != j) { 	// i==j may be NaN or something
				if (!in.good()) { return false; }
				this->order(i,j) = scan<probability>(in);
			}
		}}

		#if CRM_DEBUG
		string dummy = scan<string>(in);	// eat any remaining comments
		if (!in.eof()) { 
			cerr << "CRM::read(istream&):  WARNING:  Finished reading CRM, but stream still has data (" << dummy << ")." << endl; 
		}
		#endif

	} catch (std::exception e) { 
		cerr << "Exception caught while reading CRM data (" << e.what() << ")" << endl;
		return false;
	}

	return true;
}



string CRM::file_version() { return CRM_FILE_VERSION; }



