/*
	logscale:  A numerical (floating-point) type that actually keeps the
	logarithm of a represented number (logscale::logvalue)

	Notes:
	
	*) assignments to negative values result in a value of NaN (not a
	gaurantee)
	
	*) Additions are a little better than log(exp(p) + exp(q)), becase I need
	only be able to represent the _difference_ between the logs and not their
	linear values (see logscale::add).  However, for this class, adding logs
	still requires a call to EXP(x) and LOG(x), as opposed to doing a more
	efficient linear interpolation
		

*/

#ifndef LOGSCALE_H
#define LOGSCALE_H

#include <math.h>

typedef long double logtype;	// type of the log-scale representation

struct logscale {

  	logtype logvalue;	// the log-scale value of the number I represent

	// choose either logl or log2l (base e or 2) for these:
	static const inline logtype LOG(logtype x) { return log2l(x); }	
	static const inline logtype EXP(logtype x) { return exp2l(x); }	
	static const inline logtype BASE() { return 2; }	// corresponds to the log, exp functions defined above

	logscale() { }
	logscale(const logtype &value) { this->logvalue = LOG(value); }
	logscale(const logscale &copy) { this->logvalue = copy.logvalue; }

	template <typename _Number> inline operator _Number() const {  return (_Number)EXP(logvalue); }

	/** explicitly add two log-scale values */
	static inline logtype add(logtype p, logtype q) {
		static const logtype logzero = LOG(0.0);
		if (q == logzero) { return p; }	
		if (p == logzero) { return q; }	
		if (q > p) { logtype tmp=p; p=q; q=tmp; }	// now I guarantee p >= q
		logtype x = q-p;
		return p + LOG(1.0 + EXP(x));
	}

	/** return a logscale explicitly from a log-scale value */
	static inline logscale logscale::loginit(logtype log_value) { logscale ans;  ans.logvalue = log_value;  return ans; }

	/** Overload arithmetic operators for this new kind of number */
	const logscale& operator++() {  logvalue = LOG(EXP(logvalue) + 1.0); return *this; }	// ++L
	logscale  operator++(int) {  logscale ans=*this; logvalue = LOG(EXP(logvalue) + 1.0); return ans; } // L++
	const logscale& operator--() { logvalue = LOG(EXP(logvalue) - 1.0);  return *this; }
	logscale  operator--(int) { logscale ans=*this; logvalue = LOG(EXP(logvalue) - 1.0);  return ans; }

};


/** Arithmetic */
template <typename _Number> logtype operator+(const _Number &N, const logscale &L) { return N + logscale::EXP(L.logvalue); }
template <typename _Number> logtype operator-(const _Number &N, const logscale &L) { return N - logscale::EXP(L.logvalue); }
template <typename _Number> logtype operator*(const _Number &N, const logscale &L) { return logscale::loginit(logscale::LOG(N) + L.logvalue); }
template <typename _Number> logtype operator/(const _Number &N, const logscale &L) 
	{ return logscale::loginit(logscale::LOG(N) - L.logvalue); }


// These two are faster if you return a logtype vs. calling op(logscale,logscale)
template <typename _Number> logtype operator+(const logscale &L, const _Number &N) { return logscale::EXP(L.logvalue) + N; }
template <typename _Number> logtype operator-(const logscale &L, const _Number &N) { return logscale::EXP(L.logvalue) - N; }


inline logscale operator+(const logscale &L, const logscale &addend) {
	logscale ans;
	ans.logvalue = logscale::add(L.logvalue, addend.logvalue); 
	return ans;
}
inline logscale operator-(const logscale &L, const logscale &subtrahend) { return logscale::loginit(logscale::LOG(logscale::EXP(L.logvalue) - logscale::EXP(subtrahend.logvalue))); }
inline logscale operator*(const logscale &L, const logscale &multiplier) { return logscale::loginit(L.logvalue + multiplier.logvalue); }
inline logscale operator/(const logscale &L, const logscale &divisor) 
	{ return logscale::loginit(L.logvalue - divisor.logvalue); }




/** Arith + Assign */
// For these, return logtype in case _Number is integer
template <typename _Number> logtype operator+=(_Number &N, const logscale &L) 
	{ logtype result = ((logtype)N + logscale::EXP(L.logvalue)); N = result; return result; }
template <typename _Number> logtype operator-=(_Number &N, const logscale &L) { logtype result = ((logtype)N - logscale::EXP(L.logvalue)); N = result; return result; }
template <typename _Number> logtype operator*=(_Number &N, const logscale &L) { logtype result = ((logtype)N * logscale::EXP(L.logvalue)); N = result; return result; }
template <typename _Number> logtype operator/=(_Number &N, const logscale &L) { logtype result = ((logtype)N / logscale::EXP(L.logvalue)); N = result; return result; }
inline const logscale& operator+=(logscale &L, const logscale &a) { L.logvalue = logscale::add(L.logvalue, a.logvalue); return L; }
inline const logscale& operator-=(logscale &L, const logscale &s) { L.logvalue = logscale::LOG(logscale::EXP(L.logvalue) - logscale::EXP(s.logvalue)); return L; }
inline const logscale& operator*=(logscale &L, const logscale &m) { L.logvalue = L.logvalue + m.logvalue; return L; }
inline const logscale& operator/=(logscale &L, const logscale &d) { L.logvalue = L.logvalue - d.logvalue; return L; }





/** Comparison operators */
inline bool operator> (const logscale &L, const logscale &R) { return L.logvalue >  R.logvalue; }
inline bool operator>=(const logscale &L, const logscale &R) { return L.logvalue >= R.logvalue; }
inline bool operator< (const logscale &L, const logscale &R) { return L.logvalue <  R.logvalue; }
inline bool operator<=(const logscale &L, const logscale &R) { return L.logvalue <= R.logvalue; }
inline bool operator==(const logscale &L, const logscale &R) { return L.logvalue == R.logvalue; }
inline bool operator!=(const logscale &L, const logscale &R) { return L.logvalue != R.logvalue; }
/**

	NOTE:  For template type _Number, (_Number == logscale) and (logscale == _Number) 
	will call the above functions, or they can be defined.
	
		template <typename _Number> bool operator==(const _Number &N, const logscale &L) { ... }

	overrides
		
		inline bool operator==(const logscale &L, const logscale &R) { ... }

	although the former is ambiguous with ( *anything* == *any-number-type* )
	according to some compilers, even ( std::ios_base == int ) and such.

**/



/** I/O */
inline istream& operator>>(istream &in, logscale &L) { logtype v; in >> v; L.logvalue = logscale::LOG(v); return in; }
inline ostream &operator<<(ostream &out, const logscale &L) { 

	/**
	// COULD DO THIS:  if value is too small, use logscale.
	//	However, This may not work well with file I/O (especially I)
	logtype V = logscale::EXP(L.logvalue);
	if (V == 0.0 && L > 0.0) {
		out << logscale::BASE() << "^" << L.logvalue;
	} else {
		out << V;
	}
	*/

	out << logscale::EXP(L.logvalue);	// just print linear scale value

	return out; 
}



/** Math functions I can do faster than you */
template <typename _N> logscale pow(const logscale &L, const _N &x) { return logscale::loginit(L.logvalue * x); }
template <typename _N> logscale powf(const logscale &L, const _N &x) { return logscale::loginit(L.logvalue * x); }
template <typename _N> logscale powl(const logscale &L, const _N &x) { return logscale::loginit(L.logvalue * x); }
template <typename _N> logscale root(const logscale &L, const _N &x) { return logscale::loginit(L.logvalue / x); }
inline logscale sqrt(const logscale &L) { return logscale::loginit(L.logvalue / 2); }
inline logscale cbrt(const logscale &L) { return logscale::loginit(L.logvalue / 3); }
inline double log(const logscale &L) { static const logtype logE = logscale::LOG(::exp(1)); return L.logvalue / logE; }
inline float logf(const logscale &L) { static const logtype logE = logscale::LOG(::expf(1)); return L.logvalue / logE; }
inline long double logl(const logscale &L) { static const logtype logE = logscale::LOG(::expl(1)); return L.logvalue / logE; }
inline double log2(const logscale &L) { static const logtype log2 = logscale::LOG(2.0); return L.logvalue / log2; }
inline float log2f(const logscale &L) { static const logtype log2 = logscale::LOG(2.0); return L.logvalue / log2; }
inline long double log2l(const logscale &L) { static const logtype log2 = logscale::LOG(2.0); return L.logvalue / log2; }



/** Math functions I can't do any better (but are still ambiguous, e.g.
 * floor(double) vs. floor(long double) is resolved by defining floor(logscale)
 * */
inline logtype floor(const logscale &L) { return floor(logscale::EXP(L.logvalue)); }
inline logtype floorf(const logscale &L) { return floor(logscale::EXP(L.logvalue)); }
inline logtype floorl(const logscale &L) { return floor(logscale::EXP(L.logvalue)); }
inline logtype ceil(const logscale &L) { return ceil(logscale::EXP(L.logvalue)); }
inline logtype ceilf(const logscale &L) { return ceil(logscale::EXP(L.logvalue)); }
inline logtype ceill(const logscale &L) { return ceill(logscale::EXP(L.logvalue)); }
inline logtype fabs(const logscale &L) { return fabs(logscale::EXP(L.logvalue)); }
inline logtype fabsf(const logscale &L) { return fabs(logscale::EXP(L.logvalue)); }
inline logtype fabsl(const logscale &L) { return fabs(logscale::EXP(L.logvalue)); }
#endif
