#include <stdlib.h>

#include <iostream>
#include <iomanip>
#include <fstream>
#include <iterator>
#include <vector>
#include <list>
using namespace std;

#include "CRM.h"
#include "markov.h"
#include "Option.h"
#include "fasta.h"
#include "custom.h"

class TestExample : public Example { 

  static unsigned int next_id;

  private:
  	string dna;
	vector<probability> slm;
	string name_id;
	probability w;

  public:
	TestExample(const string &s, const MM<probability> &bg_mm, const probability &weight) {
		this->w = weight;
		ostringstream oss;
		oss << "test_example_" << (next_id++);
		this->name_id = oss.str();
		dna = s;
		slm = bg_mm.subsequence_likelihood(s);
	}
	virtual ~TestExample() { }

	const string& name() const { return this->name_id; }
	const probability& weight() const { return this->w; }
	
	const string *sequence() const { return &dna; }
	probability bg(unsigned int i, unsigned int j) const { 
		return slm[j] / slm[i];
	}
	
};
unsigned int TestExample::next_id = 0;


int main(int argc, char **argv) {
	
	double ratio;
	
	OptionParser parser("Test the CRM class and functionality.");
	parser.add("ratio", 'r', &ratio, 0.0, "Magic ratio");
	
	vector<string> args = parser.parse(argc, argv);

	CRM::MAXL = 100;	// as long as I'm going to use here.

	cout << "CRM Parameters:" << endl;
	cout << "\t" << "CRM::magic_ratio1 = " << (CRM::magic_ratio1) << endl;
	cout << "\t" << "CRM::magic_ratio2 = " << (CRM::magic_ratio2) << endl;
	cout << "\t" << "CRM::MDA = " << (CRM::MDA) << endl;
	cout << "\t" << "CRM::MAXL = " << (CRM::MAXL) << endl;
	
	cerr << "Setting CRM::magic_ratio <- " << ratio << endl;
	CRM::magic_ratio1 = CRM::magic_ratio2 = ratio;

	cout << "Create background MM..." << endl;
	MM<probability> mm(0);	// 0th order model
	mm.fill();				// set everything to zero
	mm.pseudocount();
	mm.normalize();

	cout << "Create data..." << endl;
	string motif = "ACGGGTTT";	// length of 8
	//                 01234567890123456789             01234567890123456789  // length of 40
	string sequence = "actgatcgatgcatcacttt" + motif + "tacgactatatatataatcg";
	istringstream iss(sequence);
	TestExample example(sequence, mm, 1.0);
	vector<Example*> data;
	data.push_back(&example);
	cout << "Sequence:  '" << sequence << "'" << endl;
	cout << "example.(3,5) = \n\t" << example.bg(3,5) << endl;

	cout << "Create PWM..." << endl;
	PWM<probability> pwm(motif.length());
	pwm.add(motif);
	CRM crm;
	crm.add_site(pwm);

	cout << "CRM (test printing):" << endl << crm << endl;
	cout << "CRM (test summarize):  ";
	crm.summarize(cout);
	cout << endl;
	
	probability LCRM = exp2(-40) / 2.0;	// div by 2 for strand
	probability LBGM = exp2(-48);
	probability P = LCRM / (LCRM+LBGM);

	vector<Example*> v_examples;
	v_examples.push_back(&example);

	cout << "Sequence likelihood = " << likelihood(crm, v_examples)[0] << endl;
	cout << "probability of CRM = " << probabilityof(crm, v_examples)[0] << endl;
	cout << "Theorectcal = " << P << " (a little lower due to distance)" << endl;

	cout << "Calling EM (on one sequence.  Things shouldn't change much.)" << endl;
	cout << endl << endl << "BEFORE:" << endl << crm << endl;
	
	CRM used;
	CRM normalized;
	EM(crm, data, &used, &normalized, true);		// true <=> normalize over positive-labelled sequences
	cout << endl << endl << "Used:" << endl << used << endl;
	cout << endl << endl << "Normalized:" << endl << normalized << endl;

	cout << "Done with EM" << endl;

	cout << "Add another copy as a second motif to first site:" << endl;
	crm.add_motif(crm.size()-1, pwm);
	cout << crm << endl;

	PWM<probability> dummy_pwm(10);
	for (int i=0; i<40; i++) { dummy_pwm(i%10, i/10) = rand(); }
	dummy_pwm.normalize();

	cout << "Add a second binding site (random PWM):" << endl;
	crm.add_site(dummy_pwm);
	cout << crm << endl;
	cout << "And test `summarize' again:  ";
	crm.summarize(cout);
	cout << endl;

	cerr << endl << endl << endl << "test-CRM is all done!" << endl;

	return 0;
}
	
	

	
	
