#include "metric.h"

// available metrics to use
double metric_acc(const vector<pair<probability, probability> > &class_prediction);	// plain old accuracy
double metric_cmargin(const vector<pair<probability, probability> > &class_prediction);	// classification margin
double metric_auc(const vector<pair<probability, probability> > &class_prediction);	// area under ROC curve
double metric_Fx(const int x, const vector<pair<probability, probability> > &class_prediction);	// generalized F1 score



/** THE scoring metric */
double metric(const vector<pair<probability, probability> > &class_prediction) {
	return metric_Fx(1, class_prediction);
}
string metric() { return "F1"; }





// -------------------------- ROC-style metrics ---------------------


struct SortPairBySecondDecending { 
	bool operator()(const pair<probability, probability> &LHS, const pair<probability, probability> &RHS) { 
		return (RHS.second < LHS.second);
	}
};


double metric_cmargin(const vector<pair<probability, probability> > &class_prediction) {

	vector<pair<probability, probability> > M = class_prediction;	// copy
	std::sort(M.begin(), M.end(), SortPairBySecondDecending());		// from most-confidant prediction on down

	// count classes
	probability P=0, N=0;
	for (vector<pair<probability, probability> >::const_iterator i=M.begin(); i!=M.end(); i++) {
		P += i->first;
		N += 1 - i->first;
	}

	if (N==0) { return 1; }
	if (P==0) { return 0; }

	probability TP=0, FP=0;
	double best_margin = 0;
	for (vector<pair<probability, probability> >::const_iterator i=M.begin(); i!=M.end(); i++) { 
		
		// consider example a positive prediction.
		TP += i->first;
		FP += 1 - i->first;
		
		// compute margin
		double margin = (TP/P) - (FP/N);

		// test for the best margin
		best_margin = std::max(best_margin, margin);

	}
	return best_margin;
}


/** Compute the (A)rea (U)nder the ROC (C)urve */
double metric_auc(const vector<pair<probability, probability> > &class_prediction) {

	vector<pair<probability, probability> > M = class_prediction;	// copy
	std::sort(M.begin(), M.end(), SortPairBySecondDecending());	// from most-confidant prediction on down

	// count classes
	probability P=0;
	for (vector<pair<probability, probability> >::const_iterator i=M.begin(); i!=M.end(); i++) {
		P += i->first;
	}
	probability N = ((probability)(M.size())) - P;

	if (N==0) { return 0; }
	if (P==0) { return 0; }

	probability tp=0;	// true positives (rate is tp/P)
	double area = 0;

	for (vector<pair<probability, probability> >::const_iterator i=M.begin(); i!=M.end(); i++) { 

		const probability p = i->first, n = (1-i->first);	// shorthand:  p=truepositiveness of example
		
		// add area
		area += (tp/P) * (n/N);	// add area of "rectangle"
		area += ((p/P) * (n/N)) / 2;	// add area of "triangle"
		
		// update current ROC point
		tp += p;

	}

	return area;

}


// -------------------------- Fx scoring metric ---------------------

// for a vector of these four statistics, TP=true positives, FN=false negatives, ...
const int TP=0,FP=1,TN=2,FN=3;
	
/***
 * Populate a vector of < TP, FP, TN, FN > given a list of (class, prediction) pairs.
 *
 * Each class, prediction are real numbers [0,1], indicating the extent to which they
 *	are positive, e.g., class=1.0 is positive, 0.0 is negative, 0.75 is maybe 3/4 likely to be positive.
 *
 */
vector<double> classification_counts(const vector<pair<probability, probability> > &class_prediction) {

	// return:  < TP, FP, TN, FN >


	vector<double> ans(4);
	std::fill(ans.begin(),ans.end(),0.0);
	
	for (vector<pair<probability, probability> >::const_iterator i=class_prediction.begin(); i!=class_prediction.end(); i++) { 
		//       Class        Prediction
		ans[TP] += (i->first) * (i->second);		//  C &  P
		ans[FP] += (1-i->first) * (i->second);	// ~C &  P
		ans[TN] += (1-i->first) * (1-i->second);	// ~C & ~P
		ans[FN] += (i->first) * (1-i->second);	//  C & ~P
	}

	return ans;
}

inline double precision(const vector<double> &c) { return (c[TP]+c[FP] == 0.0) ? 0.0 : c[TP] / (c[TP]+c[FP]); }
inline double recall(const vector<double> &c) { return (c[TP]+c[FN] == 0.0) ? 0.0 : c[TP] / (c[TP]+c[FN]); }


// Fx meta-function:  harmonic mean of recall and 'x' copies of precision 
struct Fx {
  private:
  	const uint x;
  public:
  	Fx(uint arg_x) : x(arg_x) { }
	double operator()(const vector<pair<probability, probability> > &class_prediction) {
		vector<double> c = classification_counts(class_prediction);
		double P = precision(c);
		double R = recall(c);
		double F = (P==0.0 && R==0.0) ? 0.0 : ((x+1)*P*R) / (x*R+P); 
		return F;
	}
};


/**
 * "Fx" is a sort of generalized F1.  F1 is the harmonic mean (reciprocal of the average reciprocal)
 *	of precision and recall.  "Fx" is the harmonic mean of recall and "x" copies of the precision.
 *	Hence, it can be used to put more weight on precision.
 */
double metric_Fx(const int x, const vector<pair<probability, probability> > &class_prediction) {
	return Fx(x)(class_prediction);
}



// (good, old-fashioned) accuracy scoring metric
double metric_acc(const vector<pair<probability, probability> > &class_prediction) {
	vector<double> c = classification_counts(class_prediction);
	return c[TP] + c[TN];
}



