#include "Option.h"
#include <string.h>	// strcpy (for reading char*)
#include <sys/ioctl.h>	// terminal width
#include <list>
#include <sstream>
using namespace std;

/** read white space and return number of newlines included in that white space */
unsigned int wss(istream &in) { 
	unsigned int ans = 0;
	while (in.good() && in.peek() <= ' ') {
		char c=in.get(); 
		ans += (c=='\n'?1:0); 
	}
	return ans;
}
	
	

/**
 *	Print in a fixed-width box, never printing past the right edge 
 *	Indent as given:

     ___________________________________
    |<- offset ->text is being printed  |
    |                   here and if a   |
    |                   word doesn't fit|
    |<----- indent ---->on a line, it   |
    |                   gets printed on |
    |                   the next.       |
    |                                   |
    |<------------ width -------------->|
    |                                   |
    |                                   |

 *
 */
void pretty(istringstream &in, ostream &out, unsigned int width=-1, 
            unsigned int indent=0, unsigned int offset=0) { 

	unsigned int c = offset;	// cursor position
	string text;		// each token
	string sep = "";	// text separator

	while (in.good()) {
		
		unsigned int newline = wss(in);	// count newlines as i read white space
		if (!in.good()) { break; }
		in >> text;

		if (newline || c + sep.length() + text.length() > width) {
			for (unsigned int i=0; i<std::max(newline, (unsigned int)1); i++) { out << endl;  }
			c = 0;
			for (unsigned int i=0; i<indent; i++) { out << " "; c++; }
		} else {
			out << sep;
			c += sep.length();
		}

		out << text;
		c += text.length();
		sep = " ";

	}
}

/** 
	[Try to] Determine terminal width and store in 'width' 
	return true iff successful
	REQUIRES:  #include <sys/ioctl.h>
 */
bool terminal_width(int &width) { 

	struct winsize ws;
	bool ans;	// true,false success value

	if (ioctl(1, TIOCGWINSZ, (void *)&ws) == 0)  {
		width = ws.ws_col;
		ans = true;
	} else {
		ans = false;
	}
	return ans;
}


bool simple_numeric(const string &s) {

	bool decimal = false;
	
	for (unsigned int i=0; i<s.length(); i++) { 
		
		       if (i==0 && (s[i]=='-' || s[i]=='+')) { 
		} else if ('0' <= s[i] && s[i] <= '9') { 
		} else if (s[i]=='.' && !decimal) { decimal = true;
		} else {
			return false;
		}
	}
	return true;
}

bool numeric(const string &s) { 
	
	if (simple_numeric(s)) { return true; }

	unsigned int e = std::min(s.find("E"), s.find("e"));
	if (e < s.length() 
			&& simple_numeric(s.substr(0,e)) 
			&& simple_numeric(s.substr(e+1))) { 
		return true; 
	}

	return false;
}


void literalize(string &s) { 

	const string R[][2] = { {"\t", "\\t"}, {"\n", "\\n"}, {"\r", "\\r"},
	                        {"\a", "\\a"}, {"\b", "\\b"}, {"\e", "\\e"},
	                        {"\f", "\\f"}, {"\v", "\\v"} };

	for (unsigned int i=0; i<2; i++) { 
		
		int p = (int)(s.find(R[i][0]));
		if (p >= 0) { s.replace(p, 1, R[i][1]); }

	}

	return;
}



//-----------------------------------------------------------------------------


OptionParser::OptionParser(string syn) : synopsis(syn) {
	this->add("help", '?', &reserved_help_param, false, "Print this help message and exit");
	this->add("usage", '-', &reserved_usage_param, false, "Print usage and exit");
	this->program = "<this-program>";
	this->rest = "<remaining-arguments>";
}

OptionParser::~OptionParser() { 
	for (vector<UntypedOption*>::iterator i=options.begin(); i!=options.end(); i++) { delete *i; }
}




vector<string> OptionParser::parse(int argc, const char **argv, string rest) {

	char **non_const_argv = new char*[argc];
	for (int i=0; i<argc; i++) { 
		string arg = argv[i];
		non_const_argv[i] = new char[arg.length()+1];
		strcpy(non_const_argv[i], arg.c_str());
	}
	return parse(argc, non_const_argv, rest);
}


vector<string> OptionParser::parse(int argc, char **argv, string rest) {

	vector<string> ans;

	bool error = false;
	ostringstream error_oss;	

	this->rest = rest;	// store for subsequent calls to 'help' or 'usage'
	this->program = argv[0];	// store for same reason

	// create a copy of command-line arguments
	list<string> arguments(argc-1);
	std::copy(argv+1, argv+argc, arguments.begin());	// +1 <=> skip program name

	// map of which options have been found (duplicates are errors, unmentioned)
	vector<bool> processed(options.size());
	std::fill(processed.begin(), processed.end(), false);

	for (list<string>::iterator arg=arguments.begin(); arg!=arguments.end(); ) {

		int applicable_index = -1;	// which option?

		for (unsigned int option_index=0; option_index<options.size(); option_index++) {
			
			UntypedOption *option = options[option_index];
			
			const unsigned int L = arg->length();	// shorthand
			const string A = *arg;
			const char F = option->flag;
			const string N = option->name;

			if (   (A.find("--")==0 && A.find(N)==2 && A.find("=")==2+N.length())
						|| (A.find("-")==0 && A[1]==F && A.find("=")==2) ) {
				
				// --name=VALUE or -f=VALUE

				if ( (A.find("-")==0 && A[1]==F && A.find("=")==2) && F==NULL_FLAG ) { 
					// special case:  -\0=value, by definition, NULL_FLAG is unflaggable
					break;
				}
				
				applicable_index = (int)option_index;
				istringstream iss(A.substr(A.find("=")+1));
				option->read(iss);
				arg = arguments.erase(arg);
				break;

			} else if (L>2 && A[0]=='-' && A[1]==F && F!='-') {
				
				// -fVALUE 

				if (F == NULL_FLAG) { break; }
				
				applicable_index = (int)option_index;
				if (option->requires_value()) { 
					istringstream iss(A.substr(2));
					option->read(iss);
					arg = arguments.erase(arg);
					break;
				} else {
					option->set();
					*arg = "-" + A.substr(2);
					// Do not inc. `arg'
					break;
				}

			} else if ( (L==2 && A[0]=='-' && A[1]==F) ||
						(L==2+N.length() && A.substr(0,2)=="--" && A.substr(2)==N) ) {
				
				// -f (white space) [value] or --name (white space) [value] 

				if ((L==2 && A[0]=='-' && A[1]==F) && F==NULL_FLAG) { break; }
				
				applicable_index = (int)option_index;
				if (option->requires_value()) {
					list<string>::iterator next=arg;
					next++;
					if (next==arguments.end()) { 	
						error=true;
						error_oss << "Expected " << option->type()
								  << " after `" << A << "' "
								  << "(" << option->usage() << ").  ";
						arg = arguments.erase(arg);
						break;
					}
					arg = arguments.erase(arg);
					istringstream iss(*arg);
					option->read(iss);
					arg = arguments.erase(arg);
					break;
				} else {
					// no value required
					option->set();
					arg = arguments.erase(arg);
					break;	
				}
		
			} else {
				// *arg is not a flagged option at all
			}
		} // next option

		if (applicable_index < 0) { 
			// not an option
			ans.push_back(*arg);
			arg++;
		} else {
			unsigned int index = (unsigned int)(applicable_index);
			if (processed[index]) { 
				error = true;
				error_oss << "Multiple initialization of option `"
				          << options[index]->usage() << ".'  ";
			}
			processed[index] = true;
		}
		
	} // next argument


	if (! (reserved_usage_param || reserved_help_param)) {
		// if help or usage given, don't consider these errors

		// Make sure all required arguments are present.
		for (unsigned int i=0; i<options.size(); i++) { 
			if (options[i]->required && !processed[i]) { 
				error = true;
				error_oss << "Missing required value for `"
						  << options[i]->usage() << ".'  ";
			}
		}

		// Okay, now make sure there are no non-Option<T> options:
		//	(something beginning with '-', but not numeric
		for (vector<string>::const_iterator arg=ans.begin(); arg!=ans.end(); arg++) { 
			if (arg->length() && arg->substr(0,1)=="-" && !numeric(*arg)) { 
				error = true;
				error_oss << "Unrecognized option `" << *arg << ".'  ";
			}
		}
	}

	if (reserved_usage_param) { usage(cout); }
	if (reserved_help_param) { help(cout); }
	
	if (reserved_usage_param || reserved_help_param) {
		for (unsigned int i=0; i<options.size(); i++) {
			delete (options[i]); 
		}
		exit(0);
	}

	if (error) { throw OptionException(error_oss.str()); }
	
	return ans;
	
} // parse



void OptionParser::usage(ostream &out) const {

	int TW;	// terminal width
	if (!terminal_width(TW)) { TW = 80; } // set term. width, otherwise use "default"

	ostringstream oss;

	oss << "Usage: " << program << " ";

	unsigned int indent = oss.str().length();

	// skip first two options, help and usage
	for (unsigned int i=0 /*NUM_RESERVED*/; i<options.size(); i++) { 
		options[i]->usage(oss);
		oss << " ";
	}
    oss << rest << endl;

	istringstream iss(oss.str());

	pretty(iss, out, std::max(10, TW), indent, 0);

	out << endl << endl << program << " --" << options[RESERVED_HELP]->name << " for help." << endl << endl;

    return;

}

void OptionParser::help(ostream &out) const {

	//
	//  program:  synopsis synopsis synopsis synopsis synopsis synopsis 
	//			  synopsis synopsis synopsis synopsis synopsis synopsis 
	//            synopsis synopsis.
	//
	//  Usage:  program [options] rest
	//
	//  options:
	//
	//	 -s|--name Integer   Required:  Alice Doesn't Live Here Anymore
	//
	//  [-s|--name String ]  Blah blah blah blah blah blah blah blah
	//                           blah blah blah blah blah blah blah blah 
	//                           blah blah blah blah blah blah blah blah
	//                           blah blah blah 
	//                           (Optional.  Default = "Blah".)
	//	...
	//

	int TW;	// terminal width
	bool display = terminal_width(TW);
	const unsigned int WIDTH = (display) 
								? std::max( (unsigned int)(TW-2), (unsigned int)40 )
								: (unsigned int)(80);
		// -2 for a little breathing room on the right side, but set a minimum value
		//	that's pretty wide.  If !display, use 80 for files


	unsigned int indent;	// used throughout
	ostringstream oss;	// used throughout

	oss.str("");
	oss << "Synopsis: ";
	indent = oss.str().length();
	oss << synopsis;
	{
		istringstream iss(oss.str());
		pretty(iss, out, WIDTH, indent, (unsigned int)0);
	}	

	out << endl << endl;

	oss.str("");
	oss << "Usage: ";
	indent = oss.str().length();
	oss << program << " [options] " << rest;
	{
		istringstream iss(oss.str());
		pretty(iss, out, WIDTH, indent, 0);
	}

	out << endl << endl << "options:" << endl << endl;

	unsigned int MAX_OPT_USAGE = 0;

	for (unsigned int i=0; i<options.size(); i++) { 
		UntypedOption *option = options[i];
		oss.str("");
		option->usage(oss);
		if (oss.str().length() > MAX_OPT_USAGE) { 
			MAX_OPT_USAGE = oss.str().length();
		}
	}

	for (unsigned int i=0; i<options.size(); i++) { 

		UntypedOption *option = options[i];
		oss.str("");
		option->usage(oss);
		unsigned int W = oss.str().length();
		unsigned int D = MAX_OPT_USAGE - W;

		indent = 0;
		out << "  "; indent += 2;
		out << oss.str(); indent += W;
		for (unsigned int i=0; i<D; i++) { out << " "; indent++; }
		out << "  "; indent += 2;

		oss.str("");
		if (option->required) { oss << "Required: "; }
		oss << option->help;
		if (!option->required && option->requires_value()) {
			oss << "\n(Optional.  Default=";
			ostringstream def_oss;
			
			option->default_value(def_oss);
			string lit_def = def_oss.str();
			literalize(lit_def);
			
			oss << lit_def;
			
			oss << ".)";
		}
		
		{
			istringstream iss(oss.str());
			pretty(iss, out, WIDTH, indent, indent);
		}

		out << endl << endl;
	}

	return;
}
		
		

	

void OptionParser::dump(ostream &out, string separator, string delim, bool terse) const {

	for (unsigned int i=NUM_RESERVED; i<options.size(); i++) { 

		const UntypedOption *option = options[i];

		string description;
		if (terse && option->flag == NULL_FLAG) { 
			description = option->name;
		} else if (terse) { 
			description = option->flag;		// string <-assign char ?
		} else {
			description = option->help;
		}

		istringstream iss(description);
		pretty(iss,out,((unsigned int)-1),0,0);	// pretty will collapse white space
		out << separator;

		ostringstream voss;
		option->value(voss);
		string lit_val = voss.str();
		literalize(lit_val);

		out << lit_val;
		out << delim;

	}

	return;
}

bool OptionParser::duplicate() { 

	bool ans = false;
	ostringstream warning_oss;
	string name = options.back()->name;
	char flag = options.back()->flag;

	for (unsigned int i=0; i<(options.size()-1); i++) { 
		if (options[i]->name == name) { 
			warning_oss << "Multiple options called `" << name << ".'  ";
			ans = true;
		}
		if (options[i]->flag == flag && flag != NULL_FLAG) { 
			warning_oss << "Multiple options with flag `-" << flag << ".'  ";
			ans = true;
		}
	}

	if (ans) { throw(OptionWarning(warning_oss.str())); }
	
	return ans;
}
	

template <> void Option<int>::type(ostream &out) const { out << "Integer"; }
template <> void Option<short>::type(ostream &out) const { out << "Integer"; }
template <> void Option<long>::type(ostream &out) const { out << "Integer"; }
template <> void Option<unsigned int>::type(ostream &out) const { out << "Positive Integer"; }
template <> void Option<unsigned long>::type(ostream &out) const { out << "Positive Integer"; }
template <> void Option<float>::type(ostream &out) const { out << "Real"; }
template <> void Option<double>::type(ostream &out) const { out << "Real"; }
template <> void Option<long double>::type(ostream &out) const { out << "Real"; }
template <> void Option<char>::type(ostream &out) const { out << "Char"; }
template <> void Option<char*>::type(ostream &out) const { out << "String"; }
template <> void Option<std::string>::type(ostream &out) const { out << "String"; }
template <> void Option<bool>::type(ostream &out) const { out << ""; }

template <> void Option<std::string>::value(ostream &out) const { out << "\"" << (*ptr) << "\""; }
template <> void Option<char*>::value(ostream &out) const { out << "\"" << (*ptr) << "\""; }
template <> void Option<bool>::value(ostream &out) const { out << ((*ptr!=def) ? "set" : "unset"); }

template <> void Option<std::string>::default_value(ostream &out) const { out << "\"" << def << "\""; }
template <> void Option<char*>::default_value(ostream &out) const { out << "\"" << def << "\""; }
template <> void Option<bool>::default_value(ostream &out) const { out << (def ? "true" : "false"); }
	
template <> bool Option<bool>::requires_value() const { return false; }
template <> void Option<bool>::set() { (*ptr) = !def; }

template <> void Option<string>::read(istream &in) { 
	// default, in >> string, will just read until white space
	(*ptr) = "";
	while (in.good()) { 
		char c = in.get();
		if (c == -1) { break; }	// EOF
		(*ptr) += c;
	}
}

template <> void Option<char*>::read(istream &in) { 
	char *position = (*ptr);
	while (in.good()) { 
		char c = in.get();
		if (c == -1) { break; }	// EOF
		*(position++) = c;
	}
	*position = '\0';
}
		


