
#include <math.h>
#include <iostream>
#include <vector>
#include <list>
#include <algorithm>
#include <iomanip>
#include <string>
#include <sstream>
#include <time.h>
using namespace std;

#include "Option.h"
#include "logscale.h"

typedef double COMPARE_TYPE;

#ifndef rnd
#define rnd() ((long double)rand() / RAND_MAX)
#endif



void check(string expression, logtype LS, logtype expectation, ostream &out = cout) { 

	out << "check  \"" << expression << "\":";
	for (unsigned int i=0; i<(25-expression.length()); i++) { out << " "; }
	out << LS;
	if (isnan(LS) || fabs((long double)(LS - expectation)) > 0.00001) {
		out << "\t\t ***** Error? (" << LS << " != " << expectation << ")";
	}
	out << endl << "-------------" << endl;
	return;
}



void speed_test(bool add, bool use_logscale, unsigned int N) { 

	vector<logscale> L(N);
	vector<double> R(N);
	for (unsigned int i=0; i<N; i++) { 
		R[i] = -16445 - (rnd() * 2);
		if (R[i] > 0) {
			cerr << "R too big (" << i << ")" << endl;
			R.resize(i);
			L.resize(i);
			break;
		}
		L[i] = logscale::loginit(R[i]);
	}

	int x = rand() % N;
	cout << "R[" << x << "] = " << R[x] << ", "
	     << "L[" << x << "] = " << L[x] << "." << endl;

	logscale P;
	COMPARE_TYPE C;

	if (!add) { 

		if (use_logscale) { 
			
			P = 1;
			for (unsigned int i=0; i<N; i++) {  P *= L[i]; }
			cout << "(logscale) product = " << P << " (logvalue = " << (P.logvalue) << ")" << endl;

		} else {

			C = (COMPARE_TYPE)(logscale::LOG(1.0));
			for (unsigned int i=0; i<N; i++) {  C = C + (COMPARE_TYPE)(R[i]); }
			cout << "(COMPARE_TYPE) product = " << logscale::EXP(C) << " (C = " << C << ")" << endl;

		}

	} else {

		if (use_logscale) { 
			P = 0;
			for (unsigned int i=0; i<N; i++) {  P = P + L[i]; }
			cout << "(logscale) sum = " << P << " (logvalue = " << (P.logvalue) << ")" << endl;
		} else {
			C = logscale::LOG(0);
			for (unsigned int i=0; i<N; i++) { 
				C = logscale::LOG( logscale::EXP(C) + logscale::EXP(R[i]) );
				
			}
			cout << "(COMPARE_TYPE) sum = " << logscale::EXP(C) << " (C = " << C << ")" << endl;
		}
	}

	return;
}

int main(int argc, char **argv) {
	
	int seed;
	int precision;
	bool fixed;

	bool test_speed;
	bool st_use_logscale, st_add;
	unsigned int st_N;

	OptionParser parser("test of logscale type");
	parser.add("seed", 's', &seed, -1, "PRNG seed");
	parser.add("fixed", 'f', &fixed, false, "fixed point notation");
	parser.add("precision", 'p', &precision, -1, "output floating point precision");

	parser.add("speed", 't', &test_speed, false, "Test speed of multiple multiplications (or additions)");
	parser.add("add", 'a', &st_add, false, "for speed test, test addition (not multiplication)");
	parser.add("nolog", 'n', &st_use_logscale, true, "for speed test, use comparison type (not logscale)");
	parser.add("N", 'N', &st_N, 1000000, "for speed test, Number of speed-test calculations");
	parser.parse(argc, argv);

	if (fixed) { cout << fixed; }
	if (precision >= 0) { cout << setprecision(precision); }

	if (test_speed) { 
		srand((unsigned int)seed);
		speed_test(st_add, st_use_logscale, st_N);
		exit(0);
	}
	
	if (seed < 0) { seed = time(NULL); }
	cout << "seeding with " << seed << endl;
	srand((unsigned int)seed);

	COMPARE_TYPE C = (COMPARE_TYPE)(100.0 * rnd());
	logscale P = (100.0*rnd());

	cout << "\n-- INITIALIZATIONS --\n" << endl;
	cout << "P = " << P << ", C = " << C << endl;
	logscale T1;	cout << "T1 = " << T1 << " (?)" << endl;
	logscale T2=C;	cout << "T2 = " << T2 << " (" << C << ")" << endl;
	logscale T3=P;	cout << "T3 = " << T3 << " (" << P << ")" << endl;

	cout << "\n-- COMPARISONS --\n" << endl;
	cout << "P = " << P << ", C = " << C << endl;
	check("P <  C", P <  C, logscale::EXP(P.logvalue) <  C); check("C <  P", C <  P, C <  logscale::EXP(P.logvalue));
	check("P <= C", P <= C, logscale::EXP(P.logvalue) <= C); check("C <= P", C <= P, C <= logscale::EXP(P.logvalue));
	check("P >  C", P >  C, logscale::EXP(P.logvalue) >  C); check("C >  P", C >  P, C >  logscale::EXP(P.logvalue));
	check("P >= C", P >= C, logscale::EXP(P.logvalue) >= C); check("C >= P", C >= P, C >= logscale::EXP(P.logvalue));
	check("P == C", P == C, logscale::EXP(P.logvalue) == C); check("C == P", C == P, C == logscale::EXP(P.logvalue));
	check("P != C", P != C, logscale::EXP(P.logvalue) != C); check("C != P", C != P, C != logscale::EXP(P.logvalue));

	cout << "\n--MATH BINARY OPERATORS--\n" << endl;
	cout << "P = " << P << ", C = " << C << endl;
	check("P + C", P+C, logscale::EXP(P.logvalue) + C); check("C + P", C+P, C + logscale::EXP(P.logvalue));
	check("P - C", P-C, logscale::EXP(P.logvalue) - C); check("C - P", C-P, C - logscale::EXP(P.logvalue));
	check("P * C", P*C, logscale::EXP(P.logvalue) * C); check("C * P", C*P, C * logscale::EXP(P.logvalue));
	check("P / C", P/C, logscale::EXP(P.logvalue) / C); check("C / P", C/P, C / logscale::EXP(P.logvalue));
	check("P + P", P+P, (logscale::EXP(P.logvalue) * 2));

	cout << "\n--UPDATE OPERATORS--\n" << endl;
	logscale OLD = P;
	cout << "P = " << P << endl;
	check("P++", P++, OLD);
	check("--P", --P, OLD);
	check("P--", P--, OLD);
	check("++P", ++P, OLD);
	C=(COMPARE_TYPE)(100*rnd()); P=C; cout << "P = " << P << ", C = " << C << endl; check("P+=C", P+=C, 2*C);  check("P", P, 2*C);
	C=(COMPARE_TYPE)(100*rnd()); P=C; cout << "P = " << P << ", C = " << C << endl; check("P-=C", P-=C, 0);  check("P", P, 0);
	C=(COMPARE_TYPE)(100*rnd()); P=C; cout << "P = " << P << ", C = " << C << endl; check("P*=C", P*=C, C*C);  check("P", P, C*C);
	C=(COMPARE_TYPE)(100*rnd()); P=C; cout << "P = " << P << ", C = " << C << endl; check("P/=C", P/=C, 1);  check("P", P, 1);
	
	C=(COMPARE_TYPE)(100*rnd()); P=C; cout << "P = " << P << ", C = " << C << endl; check("C+=P", C+=P, 2*P);  check("C", C, 2*P);
	C=(COMPARE_TYPE)(100*rnd()); P=C; cout << "P = " << P << ", C = " << C << endl; check("C-=P", C-=P, 0);  check("C", C, 0);
	C=(COMPARE_TYPE)(100*rnd()); P=C; cout << "P = " << P << ", C = " << C << endl; check("C*=P", C*=P, P*P);  check("C", C, P*P);
	C=(COMPARE_TYPE)(100*rnd()); P=C; cout << "P = " << P << ", C = " << C << endl; check("C/=P", C/=P, 1);  check("C", C, 1);

	P=(100*rnd());
	C=(COMPARE_TYPE)(100*rnd());

	cout << "\n--MATH FUNCTIONS--\n" << endl;
	cout << "P = " << P << ", C = " << C << endl;
	check("pow(P,C)", pow(P,C), pow((double)(logscale::EXP(P.logvalue)), (double)C));
	check("root(P,C)", root(P,C), root((double)(logscale::EXP(P.logvalue)), (double)C));
	check("sqrt(P)", sqrt(P), sqrt((double)(logscale::EXP(P.logvalue))));
	check("cbrt(P)", cbrt(P), cbrt((double)(logscale::EXP(P.logvalue))));
	P = 2.72;	// *close* to E
	cout << "P = " << P << endl;
	cout << "P.logvalue = " << P.logvalue << endl;
	check("log(P)", log(P), log(double(logscale::EXP(P.logvalue))));
	check("logl(P)", logl(P), logl((long double)(logscale::EXP(P.logvalue))));
	check("log2(P)", log2(P), log2((double)(logscale::EXP(P.logvalue))));
	check("log2l(P)", log2l(P), log2l((long double)(logscale::EXP(P.logvalue))));
	check("P + P * P + C * C", P + P * P + C * C, 
		logscale::EXP(P.logvalue)+logscale::EXP(P.logvalue)*logscale::EXP(P.logvalue)+C*C);
	check("fabs(P)", fabs(P), logscale::EXP(P.logvalue));
	check("fabsf(P)", fabsf(P), logscale::EXP(P.logvalue));
	check("fabsl(P)", fabsl(P), logscale::EXP(P.logvalue));
	check("ceil(P)", ceil(P), ceil(logscale::EXP(P.logvalue)));
	check("floor(P)", floor(P), floor(logscale::EXP(P.logvalue)));

	cout << "\n--STREAM INPUT--\n" << endl;
	istringstream iss1("89"), iss2("64.0"), iss3("2.8e-1984");
	cout << iss1.str() << " >> P;";  iss1 >> P; cout << "\tP = " << P << endl;
	cout << iss2.str() << " >> P;";  iss2 >> P; cout << "\tP = " << P << endl;
	cout << iss3.str() << " >> P;";  iss3 >> P; cout << "\tP = " << P << endl;

	cout << "\n--CONVERSIONS TO logscale--\n" << endl;
	cout << "P = (logscale)(3.14);"; P=(logscale)(3.14); cout << "\tP = " << P << endl;
	cout << "P = 1;";		P=1;	check("P", P, 1); 
	cout << "P = 1.1;";		P=1.1;	check("P", P, 1.1); 
	cout << "P = -1;";		P=-1;	cout << "P = " << P << " (should be nan)" << endl;
	cout << "P = -1.1;";	P=-1.1;	cout << "P = " << P << " (should be nan)" << endl;
	cout << "P = std::max(3, 9);"; P=std::max(3,9); cout << P << endl;



	cout << "\n--THAT STUPID list<logscale> PROBLEM--\n" << endl;
	list<logscale> V;
	V.push_back(0.4829423);
	V.push_back(11.34892);
	V.push_back(23.23*19.19*58.58);
	cout << "V ";
	for (list<logscale>::iterator i=V.begin(); 
	    i != V.end();	// I've had ambiguity problems with `bool operator!=(_Number, logscale)'
		i++) { 
		cout << " -> " << *i;
	}
	cout << endl;

	cout << "\n--PRINTING--\n" << endl;

	for (logscale L = 1; ; L /= 2.0) { 

		logscale k;
		logtype V;
	
		k = L / 8;
		V = logscale::EXP(k.logvalue);
		if (V == 0.0) { cout << L << "\t" << "(logvalue = " << L.logvalue << ")" << endl; }	
	
		k = L * 8;
		V = logscale::EXP(k.logvalue);
		if (V == 0.0) { break; }	
	
	}

	cout << "\n--SIZES!--\n" << endl;

	cout << "sizeof(int) = " << (sizeof(int)) << endl;
	cout << "sizeof(short) = " << (sizeof(short)) << endl;
	cout << "sizeof(char) = " << (sizeof(char)) << endl;
	cout << "sizeof(bool) = " << (sizeof(bool)) << endl;
	cout << "sizeof(long unsigned int) = " << (sizeof(long unsigned int)) << endl;
	cout << "sizeof(long int) = " << (sizeof(long int)) << endl;
	cout << "sizeof(long double) = " << (sizeof(long double)) << endl;
	cout << endl;
	cout << "sizeof(logscale) = " << (sizeof(logscale)) << endl;
	cout << "sizeof(logscale*) = " << (sizeof(logscale*)) << endl;
	cout << "sizeof(COMPARE_TYPE) = " << (sizeof(COMPARE_TYPE)) << endl;
	cout << "All done.  " << endl;

	return 0;

}






