/*


	CPD is an abstract superclass.

	All CPD classes can:

		1) train on data.  given values for other random variables
			(the parents in a BayesNet), a probability distribution
			is built over the possible values in the child variable.
			Training data is given in the form of a TrainingData pointer,
			which maps (example, variable ID, possible value) to a 
			probability that the variable has the value in the example.

		2) classify an example.  Given the values of parent variables,
			create a probability distribution over the child.
			Parent variable values are given in the form of a ExampleData
			pointer, which maps (variable, value) to a probability
			that the variable has the value.

		3) Be converted into a table (CPT).  Properly, CPDs should
			be able to sum out variables, multiply with other CPDs,
			etc., but CPTs have all these functions, so converting
			to a table is one way to do this.

		4) Get printed.  Optionally, various functions can be
			overridden to display the CPD:
				ascii(ostream&):  print text representing the CPD
				dot(ostream&):  print "dot" markup (good for graphical CPDs)


*/

#ifndef CPD_H
#define CPD_H 1

#include "probability.h"	// Variable

class CPT;	// defined in CPT.h

/** Training Data maps a (variable, example, value) to a probability
	that that variable has that value in that example
	
	It also knows how many examples there are.
 */
class TrainingData { 
  public:
  	const int N;	// number of examples
	TrainingData(int n) : N(n) { }
  	virtual double operator()(int ID, int example, int value) = 0;
};

/** ExampleData maps a (variable, value) to a probability that 
	that variable has that value
 */
class ExampleData {
  public:
    virtual double operator()(int ID, int value) = 0;
};

/** Conditional Probablity Distribution */
class CPD {

  protected:
  	vector<const Variable*> variables;		// pointers to each of the variables
											//	(in order)
	
  public:

	/** Get a list of variables actually used in this CPD */
	virtual vector<const Variable*> get_variables() const { return variables; }	// default<=>copy of vector

	/** Similar question, does a CPD include a given variable */
	virtual bool has(int ID) const { 
		vector<const Variable*> vars=get_variables();
		for (vector<const Variable*>::const_iterator i=vars.begin(); i!=vars.end(); i++) { 
			if (ID == (*i)->ID) { return true; }
		}
		return false;
	}

  	/** convert CPD to a table */
  	virtual CPT* as_table() const = 0;

	/** train the CPD based on a set of evidence */
	virtual void train(TrainingData *evidence) = 0;

	/** calcualte a probability distribution over possible values
		for the node, given information about other nodes */
	virtual distribution classify(ExampleData *data) const = 0;

	/** Normalize the distribution (over the last variable in the variables vector) */
	virtual void normalize() { return; }	// default:  assume CPD is already normalized.

	/** for display purposes */
	virtual void ascii(ostream &out) const { out<<"CPD"; }
	virtual void dot(ostream &out) const { return; }

};

#endif
