/*
	CPT is a subclass of CPD (see CPD.h)

	a CPT is a multidimensional array of doubles, stored in row-major-order.
	(That is, each variable is a dimension of the array, and the size
	is the number of possible values for that variable).

	It inheirits a vector<Variable> from CPD (called "variables").
	The ordering of these variables is the row-major-ordering. 

	When it matters, by convention, the LAST variable in the the
	"variables" vector is considered the CHILD variable (that is,
	the child variable in a Bayesian network; the other variables
	are parents).  normalize() normalizes the probability distribution
	over this variable and train() sets the probability distribution
	over this variable given data about the other variables' values.
	
*/


#ifndef CPT_H
#define CPT_H 1

#include <vector>
#include <string>
#include <sstream>
#include <iostream>
using namespace std;

#include "probability.h"
#include "CPD.h"

class CPT;

class CPT : public CPD {
	// inhierits vector<Variable> variables;

  private:
	vector<int> offset;	// offset in the CPT array to find it's data (parallel to variables)

	double *data;
	unsigned long size;		// size of data

	void allocate_and_set_offset();		// allocate data and set offset (and "size")
	                                    //	array assuming variables
	                                    //	are in place

  public:

	/** Create a CPT based on the array of Variable POINTERS 
	 *	By convention, the LAST is the variable to which the CPT belongs
	 *	(although it may not matter)
	 */
	CPT(vector<const Variable*>);		
	
	/** Create a CPT with V variables.  By convention, the LAST is the 
		variable to which the CPT belongs (although it may not matter) */
  	template <typename _Iterator>
  	CPT(_Iterator, _Iterator);

	~CPT() { delete[] data; }

	virtual CPT* as_table() const;

	/** reset all values based on training data */
	virtual void train(TrainingData *evidence);

	/** get a prob. dist. for the last variable given the others */
	virtual distribution classify(ExampleData *data) const;

	/** sum out one of the values and return the result */
	CPT* marginalize(int ID) const;

	/** multiply two CPTs together and return the result */
	CPT* multiply(const CPT*);

	/** multiply the values for a variable by a vector over values */
	template <typename _Random_Access_Iterator>
	void multiply(int ID, _Random_Access_Iterator);

	/** determine if CPT has a certain variable */
	virtual bool has(int ID) const;

	/** normalize the last variable */
	virtual void normalize();

	/** Access an entry in the table by giving a list of coordinates
		in the same order as the CPT's variables array */
	template <typename _Iterator>
	double& entry(_Iterator) const;

	/* Find the index in the CPT multidimensional array cooresponding
		to a list of coordinates
		in the same order as the CPT's variables array */
	template <typename _Iterator>
	int index(_Iterator);

	/** Translate an index into the multidimensional array into
		a list of coordinates */
	vector<int> coordinates(int) const;

	/** Pretty print (well, not-too-bad-looking print) */
	ostream& print(ostream &out) const;
	ostream& operator<<(ostream &out) const { return print(out); }
	friend ostream& operator<<(ostream &out, const CPT &cpt);
	void ascii(ostream &out) const { print(out); }
	void dot(ostream&) const;

	double *begin() const { return data; }
	double *end() const { return data+size; }

	/** ask the table how much data entries total */
	inline unsigned long total_size() const { return size; }

	/** ask the table for an offset for a specific variable (-1 if not found) */
	int var_offset(int ID) const;
		
};	// CPT

template <typename _Iterator>
CPT::CPT(_Iterator begin, _Iterator end) {
	
	for (_Iterator v=begin; v!=end; v++) { 
		variables.push_back(&(*v));	// store the ADDRESS of the variable here
	}
	allocate_and_set_offset();
	
}

/* Access an entry in the table by giving a list of coordinates
	in the same order as the CPT's variables array */
template <typename _Iterator>
double& CPT::entry(_Iterator coordinates) const {
	unsigned long index = 0;
	for (vector<int>::const_iterator off=offset.begin(); off!=offset.end(); off++) { 
		index += (*off * *(coordinates++));
			// computing <coordinates> DOT <variable-offsets>
	}
	return data[index];
}



/* Find the index in the CPT multidimensional array cooresponding
	to a list of coordinates
	in the same order as the CPT's variables array */
template <typename _Iterator>
int CPT::index(_Iterator coordinates) {
	int ans = 0;
	for (vector<int>::iterator off=offset.begin(); off!=offset.end(); off++) { 
		ans += (*off * *(coordinates++));
			// computing <coordinates> DOT <variable-offsets>
	}
	return ans;
}
	
template <typename _Random_Access_Iterator>
void CPT::multiply(int ID, _Random_Access_Iterator value_distribution) { 
	// mult. distribution through for each entry that has
	//	that value for that variable

	vector<const Variable*>::iterator v = variables.begin();
	vector<int>::iterator off = offset.begin();
	while ((*v)->ID != ID && v!=variables.end()) { 
		v++;
		off++;
	}

	// necessary?
	if (v == variables.end()) {
		cerr << "CPT::multiply (ID=" << ID
		     << ", Iterator) variable ID out-of-bounds." << endl;
	}
	
	for (unsigned long p=0; p<size; p++) { 
		
		unsigned long value = (p / *off) % (*v)->arity();
		data[p] *= *(value_distribution + value);

	}
	return;
}
		




#endif
