#include "OptionParser.h"
#include <string.h>



void Option::init() { 
	// set value to default
	processed = false;
	switch (type) {
		case(OPTION_INT):  *(int_ptr) = int_defv;    break;
		case(OPTION_DOUBLE):  *(double_ptr) = double_defv;    break;
		case(OPTION_STRING):  *(string_ptr) = string_defv;    break;
		case(OPTION_BOOL):  *(bool_ptr) = bool_defv;        break;
	}
}

string Option::tostring() const { 
	string ans = "-x|--" + longname;
	ans[1] = shortname;    // reset correct single-char identifier
	switch(this->type) {
		case(OPTION_INT): ans+=" INT"; break;
		case(OPTION_DOUBLE): ans+=" DOUBLE"; break;
		case(OPTION_STRING): ans+=" STRING"; break;
	}
	if (!this->required) { ans = "[" + ans + "]"; }
	return ans;
}

	
vector<string> OptionParser::parse(int argc, const char **argv, string rest) {

	char **non_const_argv = new char*[argc];
	for (int i=0; i<argc; i++) { 
		string arg = argv[i];
		non_const_argv[i] = new char[arg.length()+1];
		strcpy(non_const_argv[i], arg.c_str());
	}
	return parse(argc, non_const_argv, rest);
}


vector<string> OptionParser::parse(int argc, char **argv, string rest) {

	this->rest = rest;	// store for subsequent calls to 'help' or 'usage'
	this->program = argv[0];	// store for same reason

    // iterators for this function
    vector<string>::iterator given_itr;
    vector<Option>::iterator option_itr;
    
    vector<string> given_args;  // all arguments, after parameters 
	                            // are removed, this will be the
								// return value; all the rest of the arg's

    bool error_condition = false;
    const string program = argv[0];

    // copy arguments to a vector of strings
    for (int i=1; i<argc; i++) { given_args.push_back(argv[i]); }

    // parse each argument:
    for (given_itr = given_args.begin(); given_itr != given_args.end();) {
    
        bool given_processed = false;	// if this stays false, arg is not an option
										// and is left in the list (given_args) which
										//	becomes the list of command line args.

        string given = *given_itr;    // shorthand

        // check for each argument, skip first two (reserved), 
		//	remove given arg(s) if necessary
        for (option_itr = arguments.begin(); 
		     option_itr != arguments.end(); 
			 option_itr++) {

            if (option_itr->processed) { continue; }

			if (   given[0] == '-' 
			    && given[1] == option_itr->shortname
			    && given.length() > 2
				&& option_itr->type == Option::OPTION_BOOL) {

				// special case:  Boolean option is given as part of
				//	a list of booleans (e.g. -abcd where a,b,c are bool's)
				*(option_itr->bool_ptr) = !(option_itr->bool_defv);	// 2004-10-18 incl. "default=true"
				                                                    // boolean options
				*given_itr = "-" + given_itr->substr(2);	// remove boolean shortname
				given_processed = true;	// mark as processed by don't remove
				break;	// jump out of options loop
			}

            string shortname = "-x"; shortname[1] = option_itr->shortname;
            string longname = "--" + option_itr->longname;

            if (given == shortname ||
                toupper(given) == toupper(longname)) { 
                
                // argument is given, value is next

                vector<string>::iterator next_given_itr = given_itr + 1;

                if (option_itr->type != Option::OPTION_BOOL &&
                    next_given_itr == given_args.end()) { 

                    //  following argument required, not given
                    cerr << endl << "Error:  expected value after "
					     << given << endl;
                    error_condition = true;
                    
                } else if (option_itr->type == Option::OPTION_BOOL) {

                    // boolean argument set to true
                    *(option_itr->bool_ptr) = !(option_itr->bool_defv);	// 2004-10-18 incl. 
					                                                    // "default-true" boolean options
                    given_itr = given_args.erase(given_itr);
                    given_processed = option_itr->processed = true;
                    
                } else {

                    option_itr->set(*next_given_itr);
                    given_itr = given_args.erase(given_itr);
                    given_itr = given_args.erase(given_itr);
                    given_processed = option_itr->processed = true;

                }
                
            } else if ( given.substr(0, shortname.length()) == shortname) { 

                // arg starts with -s, legal args are -sXXX or -s=XXX

                string rest_of_arg = given.substr(shortname.length());
                if (rest_of_arg[0] == '=') {
					rest_of_arg = rest_of_arg.substr(1); 
				}
                option_itr->set(rest_of_arg);
                given_itr = given_args.erase(given_itr);
                given_processed = option_itr->processed = true;
            
            
            } else if (    toupper(given.substr(0,longname.length()+1))
			            == toupper(longname+"=")   ) { 

                // arg starts with --sss=, get XXX part of --sss=XXX

                string rest_of_arg = given.substr(longname.length()+1);
                option_itr->set(rest_of_arg);
                given_itr = given_args.erase(given_itr);
                given_processed = option_itr->processed = true;

            }

            if (option_itr->processed) { break; } // stop looking at 
			                                       // parameter list because
                                                   // this arg was processed

        } // next parameter

        // if the given argument was processed, given_itr has already been
        //    updated.  otherwise, increment iterator now.
        if (!given_processed) { given_itr++; }

    } // next given argument



    // post-processing:  check for help and/or usage
    if (reserved_help_param) { 
        help(cerr, program, arguments, rest, this->synopsis);
        exit(0);
    }

    if (reserved_usage_param) { 
        usage(cerr, program, arguments, rest);
        exit(0);
    }


    // post-processing:  check that all required arguments are given:
    for (option_itr = arguments.begin(); 
	     option_itr != arguments.end(); 
		 option_itr++) {
    
        if (option_itr->required && !option_itr->processed) { 
            cerr << endl << "Error:  required argument "
                 << option_itr->tostring() << " not given." << endl;
            error_condition = true;
        }
    }

    if (error_condition) { 
        cerr << endl;
        //help(cerr, program, arguments, rest, this->synopsis);		// print help 
        cerr << program << " --" << arguments[RESERVED_HELP].longname 
             << " for help." << endl << endl; 
        exit(-1);
    }


    return given_args;

} // parse
                    
                    
void OptionParser::usage(ostream &out, const string &program, 
                  const vector<Option> &arguments, const string &rest, bool abbrev) {

	// abbrev:  just print "options" instead of listing them all

	const bool INCL_RESERVED_OPTIONS = false;	// should I list [-?|--help]
												// and such 
												// among the options?

	

	const int ARG1 = (INCL_RESERVED_OPTIONS) ? 0 : NUM_RESERVED;

    out << "Usage:  " << program << " ";
	if (abbrev) {
		out << "[options] ";
	} else {
	    for (int a=ARG1; a<arguments.size(); a++) { 
	        out << (arguments[a].tostring()) << " ";
	    }
	}
    out << rest << endl;
    return;

} // usage

void OptionParser::help(ostream &out, const string &program, 
                 const vector<Option> &arguments, const string &rest, 
                 string synopsis) { 

	const int JWIDTH = 80;

    out << endl
        << justify("Synopsis:  " + synopsis, JWIDTH)
        << endl;
    
    usage(out, program, arguments, rest, true);
    out << endl;

    // want this pretty...
    
    // first create array of arguments
    vector<string> arghelp;
    int max = strlen("  [-?|--help]  ");
    for (int a=0; a<arguments.size(); a++) { 
    
        string helpstr = "  " + arguments[a].tostring() + "  ";
        if (helpstr.length() > max) { max = helpstr.length(); }
        arghelp.push_back(helpstr);
    }

    // pad the rest of the arguments
    for (int a=0; a<arguments.size(); a++) { 
        int padding_needed = max-arghelp[a].length();
        for (int i=0; i<padding_needed; i++) { arghelp[a] += " "; }
    }

    // print each with description
	out << "options:" << endl << endl;
    for (int a=0; a<arguments.size(); a++) { 
    
        out << arghelp[a]
            << justify( arguments[a].description, JWIDTH-max, max, 0, false );
        ostringstream oss;
        if (arguments[a].type != Option::OPTION_BOOL) {	
			// (bool are never required, option toggles default value)

            oss << " (";
            if (arguments[a].required) {
                
                oss << "Required.";

            } else {

                oss << "Optional; default=";

                switch (arguments[a].type) { 
                    case (Option::OPTION_INT):  
						oss << arguments[a].int_defv; break;
                    case (Option::OPTION_DOUBLE): 
						oss << arguments[a].double_defv; break;
                    case (Option::OPTION_STRING):  
						oss << arguments[a].string_defv; break;
                }
            }
            oss << ")";
            out << justify( oss.str(), JWIDTH-max, max, 0, true );
            
        }    
        out << endl;    // separate each parameter

    }

    return;

} // help



void Option::set(string value) { 

    switch (this->type) { 
        case(OPTION_BOOL):
            break;
        case(OPTION_INT):
            *(this->int_ptr) = atoi(value.c_str());
            break;
        case(OPTION_DOUBLE):
            *(this->double_ptr) = atof(value.c_str());
            break;
        case(OPTION_STRING):
            *(this->string_ptr) = value;    // copy
            break;
        default:
            cerr << "OptionParser.h bug:  Option object has illegal type (" 
			     << this->type << ")\n";
            exit(-1);
    }
    return;
}
            

string OptionParser::justify(const string &input, int width, int lpad, int rpad,
                         bool padfirstline) { 

    istringstream iss(input);
    string ans = "";
    string line = "";

    string slpad = "";
    for (int i=0; i<lpad; i++) { slpad += " "; }
    string srpad = "";
    for (int i=0; i<rpad; i++) { srpad += " "; }

    int lines = 0;

    while (iss.good()) { 

        string token;

        iss >> token;

        if (line.length() + token.length() > width) { 
        
            if (lines || padfirstline) {
                ans += slpad + line + srpad + "\n"; // normal case
            } else {
                ans += line + "\n";    // first-line unpadded case
            }
            line = token + " ";
            lines++;
        } else { 
            line += token + " ";
        }
    }

    if (lines || padfirstline) {
        ans += slpad + line + srpad + "\n"; // normal case
    } else {
        ans += line + "\n";    // first-line unpadded case
    }
    

    return ans;
}

string OptionParser::toupper(const string &s) {
    
    string ans = s;
    for (int i=0; i<s.length(); i++) { 
        if ('a' <= s[i] && s[i] <= 'z') { 
            ans[i] = s[i] + 'A' - 'a';
        }
    }
    return ans;
}
    
