/*

	Specialized class:  a functor that takes a GaussianMixture
	and makes sure it conforms to the constraints specified in the
	GRN project.

	C++ - wise, this is a function object that takes a GaussianMixture
	object as an argument and modifies it so that it conforms to the
	constraints described below.

	Constraints:

		1) Each Gaussian must have a minimum WEIGHT and VARIANCE

			This is to avoid overfitting.  
			
			Weight requirement:  A 3-Gaussian mixture 
			with one Gaussian weighted zero (or arbitrarilly small)
			is really a degenarate case of a 2-Gaussian mixture.
			We plan on testing 2-Gaussian mixtures anyway, so there's
			no point in considering them again.  We force each Gaussian
			to have non-negligable weight in hopes that a legitimate
			3-Gaussian mixture is found.

			Variance requirement:  A Gaussian that explains 1 data point
			will have zero variance (and be ill-defined).  Part of this
			requirement is to exclude consideration of ill-defined 
			mixtures, but also to avoid overfitting a single data point.

			Implementation:  Weights or Variances below the minimum
			will be reset to the minimum (after renormalization, weights
			will change, but they will still accomplish the goal of
			being non-negligable)

		2) Each Gaussian must have the maximum density within K
			standard distributions of its mean.  This is because we
			wish to consider only mixtures that separate data into
			DISTINCT "states" (expression ranges).  For evaluation
			purposes, It is impossible
			to distinguish between two Gaussians that are too close
			together.

			Implementation:  This is accomplished by keeping the 
			LEFTMOST Gaussian in place and pushing the others to the
			right by just enough to satisfy the constraint.  THEN,
			All Gaussians are shifted back to the LEFT, so that the
			MIDDLE Gaussian is in its original position.  The reason
			for this is because we want to minimize the distance each
			Gaussian is forced to move from its originall mean which
			was determined by an iteration of the EM algorithm
			(trust me, it helps)


*/ 


#ifndef GRN_GAUSSIAN_MIXTURE_CONSTRAINTS_H
#define GRN_GAUSSIAN_MIXTURE_CONSTRAINTS_H

#include "gaussian.h"

#include "math.h"		// definition of INFINITY

#include <functional>
using namespace std;

#ifndef INFINITY
const double INFINITY = -log(0.0);
#endif


struct GRN_GaussianMixtureConstraints;

struct GRN_GaussianMixtureConstraints {

	double K;	// number of std dev's that separate each Gaussian
	double min_stdev;	// minimum allowable values
	double min_weight;

	GRN_GaussianMixtureConstraints(double stdev_separation = 1.0, 
	                           double minimum_stdev = 0.001, 
							   double minimum_weight = 0.0)
		: K(stdev_separation), min_stdev(minimum_stdev), min_weight(minimum_weight) { }

	void operator()(GaussianMixture &gm) const;
	
	
};

#endif
