from sqlalchemy import *
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy.orm import relation, sessionmaker
import datetime

def default_parser(output, row):
    row.block_number = output[0]
    if '+' in output:
        row.size = output[2]
    row.program = output[-1][1:-1]

def remap_parser(output, row):
    row.block_number = output[0]
    row.size = output[2]
    row.orig_block_number = int(output[-1])
    major, minor = output[4][1:-1].split(r',')
    row.orig_major = int(major)
    row.orig_minor = int(minor)

def program_parser(output, row):
    row.program = output[0][1:-1]

output_parsers = {
        'A' : remap_parser,
        'B' : default_parser,
        'C' : default_parser,
        'D' : default_parser,
        'F' : default_parser,
        'G' : default_parser,
        'I' : default_parser,
        'M' : default_parser,
        'P' : program_parser,
        'Q' : default_parser,
        'S' : default_parser,
        'T' : default_parser,
        'U' : program_parser,
        'UT' : program_parser,
}

actions = list('ABCDFGIMPQSTUX')

rwd = list('RWD')

Base = declarative_base()

class Trial(Base):
    __tablename__ = 'trials'

    id = Column(Integer, primary_key=True)
    fs = Column(String, nullable=False)
    slots_millis_maps = Column(BigInteger)
    slots_millis_reduces = Column(BigInteger)
    elapsed_time = Column(BigInteger)  # milliseconds
    heap_usage = Column(BigInteger)    # Total committed heap usage (bytes)
    run_type = Column(String)

    def __init__(self, fs, slots_millis_maps, slots_millis_reduces, elapsed_time, heap_usage, run_type):
        self.fs = fs
        self.slots_millis_maps = slots_millis_maps
        self.slots_millis_reduces = slots_millis_reduces
        self.elapsed_time = elapsed_time
        self.heap_usage = heap_usage
        self.run_type = run_type

    def __str__(self):
        return "<fs: %s, slots_millis_maps: %d, slots_millis_reduces: %d, elapsed_time: %d, heap_usage: %d, run_type: %s>" % \
            (self.fs, self.slots_millis_maps, self.slots_millis_reduces, self.elapsed_time, self.heap_usage, self.run_type)

class Trace(Base):
    __tablename__ = 'traces'

    id = Column(Integer, primary_key=True)
    ts = Column(Float)
    program = Column(String)
    action = Column(String(length=1))
    rwd = Column(String(length=1))
    barrier = Column(Boolean, default=False)
    synchronous = Column(Boolean, default=False)
    remapped = Column(Boolean, default=False)
    merged = Column(Boolean, default=False)
    N = Column(Boolean, default=False)
    block_number = Column(BigInteger)
    size = Column(Integer) # in sectors
    orig_block_number = Column(BigInteger)
    orig_major = Column(Integer)
    orig_minor = Column(Integer)
    major = Column(Integer)
    minor = Column(Integer)
    sequence_number = Column(Integer)
    process_id = Column(Integer)
    trial_num = Column(Integer)
    cpu_id = Column(Integer)
    fs = Column(String)
    run_type = Column(String)

    def __init__(self, data, fs, trial_num, run_type):
        self.major = int(data.device_numbers.split(r',')[0])
        self.minor = int(data.device_numbers.split(r',')[1])
        self.cpu_id = data.cpu_id
        self.sequence_number = data.sequence_number
        self.ts = data.ts
        self.process_id = data.process_id
        self.action = data.action[0]
        self.rwd = data.rwdbs[0]
        self.barrier = 'B' in data.rwdbs
        self.synchronous = 'S' in data.rwdbs
        self.remapped = 'A' in data.rwdbs
        self.merged = 'M' in data.rwdbs
        self.N = 'N' in data.rwdbs
        output_parsers[data.action](data.output, self)
        self.trial_num = trial_num
        self.fs = fs
        self.run_type = run_type

engine = None

def initialize(dbname):
    global engine
    if engine == None:
        engine = create_engine(dbname)
    Base.metadata.create_all(engine)

def get_session(dbname):
    global engine
    if engine == None:
        engine = create_engine(dbname)
    return sessionmaker(bind=engine)()
