import struct

__author__ = 'jpaton'

BLK_TC_READ	= 1 << 0        # reads 
BLK_TC_WRITE	= 1 << 1	# writes
BLK_TC_BARRIER	= 1 << 2	# barrier
BLK_TC_SYNC	= 1 << 3	# sync
BLK_TC_QUEUE	= 1 << 4	# queueing/merging
BLK_TC_REQUEUE	= 1 << 5	# requeueing
BLK_TC_ISSUE	= 1 << 6	# issue
BLK_TC_COMPLETE	= 1 << 7	# completions
BLK_TC_FS	= 1 << 8	# fs requests
BLK_TC_PC	= 1 << 9	# pc requests
BLK_TC_NOTIFY	= 1 << 10	# special message
BLK_TC_AHEAD	= 1 << 11	# readahead
BLK_TC_META	= 1 << 12	# metadata
BLK_TC_DISCARD	= 1 << 13	# discard requests
BLK_TC_DRV_DATA	= 1 << 14	# binary driver data

BLK_TC_END	= 1 << 15	# only 16-bits reminder

__BLK_TA_QUEUE = 1		# queued 
__BLK_TA_BACKMERGE = 2		# back merged to existing rq
__BLK_TA_FRONTMERGE	= 3	# front merge to existing rq
__BLK_TA_GETRQ = 4			# allocated new request
__BLK_TA_SLEEPRQ = 5		# sleeping on rq allocation
__BLK_TA_REQUEUE = 6		# request requeued
__BLK_TA_ISSUE = 7			# sent to driver
__BLK_TA_COMPLETE = 8		# completed by driver
__BLK_TA_PLUG = 9			# queue was plugged
__BLK_TA_UNPLUG_IO = 10		# queue was unplugged by io
__BLK_TA_UNPLUG_TIMER = 11		# queue was unplugged by timer
__BLK_TA_INSERT	= 12	# insert request
__BLK_TA_SPLIT = 13			# bio was split
__BLK_TA_BOUNCE = 14		# bio was bounced
__BLK_TA_REMAP = 15			# bio was remapped
__BLK_TA_ABORT = 16			# request aborted
__BLK_TA_DRV_DATA = 17		# binary driver data

__BLK_TN_PROCESS = 0		# establish pid/name mapping 
__BLK_TN_TIMESTAMP = 1  	# include system clock
__BLK_TN_MESSAGE = 2              # Character string message

BLK_TC_SHIFT		= 16
def BLK_TC_ACT(act):		return ((act) << BLK_TC_SHIFT)

BLK_TA_QUEUE		= (__BLK_TA_QUEUE | BLK_TC_ACT(BLK_TC_QUEUE))
BLK_TA_BACKMERGE	= (__BLK_TA_BACKMERGE | BLK_TC_ACT(BLK_TC_QUEUE))
BLK_TA_FRONTMERGE	= (__BLK_TA_FRONTMERGE | BLK_TC_ACT(BLK_TC_QUEUE))
BLK_TA_GETRQ		= (__BLK_TA_GETRQ | BLK_TC_ACT(BLK_TC_QUEUE))
BLK_TA_SLEEPRQ		= (__BLK_TA_SLEEPRQ | BLK_TC_ACT(BLK_TC_QUEUE))
BLK_TA_REQUEUE		= (__BLK_TA_REQUEUE | BLK_TC_ACT(BLK_TC_REQUEUE))
BLK_TA_ISSUE		= (__BLK_TA_ISSUE | BLK_TC_ACT(BLK_TC_ISSUE))
BLK_TA_COMPLETE		= (__BLK_TA_COMPLETE| BLK_TC_ACT(BLK_TC_COMPLETE))
BLK_TA_PLUG		    = (__BLK_TA_PLUG | BLK_TC_ACT(BLK_TC_QUEUE))
BLK_TA_UNPLUG_IO	= (__BLK_TA_UNPLUG_IO | BLK_TC_ACT(BLK_TC_QUEUE))
BLK_TA_UNPLUG_TIMER	= (__BLK_TA_UNPLUG_TIMER | BLK_TC_ACT(BLK_TC_QUEUE))
BLK_TA_INSERT		= (__BLK_TA_INSERT | BLK_TC_ACT(BLK_TC_QUEUE))
BLK_TA_SPLIT		= (__BLK_TA_SPLIT)
BLK_TA_BOUNCE		= (__BLK_TA_BOUNCE)
BLK_TA_REMAP		= (__BLK_TA_REMAP | BLK_TC_ACT(BLK_TC_QUEUE))
BLK_TA_ABORT		= (__BLK_TA_ABORT | BLK_TC_ACT(BLK_TC_QUEUE))
BLK_TA_DRV_DATA		= (__BLK_TA_DRV_DATA | BLK_TC_ACT(BLK_TC_DRV_DATA))

BLK_TN_PROCESS		= (__BLK_TN_PROCESS | BLK_TC_ACT(BLK_TC_NOTIFY))
BLK_TN_TIMESTAMP	= (__BLK_TN_TIMESTAMP | BLK_TC_ACT(BLK_TC_NOTIFY))
BLK_TN_MESSAGE		= (__BLK_TN_MESSAGE | BLK_TC_ACT(BLK_TC_NOTIFY))

BLK_IO_TRACE_MAGIC	= 0x7746100
BLK_IO_TRACE_VERSION	= 0x65

class Trace(object):
    """
    /*
     * The trace itself
     */
    struct blk_io_trace {
    	__u32 magic;		/* MAGIC << 8 | version */
    	__u32 sequence;		/* event number */
    	__u64 time;		/* in nanoseconds */
    	__u64 sector;		/* disk offset */
    	__u32 bytes;		/* transfer length */
    	__u32 action;		/* what happened */
    	__u32 pid;		/* who did it */
    	__u32 device;		/* device identifier (dev_t) */
    	__u32 cpu;		/* on what cpu did it happen */
    	__u16 error;		/* completion error */
    	__u16 pdu_len;		/* length of data after this trace */
    };
    """

    trace_size = 48     # size in bytes of trace struct
    first_trace = False # True iff first trace has been seen

    def __init__(self, bytestring):
        unpacked = struct.unpack(r'>LLQQLLLLLHH', bytestring)
        self.magic      = unpacked[0]
        self.sequence   = unpacked[1]
        self.time       = float(unpacked[2]) / 1000000000.0
        self.sector     = unpacked[3]
        self.bytes      = unpacked[4]
        self.action     = unpacked[5]
        self.pid        = unpacked[6]
        self.device     = unpacked[7]
        self.cpu        = unpacked[8]
        self.error      = unpacked[9]
        self.pdu_len    = unpacked[10]
        self.extra_data = None

#        print '%x' % (self.magic & 0xffffff00)
        if not Trace.first_trace and self.magic & 0xffffff00 != BLK_IO_TRACE_MAGIC:
            raise Exception('wrong magic')
        Trace.first_trace = True

    def __cmp__(self, other):
        return self.time - other.time

    def __str__(self):
        return '<sequence: %d, time: %d, sector: %d, bytes: %d, action: %d, pid: %d>' % \
            (self.sequence, self.time, self.sector, self.bytes, self.action, self.pid)

    @property
    def verified(self):
#        print '%x' % self.magic
        if self.magic & 0xffffff00 != BLK_IO_TRACE_MAGIC:
            return True
        if self.magic & 0xff != BLK_IO_TRACE_VERSION:
            return True
        return False