/* Adaptive program which calls specWl and doWl with a variety of values
*/

/* possible compilation flags:
    RUNREMOTE: run remotely instead of on the client
    ITERPERRUN_MAX: max number of iterations on the same spec (except seed)
    ITERPERRUN_MIN: min number of iterations on the same spec (except seed)
    SPRITE: denotes running on Sprite
    SUN_OS: denotes running on SUN_OS
*/

/* in this version, kneeSpec doesn't refer to the knee point, but rather
    the point at which parameters are fixed while I'm exploring other
    parameters.  The actual knee is printed out for informational
    value only */

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include "adaptWl.h"
/* #include <bstring.h> */
#include <string.h>
#include <math.h>
#include "stat.h"

/* #define numExplore 10 */ /* how many points to explore along each axis */
/* #define numExplore 50 */
#define FLATCRITERION .1 /* fraction difference between min and max
			    < FLATCRITERION  ==> a flat curve */
			    /* or, the difference between 0 and max to be a
				flat curve */

#ifdef SPRITE
#define ITERPERRUN_MAX 25
#define ITERPERRUN_MIN 2
#else
#define ITERPERRUN_MAX 25
#define ITERPERRUN_MIN 2
#endif

#define MAXITERPEREXPLORE 100
#define KNEEFRACTION (3.0/4.0)
#define MIN_FILESIZE_OVER_SIZE 5.0 /* how large the ratio of fileSize to
					sizeMean must be */
/* these are definitions for exploreParam */
#define FULL 0
#define INTERPOLATE 1
#define BINARY 2

#define REFINE_KNEE_ITER 4 /* 3 worked pretty well--explore10 */

#define MAXPLATEAUNUM 50 /* maximum number of plateaus in uniqueBytes */

main(argc,argv)
    int argc;
    char *argv[];
{
    int i;
    char tmpString[1000];

    WLPARAM minSpec, maxSpec, kneeSpec;

    /* WLPERF perfResult;
    WLPARAM paramResult; */
    int whichParam;

    WLRESULT *result;
    int resultNum;
    WLRESULT *compareResult;
    int compareResultNum;
    int maxRunTime; /* max running time of EACH RUN in seconds */
    char *fileServer;
    char *readResultFile; /* a debugging flag which causes the program to read
			    in the results from another adaptWl run (after the
			    line of =========) instead of doing the runs over
			    again */
    char *compareFile;
    int noNarrow; /* flag to say whether or not adaptWl should attempt to
			narrow the range of the parameters */
    int refineIter; /* how many refinements of the knee point */
    int randomExploreNum; /* do a random exploration of the parameter space
			    randomExploreNum = the number of points to take */
    int orthogonalExploreNum; /* do a full orthogonal exploration of the
				    parameter space
			    orthogonalExploreNum = number of points for each
				    dimension*/
    int kneeExploreNum; /* do a random exploration around the knee point */
    int useAllResultsFlag;
	    /* should I immediately use all results from the previous run? */
    int minRunTime;
    int warmStartFlag;
    int findPlateau;
    int flushPossible;
    int numExplore;
    char *focalUnique; /* comma separated list of focal points for
			    uniqueBytes */

    int refineFlag;
    int plateauNum;
    double plateauMin[MAXPLATEAUNUM];
    double plateauMax[MAXPLATEAUNUM];
    double plateauFocal[MAXPLATEAUNUM];
    char *pos;

    void getParamAdaptWl();
    char *convertParamStr();
    double convertParamVal();
    double describeKnee();
    int describePlateau();
    double exploreParam();
    void *malloc();
    double round();
    double run();

    for (i=0; i<argc; i++) {
	printf("%s ", argv[i]);
    }
    printf("\n");

    /* set up initial variable, mallocs */
    resultNum = 0;
    result = (WLRESULT *) calloc(MAXRESULTNUM, sizeof(WLRESULT));
    compareResult = (WLRESULT *) calloc(MAXRESULTNUM, sizeof(WLRESULT));
    /* note that I depend on the result array being zero when it starts
	up.  This is to make sure I don't assume any of the result buffer
	is valid when I'm trying to continue a run (see routine run() ) */

    refineIter = REFINE_KNEE_ITER;

    getParamAdaptWl(argc, argv, &minSpec, &maxSpec, &kneeSpec, &maxRunTime,
	    &fileServer, &readResultFile, &noNarrow, &refineIter,
	    &randomExploreNum, &orthogonalExploreNum, &kneeExploreNum,
	    &compareFile, &useAllResultsFlag, &minRunTime, &warmStartFlag,
	    &findPlateau, &flushPossible, &focalUnique, &numExplore);
    printf("numExplore=%d\n", numExplore);
    printf("readResultFile=%s\n", readResultFile);
    printf("noNarrow=%d\n", noNarrow);
    printf("refineIter=%d\n", refineIter);
    printf("findPlateau=%d\n", findPlateau);
    printf("flushPossible=%d\n", flushPossible);
    printf("randomExploreNum = %d, orthogonalExploreNum = %d, kneeExploreNum=%d\n",
	randomExploreNum, orthogonalExploreNum, kneeExploreNum);
    printf("useAllResultsFlag = %d\n", useAllResultsFlag);
    if (warmStartFlag) {
	printf("warm starting, min run time is %d seconds\n", minRunTime);
    }

    if (readResultFile[0] != '\0') {
	readResultAll(result, &resultNum, readResultFile, useAllResultsFlag);
	printf("read %d results from a previous run\n", resultNum);
	if (!useAllResultsFlag) {
	    resultNum = 0; /* this will get incremented as we go (in run() ) */
	}
    }
    if (compareFile[0] != '\0') {
	readResultAll(compareResult, &compareResultNum, compareFile, 0);
	printf("read %d results to compare against\n", compareResultNum);
    }
    if (focalUnique[0] != '\0') {
	printf("manually specified focal points %s\n", focalUnique);
    }
    printf("starting parameter ranges\n");
    printWlParamRange(&minSpec, &maxSpec, &kneeSpec, maxRunTime);

#ifdef SPRITE
    printf("---- file cache statistics ----\n");
    printf("---- client ----\n");
    sprintf(tmpString, "fsstat | head");
    mySystem(tmpString);
    if (fileServer[0] != '\0') {
	printf("---- server ----\n");
	sprintf(tmpString, "rsh %s 'fsstat | head'", fileServer);
	mySystem(tmpString);
    }
    printf("---- end of file cache statistics ----\n");
    mySystem("migDisallow");
#endif /* SPRITE */

#ifdef SUN_OS
    mySystem("loginDisallow");
#endif /* SUN_OS */

    /* narrow the ranges of min and max, and find the starting knee */
    if (!noNarrow) {
	narrowRange(&minSpec, &maxSpec, &kneeSpec, result, &resultNum,
	    maxRunTime, fileServer, warmStartFlag, minRunTime, flushPossible,
	    numExplore);
    }
    printf("final parameter ranges\n");
    printWlParamRange(&minSpec, &maxSpec, &kneeSpec, maxRunTime);

    if ( (randomExploreNum>0) + (kneeExploreNum>0) + (orthogonalExploreNum>0) +
	    (compareFile[0] != '\0') > 1) {
	printf("more than one of randomExploreNum, kneeExploreNum, orthogonalExploreNum, compareFile[0] is positive\n");
	exit(1);
    }
	

    if (randomExploreNum > 0) {
	printf("srandom with seed %d\n", kneeSpec.seed);
	mySrandom(kneeSpec.seed);
	randomExplore(&minSpec, &maxSpec, result, &resultNum, maxRunTime,
			fileServer, randomExploreNum, &kneeSpec, warmStartFlag,
			minRunTime, flushPossible);
    }

    if (kneeExploreNum > 0) {
	kneeExplore(&minSpec, &maxSpec, result, &resultNum, maxRunTime,
			fileServer, kneeExploreNum, &kneeSpec, warmStartFlag,
			minRunTime, flushPossible);
    }

    if (orthogonalExploreNum > 0) {
	printf("kneeSpec.dirName = %s\n", kneeSpec.dirName);
	orthogonalExplore(&minSpec, &maxSpec, result, &resultNum, maxRunTime,
			fileServer, orthogonalExploreNum, 0, &kneeSpec,
			warmStartFlag, minRunTime, flushPossible);
    }

    if (compareFile[0] != '\0') {
	/* duplicate results from compareFile */
	duplicateResults(result, &resultNum, maxRunTime, fileServer,
	    compareResult, compareResultNum, &kneeSpec, warmStartFlag,
	    minRunTime, flushPossible);
    }

    if (findPlateau && orthogonalExploreNum<=0 && compareFile[0]=='\0' &&
	    randomExploreNum<=0 && kneeExploreNum<=0) {
	/* find plateaus in uniqueBytes, graph family at each plateau,
	    don't refine knee point at all (except uniqueBytes) */
	printf("finding plateaus in uniqueBytes\n");
	whichParam = UNIQUEBYTES;
	exploreParam(whichParam, &kneeSpec, &minSpec, &maxSpec, result,
	    &resultNum, maxRunTime, fileServer, KNEEFRACTION, 1, FULL,
	    numExplore*2, warmStartFlag, minRunTime, flushPossible);
	
	if (focalUnique[0] != '\0') {
	    /* manually specified focal points for uniqueBytes */
	    for (plateauNum = 0; focalUnique[0] != '\0'; plateauNum++) {
		sscanf(focalUnique, "%lf", plateauFocal+plateauNum);
		printf("manually specified plateau %lf KB\n",
		    plateauFocal[plateauNum]);
		if ( (pos = strchr(focalUnique, ',') ) !=NULL) {
		    focalUnique = pos+1;
		} else {
		    focalUnique[0] = '\0';
		}
	    }
	} else {
	    plateauNum = describePlateau(&kneeSpec, &minSpec, &maxSpec, result,
			    resultNum, plateauMin, plateauMax);
	    printf("%d plateaus returned\n", plateauNum);
	    for (i=0; i<plateauNum; i++) {
		printf("%d: %lf KB - %lf KB\n", i, plateauMin[i],
			plateauMax[i]);
		plateauFocal[i] = (plateauMin[i]+plateauMax[i])/2.0;
	    }
	}
	printf("plateauNum = %d\n", plateauNum);
	for (i=0; i<plateauNum; i++) {
	    assignParamVal(&kneeSpec, UNIQUEBYTES, plateauFocal[i]);
	    printf("exploring plateau %d: with uniqueBytes %lf KB\n",
		i, convertParamVal(&kneeSpec, UNIQUEBYTES));
	    graphFamily(0, &kneeSpec, &minSpec, &maxSpec,
		    result, &resultNum, maxRunTime, fileServer, KNEEFRACTION,
		    warmStartFlag, minRunTime,
		    EXPLORING_PARAM & ~UNIQUEBYTES, flushPossible, numExplore);
	    printf("\n--- focal point #%d, uniqueBytes = %lf KB ---\n", i,
		convertParamVal(&kneeSpec, UNIQUEBYTES));
	    printWlParam(&kneeSpec);
	    printf("\n");

	    for (whichParam=1; whichParam<=MAXWLPARAM; whichParam*=2) {
		if (whichParam & EXPLORING_PARAM) {
		    plotParam(whichParam, &kneeSpec, &minSpec, &maxSpec, result,
			resultNum, convertParamVal(&kneeSpec, whichParam));
		}
	    }
	}
    } else if (orthogonalExploreNum<=0 && compareFile[0]=='\0' &&
	    randomExploreNum<=0 && kneeExploreNum<=0) {
	/* do iterative adaptation of knee point */
	for (i=0; i<refineIter; i++) {
	    if (i!=refineIter-1) {
		refineFlag = 1;
	    } else {
		/* last time through, don't refine */
		refineFlag = 0;
	    }
	    graphFamily(refineFlag, &kneeSpec, &minSpec, &maxSpec,
		    result, &resultNum, maxRunTime, fileServer, KNEEFRACTION,
		    warmStartFlag, minRunTime, EXPLORING_PARAM, flushPossible,
		    numExplore);
	}
	if (refineIter > 0) {
	    printf("\n--- final knee workload ---\n");
	    printWlParam(&kneeSpec);
	    printf("\n");

	    for (whichParam=1; whichParam<=MAXWLPARAM; whichParam*=2) {
		if (whichParam & EXPLORING_PARAM) {
		    plotParam(whichParam, &kneeSpec, &minSpec, &maxSpec, result,
			resultNum, convertParamVal(&kneeSpec, whichParam));
		}
	    }
	}
    } /* end !findPlateau */

    printResultAll(result, resultNum);

#ifdef SPRITE
    mySystem("migAllow");
#endif

#ifdef SUN_OS
    mySystem("loginAllow");
#endif /* SUN_OS */
    free((char *) result);
    free((char *) compareResult);
    exit(0);
}

int describePlateau(kneeSpecPtr, minSpecPtr, maxSpecPtr, result, resultNum,
	plateauMin, plateauMax)
    WLPARAM *kneeSpecPtr;
    WLPARAM *minSpecPtr;
    WLPARAM *maxSpecPtr;
    WLRESULT result[MAXRESULTNUM];
    int resultNum;
    double plateauMin[MAXPLATEAUNUM];
    double plateauMax[MAXPLATEAUNUM];
{
    WLSIMPLERESULT simpleResult[MAXSIMPLERESULTNUM];
    int simpleResultNum;
    int diskI;
    double avSlope;
    int whichParam;
    int i;
    int startPlateau, endPlateau;
    int plateauNum;

    int condense();
    double computeSlope();
    char *convertParamStr();

    whichParam = UNIQUEBYTES;
    /* look at uniqueBytes graph and figure out plateaus */
    simpleResultNum = condense(whichParam, kneeSpecPtr, minSpecPtr,
	maxSpecPtr, result, resultNum, simpleResult);

    /* print out the results */
    for (i=0; i<simpleResultNum; i++) {
	printf("%d: %s = %lf, throughput=%lf MB/s\n", i,
	    convertParamStr(whichParam), simpleResult[i].paramValue,
	    simpleResult[i].performance);
    }

    /* now compute the average slope */
    printf("performance at uniqueBytes %lf = %lf MB/s\n",
	simpleResult[simpleResultNum-1].paramValue,
	simpleResult[simpleResultNum-1].performance);

    for (i=simpleResultNum-2; i>0 &&
	    simpleResult[i].performance/
	    simpleResult[simpleResultNum-1].performance < 2.0 && 
	    (simpleResult[i].performance/
	    simpleResult[simpleResultNum-1].performance < 1.2 ||
	    computeSlope(simpleResult, 0, i+1) > computeSlope(simpleResult,0,i));
	    i--){
    }
    diskI = i+1; /* this is where disk performance starts */
    printf("diskI=%d\n", diskI);

    /* note that slope should be negative */
    avSlope = computeSlope(simpleResult, 0, diskI);
    printf("average slope is %g MB/s per KB\n", avSlope);

    /* let's try a new method, similar to how I find the start of disk
	    performance */
    /* this method looks at if slope is increasing or decreasing to find
	plateaus */
    /* first plateau (with largest uniqueBytes) is defined to be from
	diskI to end */
    endPlateau = simpleResultNum-1;
    startPlateau = simpleResultNum-1; /* needed to start this for loop */
    for (plateauNum=0; startPlateau > 0; plateauNum++) {
	for (; startPlateau>0 && (plateauNum!=0 || startPlateau != (diskI-1)) &&
		(computeSlope(simpleResult, 0, endPlateau) > 0 ||
		simpleResult[startPlateau].performance/
		simpleResult[endPlateau].performance < 1.3 ||
		computeSlope(simpleResult, 0, startPlateau+1) >
		    computeSlope(simpleResult,0,startPlateau));
		startPlateau--){
	}
	if (startPlateau == 0) {
	    startPlateau = -1;
	}
	printf("plateau %d: %lf KB - %lf KB\n", plateauNum,
	    simpleResult[startPlateau+1].paramValue,
	    simpleResult[endPlateau].paramValue);

	if (plateauNum >= MAXPLATEAUNUM) {
	    printf("error: too many plateaus (%d)\n", plateauNum);
	    exit(1);
	}
	plateauMin[plateauNum] = simpleResult[startPlateau+1].paramValue;
	plateauMax[plateauNum] = simpleResult[endPlateau].paramValue;

	/* skip the steep portion */

	/* define a steep portion as any portion that has slope steeper than
	    the average slope */
	for (endPlateau = startPlateau; endPlateau>0 &&
	    computeSlope(simpleResult, endPlateau-1, endPlateau) < avSlope;
	    endPlateau--) {
	}
	printf("endPlateau = %d\n", endPlateau);
	startPlateau = endPlateau;
    }

#ifdef UNDEF
    startPlateau = 0;

    /* start by finding the first stable performance plateau */
    for (startPlateau=0; startPlateau<simpleResultNum-1 &&
	computeSlope(simpleResult, startPlateau, startPlateau+1) < avSlope;
	startPlateau++) {
    }

    /* for (plateauNum = 0; startPlateau < simpleResultNum; plateauNum++) {} */
    for (plateauNum = 0; startPlateau < diskI; plateauNum++) {
	printf("looking at plateau %d, startPlateau = %d\n", plateauNum,
	    startPlateau);
	for (i=startPlateau+1; i<diskI &&
	    computeSlope(simpleResult, startPlateau, i) > (avSlope*2); i++) {
	    printf("slope between points %d and %d = %g\n", startPlateau, i, 
		computeSlope(simpleResult, startPlateau, i));
	}
	printf("** slope between points %d and %d = %g\n", startPlateau, i, 
	    computeSlope(simpleResult, startPlateau, i));
	endPlateau = i-1;
	printf("plateau %d: %lf KB - %lf KB\n", plateauNum,
	    simpleResult[startPlateau].paramValue,
	    simpleResult[endPlateau].paramValue);
	printf("endPlateau=%d\n", endPlateau);
	/* skip steep section */
	for (startPlateau=i; startPlateau<simpleResultNum-1 &&
	    computeSlope(simpleResult, startPlateau, startPlateau+1) < avSlope;
	    startPlateau++) {
	}
	if (startPlateau < diskI) {
	    avSlope = computeSlope(simpleResult, startPlateau, diskI);
	    printf("avSlope is %g MB/s per KB\n", avSlope);
	}
    }
    printf("plateau %d: %lf KB - %lf KB\n", plateauNum,
	simpleResult[diskI].paramValue,
	simpleResult[simpleResultNum-1].paramValue);
#endif
    return(plateauNum);
}

double computeSlope(simpleResult, i, j)
    WLSIMPLERESULT simpleResult[MAXSIMPLERESULTNUM];
{
    /* compute the slope between two points on a simpleResult graph */

    /* check to make sure points are not identical */
    if (i==j) {
	printf("error in computeSlope: %d=%d\n", i,j);
	exit(1);
    } else if (simpleResult[i].paramValue == simpleResult[j].paramValue) {
	printf("error in computeSlope: %lf=%lf\n", simpleResult[i].paramValue,
	    simpleResult[j].paramValue);
	exit(1);
    }
    return( (simpleResult[i].performance - simpleResult[j].performance) /
	    (simpleResult[i].paramValue - simpleResult[j].paramValue));
}

graphFamily(refineFlag, kneeSpecPtr, minSpecPtr, maxSpecPtr, result,
	resultNumPtr, maxRunTime, fileServer, kneeFraction, warmStartFlag,
	minRunTime, exploringParam, flushPossible, numExplore)
    int refineFlag; /* refine the knee point or just graph? */
    WLPARAM *kneeSpecPtr;
    WLPARAM *minSpecPtr;
    WLPARAM *maxSpecPtr;
    WLRESULT result[MAXRESULTNUM];
    int *resultNumPtr;
    int maxRunTime;
    char *fileServer;
    double kneeFraction;
    int warmStartFlag;
    int minRunTime;
    int exploringParam;
    int flushPossible;
    int numExplore;
{
    /* graph a family of curves, possibly assigning a new knee value */

    WLPARAM spec;
    int type;
    int whichParam;
    double paramValue;

    char *convertParamStr();
    double run();
    double round();
    double exploreParam();

    if (!refineFlag) {
	/* last time through */
	/* don't find the knee, just fill out the graph */
	type = FULL;
    } else {
	/* type = FULL; */
	/* type = INTERPOLATE; */
	type = BINARY;
    }

    /* first measure at the knee point */
    copyWlParam(kneeSpecPtr, &spec);
    run(&spec, result, resultNumPtr, maxRunTime, fileServer,
	warmStartFlag, minRunTime, flushPossible);

    for (whichParam=1; whichParam<=MAXWLPARAM; whichParam*=2) {
	if (whichParam & exploringParam) {
	    printf("\n---current knee workload---\n");
	    printWlParam(kneeSpecPtr);
	    printf("highest level call to get the knee for %s\n",
		convertParamStr(whichParam));

	    paramValue = exploreParam(whichParam, kneeSpecPtr, minSpecPtr,
		    maxSpecPtr, result, resultNumPtr, maxRunTime,
		    fileServer, kneeFraction, 1, type,
		    (type==FULL)?numExplore:(int) round(numExplore/2.0),
		    warmStartFlag, minRunTime, flushPossible);
/* ???
*/
	    if (refineFlag) {
		printf("assigning new knee value of %lf to %s\n",
		    paramValue, convertParamStr(whichParam));
		assignParamVal(kneeSpecPtr, whichParam, paramValue);
	    }
	}
    }
}


randomExplore(minSpecPtr, maxSpecPtr, result, resultNumPtr,
	maxRunTime, fileServer, sampleNum, specPtr, warmStartFlag, minRunTime,
	flushPossible)
    WLPARAM *minSpecPtr;
    WLPARAM *maxSpecPtr;
    WLRESULT result[MAXRESULTNUM];
    int *resultNumPtr;
    int maxRunTime;
    char *fileServer;
    int sampleNum;
    WLPARAM *specPtr;
    int warmStartFlag;
    int minRunTime;
    int flushPossible;
{
    /* take a number of random samples from the entire parameter space */

    int i;
    int whichParam;
    double paramValue;

    double convertParamVal();
    char *convertParamStr();
    long myRandom();
    double run();

    for (i=0; i<sampleNum; i++) {
	for (whichParam=1; whichParam<=MAXWLPARAM; whichParam*=2) {
	    paramValue = convertParamVal(minSpecPtr, whichParam) +
		    (myRandom()%1000000) / 1000000.0 * 
			(convertParamVal(maxSpecPtr, whichParam) -
			 convertParamVal(minSpecPtr, whichParam));
	    assignParamVal(specPtr, whichParam, paramValue);
	    /*
	    printf("assigning value %lf to %s\n", paramValue,
		convertParamStr(whichParam));
	    */
	}
	run(specPtr, result, resultNumPtr, maxRunTime, fileServer,
	    warmStartFlag, minRunTime, flushPossible);
    }
}

kneeExplore(minSpecPtr, maxSpecPtr, result, resultNumPtr,
	maxRunTime, fileServer, sampleNum, kneeSpecPtr, warmStartFlag,
	minRunTime, flushPossible)
    WLPARAM *minSpecPtr;
    WLPARAM *maxSpecPtr;
    WLRESULT result[MAXRESULTNUM];
    int *resultNumPtr;
    int maxRunTime;
    char *fileServer;
    int sampleNum;
    WLPARAM *kneeSpecPtr;
    int warmStartFlag;
    int minRunTime;
    int flushPossible;
{
    /* take a number of random samples from the parameter space
	but only allow ONE parameter to differ from the knee point */

    int i;
    int whichParam;
    double paramValue;
    WLPARAM spec;

    double convertParamVal();
    char *convertParamStr();
    long myRandom();
    double run();

    for (i=0; i<sampleNum; i++) {
	for (whichParam=1; whichParam<=MAXWLPARAM; whichParam*=2) {
	    if (whichParam & EXPLORING_PARAM) {
		copyWlParam(kneeSpecPtr, &spec);
		paramValue = convertParamVal(minSpecPtr, whichParam) +
			(myRandom()%1000000) / 1000000.0 * 
			    (convertParamVal(maxSpecPtr, whichParam) -
			     convertParamVal(minSpecPtr, whichParam));
		assignParamVal(&spec, whichParam, paramValue);
		/*
		printf("assigning value %lf to %s\n", paramValue,
		    convertParamStr(whichParam));
		*/
		run(&spec, result, resultNumPtr, maxRunTime, fileServer,
		    warmStartFlag, minRunTime, flushPossible);
	    }
	}
    }
}

duplicateResults(result, resultNumPtr, maxRunTime, fileServer,
	compareResult, compareResultNum, specPtr, warmStartFlag, minRunTime,
	flushPossible)
    WLRESULT result[MAXRESULTNUM];
    int *resultNumPtr;
    int maxRunTime;
    char *fileServer;
    WLRESULT compareResult[MAXRESULTNUM];
    int compareResultNum;
    WLPARAM *specPtr;
    int warmStartFlag;
    int minRunTime;
    int flushPossible;
{
    /* duplicate the results from another result file */

    int i,j;
    int whichParam;

    double convertParamVal();
    long myRandom();
    unsigned int paramDiff();
    double run();

#ifndef REVERSE
    for (i=0; i<compareResultNum; i=j) {
	for (j=i; j<compareResultNum &&
	    paramDiff(&(compareResult[i].target), &(compareResult[j].target),
		NOPARAMETER, 0, 10.0)==0; j++) {
	    for (whichParam=1; whichParam<=MAXWLPARAM; whichParam *= 2) {
		assignParamVal(specPtr, whichParam,
		    convertParamVal(specPtr, whichParam) *
			(j-i) / (double) (j-i+1) +
		    convertParamVal(&(compareResult[j].target), whichParam) /
			(double) (j-i+1) );
	    }
	}
	printf("running to compare against old runs %d-%d\n", i,j-1);
	run(specPtr, result, resultNumPtr, maxRunTime, fileServer,
	    warmStartFlag, minRunTime, flushPossible);
    }
#else /* REVERSE */
    /* duplicate the runs in reverse (just in case there's some run-order
	dependency) */
    for (i=compareResultNum-1; i>=0; i=j) {
	for (j=i; j>=0 &&
	    paramDiff(&(compareResult[i].target), &(compareResult[j].target),
		NOPARAMETER, 0, 10.0)==0; j--) {
	    for (whichParam=1; whichParam<=MAXWLPARAM; whichParam *= 2) {
		assignParamVal(specPtr, whichParam,
		    convertParamVal(specPtr, whichParam) *
			(i-j) / (double) (i-j+1) +
		    convertParamVal(&(compareResult[j].target), whichParam) /
			(double) (i-j+1) );
	    }
	}
	printf("running to compare against old runs %d-%d\n", i,j+1);
	run(specPtr, result, resultNumPtr, maxRunTime, fileServer,
	    warmStartFlag, minRunTime, flushPossible);
    }
#endif /* REVERSE */
}

orthogonalExplore(minSpecPtr, maxSpecPtr, result, resultNumPtr, maxRunTime,
	fileServer, numPerDimension, whichParam, specPtr, warmStartFlag,
	minRunTime, flushPossible)
    WLPARAM *minSpecPtr;
    WLPARAM *maxSpecPtr;
    WLRESULT result[MAXRESULTNUM];
    int *resultNumPtr;
    int maxRunTime;
    char *fileServer;
    int numPerDimension;
    int whichParam;
    WLPARAM *specPtr;
    int warmStartFlag;
    int minRunTime;
    int flushPossible;
{
    /* recursively explore the entire workload space */
    int iter;
    double minParamValue, maxParamValue;
    double paramValue, paramValueOld;
    double increment;
    int multiplyFlag;
    WLPARAM tmpSpec1;

    double convertParamVal();
    double convertParamMinIncr();
    char *convertParamStr();
    double run();

    printf("called orthogonal with whichParam = %d\n", whichParam);

    if (whichParam == 0) {
	whichParam = 1;
    } else {
	whichParam *= 2;
    }

    if (whichParam > MAXWLPARAM) {
	run(specPtr, result, resultNumPtr, maxRunTime, fileServer,
	    warmStartFlag, minRunTime, flushPossible);
	return;
    }

    if (!(whichParam&EXPLORING_PARAM)) {
	orthogonalExplore(minSpecPtr, maxSpecPtr, result, resultNumPtr,
		maxRunTime, fileServer, numPerDimension, whichParam,
		specPtr, warmStartFlag, minRunTime, flushPossible);
	return;
    }

    minParamValue = convertParamVal(minSpecPtr, whichParam);
    maxParamValue = convertParamVal(maxSpecPtr, whichParam);

    if ( (whichParam & LOGPARAM) && minParamValue > 0) {
	increment = maxParamValue/minParamValue;
	increment = exp( log(increment) / (numPerDimension-1) );
	multiplyFlag = 1;
	printf("multiplying %s by %lf each time\n", convertParamStr(whichParam),
	    increment);
    } else {
	increment = MAX(convertParamMinIncr(whichParam),
		(maxParamValue - minParamValue) / (numPerDimension-1));
	multiplyFlag = 0;
	printf("incrementing %s by %lf each time\n",
	    convertParamStr(whichParam), increment);
    }

    iter = 1;
    for (paramValue = minParamValue;
	    paramValue <= maxParamValue && iter<=numPerDimension;
	 paramValue = (iter==numPerDimension)?
	    maxParamValue:
	    (multiplyFlag? (paramValue*increment) : (paramValue+increment)) ) {
	assignParamVal(specPtr, whichParam, paramValue);
	printf("paramValue for %s is %lf\n", convertParamStr(whichParam),
	    convertParamVal(specPtr, whichParam));
	if (iter==1 ||
	    (convertParamVal(specPtr, whichParam) - paramValueOld) >= convertParamMinIncr(whichParam)) {

	    /* (don't run if this parameter value is too close to the last) */

	    orthogonalExplore(minSpecPtr, maxSpecPtr, result, resultNumPtr,
		    maxRunTime, fileServer, numPerDimension, whichParam,
		    specPtr, warmStartFlag, minRunTime, flushPossible);
	    assignParamVal(&tmpSpec1, whichParam, paramValue);
	    paramValueOld = convertParamVal(&tmpSpec1, whichParam);
	} else {
	    printf("not running, since the paramValue is too close to the last one\n");
	}
	iter++;
    }

}

double exploreParam(whichParam, kneeSpecPtr, minSpecPtr, maxSpecPtr, result,
	resultNumPtr, maxRunTime, fileServer, kneeFraction, kneeAbsolute,
	type, numExplore, warmStartFlag, minRunTime, flushPossible)
    int whichParam;
    WLPARAM *kneeSpecPtr;
    WLPARAM *minSpecPtr;
    WLPARAM *maxSpecPtr;
    WLRESULT result[MAXRESULTNUM];
    int *resultNumPtr;
    int maxRunTime;
    char *fileServer;
    double kneeFraction;
    int kneeAbsolute;
    int type; /* when type = FULL, explore the whole parameter space at evenly
		    spaced intervals.  When type = INTERPOLATE, try to find the
		    knee faster */
    int numExplore; /* how many points to take at regular intervals before
			    searching for the knee */
    int warmStartFlag;
    int minRunTime;
    int flushPossible;
{
    /* explore the parameter space, varying whichParam.

	If type=FULL, explore the space at regularly spaced intervals (either
	    incrementing by a set amount, or multiplying by a set amount if
	    it's a logarithmic parameter).
	If type=INTERPOLATE, get the min and max, then look for the
	    knee point using linear approximation
	If type=BINARY (used from narrowRange), get the min and max, then
	    look for the knee point using a binary search

	return the knee value of the whichParam
    */

    WLPARAM spec;
    WLPARAM tmpSpec1, tmpSpec2, tmpSpec3, tmpSpec4;
    double paramValue;
    double paramValueOld;
    double lowerParamValue, upperParamValue;
    double lowerPerf, upperPerf;
    double lowerParamValueOld, upperParamValueOld;
    double minPerf, maxPerf;
    double increment; /* what we increment the paramValue by */
    int positive;
    int iter;
    int multiplyFlag;

    double minParamValue, maxParamValue;

    double convertParamVal();
    double convertParamMinIncr();
    unsigned int paramDiff();
    char *convertParamStr();
    double describeKnee();
    double interpolate();
    double dynamicMinMax();
    double run();

    printf("\n");
    printf("exploring %s with type %d\n", convertParamStr(whichParam), type);

    if (numExplore < 2) {
	printf("error in exploreParam, numExplore=%d, should be >= 2\n",
	    numExplore);
	exit(1);
    }

    minParamValue = convertParamVal(minSpecPtr, whichParam);
    maxParamValue = convertParamVal(maxSpecPtr, whichParam);

    copyWlParam(kneeSpecPtr, &spec);
    copyWlParam(kneeSpecPtr, &tmpSpec1);
    copyWlParam(kneeSpecPtr, &tmpSpec2);
    copyWlParam(kneeSpecPtr, &tmpSpec3);
    copyWlParam(kneeSpecPtr, &tmpSpec4);

    if (whichParam == SIZEMEAN) {
	paramValue = dynamicMinMax(SIZEMEAN, &spec, minSpecPtr, maxSpecPtr, 0);
	if (paramValue < maxParamValue) {
	    printf("note: setting max %s = %lf\n", convertParamStr(whichParam),
		paramValue);
	    maxParamValue = paramValue;
	}
    } else if (whichParam == UNIQUEBYTES) {
	paramValue = dynamicMinMax(UNIQUEBYTES, &spec, minSpecPtr,maxSpecPtr,1);
	if (paramValue > minParamValue) {
	    printf("note: setting min %s = %lf\n", convertParamStr(whichParam),
		paramValue);
	    minParamValue = paramValue;
	}
    } else if (whichParam == PROCESSNUM) {
	paramValue=dynamicMinMax(PROCESSNUM, &spec, minSpecPtr, maxSpecPtr, 0);
	if (paramValue < maxParamValue) {
	    printf("note: setting max %s = %lf\n", convertParamStr(whichParam),
		paramValue);
	    maxParamValue = paramValue;
	}
    }


    if ( (whichParam & LOGPARAM) && minParamValue > 0 && numExplore > 2) {
	increment = maxParamValue/minParamValue;
	increment = exp( log(increment) / ((double) numExplore-1.0) );
	multiplyFlag = 1;
	printf("multiplying parameter by %lf each time\n", increment);
    } else {
	increment = MAX(convertParamMinIncr(whichParam),
		(maxParamValue - minParamValue) / ((double) numExplore-1.0));
	multiplyFlag = 0;
	printf("incrementing parameter by %lf each time\n", increment);
    }

    iter = 1;
    for (paramValue = minParamValue;
	    paramValue <= maxParamValue && iter<=numExplore;
	 paramValue = (iter==numExplore)?
	    maxParamValue:
	    (multiplyFlag? (paramValue*increment) : (paramValue+increment)) ) {
	assignParamVal(&spec, whichParam, paramValue);
	printf("paramValue for %s is %lf\n", convertParamStr(whichParam),
	    convertParamVal(&spec, whichParam));
	if (iter==1 ||
	    (convertParamVal(&spec, whichParam) - paramValueOld) >= convertParamMinIncr(whichParam)) {

	    /* (don't run if this parameter value is too close to the last) */

	    run(&spec, result, resultNumPtr, maxRunTime, fileServer,
		warmStartFlag, minRunTime, flushPossible);
	    assignParamVal(&tmpSpec1, whichParam, paramValue);
	    paramValueOld = convertParamVal(&tmpSpec1, whichParam);
	} else {
	    printf("not running, since the paramValue is too close to the last one\n");
	}
	iter++;
    }

    paramValue = describeKnee(whichParam, result, *resultNumPtr,
	kneeSpecPtr, minSpecPtr, maxSpecPtr, kneeFraction, kneeAbsolute,
	&lowerParamValue, &lowerPerf, &upperParamValue, &upperPerf, &minPerf,
	&maxPerf,&positive);
    printf("knee value for %s is %lf\n", convertParamStr(whichParam),
	paramValue);
    printf("parameter range of knee is [%lf (perf %lf), %lf (perf %lf)]\n",
	lowerParamValue, lowerPerf, upperParamValue, upperPerf);
    printf("%s parameter\n", positive?"positive":"negative");

    iter = 0;
    if (type == INTERPOLATE || type == BINARY) {
	do {
	    iter++;
	    if (iter > MAXITERPEREXPLORE) {
		printf("too many iterations for this exploreParam\n");
		printf("was exploring %s\n", convertParamStr(whichParam));
		printf("parameter range of knee is [%lf (perf %lf), %lf (perf %lf)]\n",
		    lowerParamValue, lowerPerf, upperParamValue, upperPerf);
		exit(1);
	    }
	    /* calculate the parameter value at the estimated knee (using the
		current values of the parameter range that the knee is in) */
	    if (type == INTERPOLATE) {
		paramValue = interpolate(upperParamValue, upperPerf,
		    lowerParamValue, lowerPerf,
		    minPerf + kneeFraction * (maxPerf - minPerf), 1);
	    } else {
		/* BINARY */
		paramValue = .5 * (upperParamValue + lowerParamValue);
	    }
	
	    /* run at that parameter value */
	    printf("running at %s = %lf\n", convertParamStr(whichParam),
		paramValue);
	    assignParamVal(&spec, whichParam, paramValue);

	    run(&spec, result, resultNumPtr, maxRunTime, fileServer,
		warmStartFlag, minRunTime, flushPossible);

	    /* re-describe the knee and knee range */

	    lowerParamValueOld = lowerParamValue;
	    upperParamValueOld = upperParamValue;

	    paramValue = describeKnee(whichParam, result, *resultNumPtr,
		kneeSpecPtr, minSpecPtr, maxSpecPtr, kneeFraction, kneeAbsolute,
		&lowerParamValue, &lowerPerf, &upperParamValue, &upperPerf,
		&minPerf, &maxPerf, &positive);
	    printf("knee value for %s is %lf\n", convertParamStr(whichParam),
		paramValue);
	    printf("parameter range of knee is [%lf (perf %lf), %lf (perf %lf)]\n",
		lowerParamValue, lowerPerf, upperParamValue, upperPerf);
	    printf("%s parameter\n", positive?"positive":"negative");
	    /* if the parameter range hasn't changed (be pretty strict on
		this one in paramDiff), or the range is pretty tight, then
		stop */
	    assignParamVal(&tmpSpec1, whichParam, lowerParamValue);
	    assignParamVal(&tmpSpec2, whichParam, lowerParamValueOld);
	    assignParamVal(&tmpSpec3, whichParam, upperParamValue);
	    assignParamVal(&tmpSpec4, whichParam, upperParamValueOld);

	} while (whichParam &
	    ( (paramDiff(&tmpSpec1, &tmpSpec2, NOPARAMETER, 0, 1.0) |
	    paramDiff(&tmpSpec3, &tmpSpec4, NOPARAMETER, 0, 1.0)) &
	    paramDiff(&tmpSpec1, &tmpSpec3, NOPARAMETER, 0, 1.0) )
	    );
    }
    if (kneeFraction > .4) {
	if (lowerPerf > upperPerf) {
	    paramValue = lowerParamValue;
	} else {
	    paramValue = upperParamValue;
	}
    } else {
	/* this must be for trying to narrow uniqueBytes, so I should return
	    the one with lower performance */
	printf("warning: taking the side with lower performance\n");
	if (whichParam != UNIQUEBYTES) {
	    printf("error in exploreParam: kneeFraction = %lf, whichParam = %d\n",
		kneeFraction, whichParam);
	    exit(1);
	}
	if (lowerPerf < upperPerf) {
	    paramValue = lowerParamValue;
	} else {
	    paramValue = upperParamValue;
	}
    }
    printf("final knee is %lf\n", paramValue);
    return(paramValue);
}

double describeKnee(whichParam, result, resultNum, kneeSpecPtr, minSpecPtr,
	maxSpecPtr, kneeFraction, kneeAbsolute, lowerParamValuePtr,
	lowerPerfPtr, upperParamValuePtr, upperPerfPtr, minPerfPtr,
	maxPerfPtr, positivePtr)
    int whichParam;
    WLRESULT result[MAXRESULTNUM];
    int resultNum;
    WLPARAM *kneeSpecPtr;
    WLPARAM *minSpecPtr;
    WLPARAM *maxSpecPtr;
    double kneeFraction;
    int kneeAbsolute; /* 1 ==> define the knee as an absolute fraction of the
				max
			 0 ==> define the knee as the kneeFraction way
				between min and max */
    double *lowerParamValuePtr, *lowerPerfPtr;
    double *upperParamValuePtr, *upperPerfPtr;
    double *minPerfPtr, *maxPerfPtr;
    int *positivePtr;
{
    /* finds the knee of the performance vs. whichParam curve
	returns the parameter value at the knee.  See ~/bench/wl.notes for
	    discussion of knee
	also returns the min and max bounding range of the knee (value
	    of the parameter at the nearest point below and above the knee).
	    note that the value of the function will be one of these points

	describeKnee assumes the curve is completely traced already.
	exploreParam (with full=0) will trace the curve out as it goes, trying
	    to save running time by searching for the knee
    */

    int i;
    WLSIMPLERESULT simpleResult[MAXSIMPLERESULTNUM];
    int simpleResultNum;
    int mini, maxi, kneei;
    double maxPerf, minPerf;
    double kneePerf;
    double paramValue;

    char *convertParamStr();
    double convertParamVal();
    double convertParamFlat();
    int condense();
    double fractionDiff();

#ifdef DEBUG
    printf("in describeKnee for parameter %s\n", convertParamStr(whichParam));
#endif

    /* plot curve */
    plotParam(whichParam, kneeSpecPtr, minSpecPtr, maxSpecPtr, result,
		resultNum, convertParamVal(kneeSpecPtr, whichParam));
    
    (*positivePtr) = whichParam & POSITIVE_PARAM;
	    /* to be changed later if parameter is negative */

    simpleResultNum = condense(whichParam, kneeSpecPtr, minSpecPtr, maxSpecPtr,
				    result, resultNum, simpleResult);
    /* condense returns a list of composite runs, ordered from i=0 ==>smallest
	parameter value to i=simpleResultNum-1 ==> largest parameter value */

    printf("simpleResultNum = %d\n", simpleResultNum);
    if (simpleResultNum <= 1) {
	/* not enough runs matched */
	printf("not enough runs matched in search for knee of %s\n",
	    convertParamStr(whichParam));
	
	/* return old knee point */
	(*lowerParamValuePtr) = (*upperParamValuePtr) = paramValue =
	    convertParamVal(kneeSpecPtr, whichParam);
	if (simpleResultNum < 1) {
	    /* no runs--return garbage value */
	    printf("returning -1 as the maxPerf\n");
	    (*lowerPerfPtr) = (*upperPerfPtr) = (*minPerfPtr) = (*maxPerfPtr) =
		-1.0;
	} else {
	    printf("returning %lf as the maxPerf\n",
		simpleResult[0].performance);
	    (*lowerPerfPtr) = (*upperPerfPtr) = (*minPerfPtr) = (*maxPerfPtr) =
		simpleResult[0].performance;
	}
	return(paramValue);
    }

    /* find max and min points */
    maxi = mini = -1;
    for (i=0; i<simpleResultNum; i++) {
	if (maxi<0 ||
	    simpleResult[i].performance > simpleResult[maxi].performance) {
	    maxi = i;
	}
	if (mini<0 ||
	    simpleResult[i].performance < simpleResult[mini].performance) {
	    mini = i;
	}
    }

    /* for parameters with well-defined "easiest" value, the min
	should be the easiest value */
    if ( whichParam & POSITIVE_PARAM) {
	mini = 0;
    } else if ( whichParam & NEGATIVE_PARAM) {
	mini = simpleResultNum-1;
    }

    if (mini<0 || maxi<0) {
	/* no runs matched */
	printf("error: no runs matched (after initial test for enough matches) in search for knee of %s\n",
	    convertParamStr(whichParam));
	exit(1);
    }

    maxPerf = (*maxPerfPtr) = simpleResult[maxi].performance;
    minPerf = (*minPerfPtr) = simpleResult[mini].performance;

    printf("for %s, max performance of %lf MB/s is achieved at %lf\n",
	convertParamStr(whichParam), maxPerf,
	simpleResult[maxi].paramValue);
    printf("min performance of %lf MB/s is achieved at %lf\n",
	minPerf, simpleResult[mini].paramValue);

    /* is this a flat curve? */

/*
    if ( (!kneeAbsolute && fractionDiff(maxPerf, minPerf) < FLATCRITERION) ||
	    (kneeAbsolute && (maxPerf*kneeFraction) < minPerf) ) {}
*/
    if ( (!kneeAbsolute && fractionDiff(maxPerf, minPerf) < FLATCRITERION) ||
	    (kneeAbsolute && (maxPerf*(1.0-FLATCRITERION)) < minPerf) ) {
	printf("this curve is flat\n");

	/* return a zero size parameter range */
	(*lowerParamValuePtr) = (*upperParamValuePtr) =
	    paramValue = convertParamFlat(whichParam,
		simpleResult[mini].paramValue, simpleResult[maxi].paramValue);
	    /* convertParamFlat(whichParam, minSpecPtr, maxSpecPtr); */
	(*lowerPerfPtr) = (*upperPerfPtr) = maxPerf;
	return(paramValue);
    }

    if (kneeAbsolute) {
        kneePerf = kneeFraction * maxPerf;
    } else {
	kneePerf = minPerf + kneeFraction * (maxPerf-minPerf);
    }
    printf("looking for knee of %lf MB/s\n", kneePerf);

    if ( (whichParam & NEGATIVE_PARAM) ||
	((whichParam & (~POSITIVE_PARAM)) &&
	simpleResult[mini].paramValue > simpleResult[maxi].paramValue) ) {
	/* this is a negative parameter curve */
	/* find the largest possible knee (or what the kneeFraction is) point */
	for (kneei=simpleResultNum-1; kneei>=0; kneei--) {
	    if (simpleResult[kneei].performance > kneePerf) {
		break;
	    }
	}
	(*positivePtr) = 0;
	(*lowerParamValuePtr) = simpleResult[kneei].paramValue;
	(*lowerPerfPtr) = simpleResult[kneei].performance;
	/* note: don't have to worry about kneei+1 being in the array,
	    since there's guaranteed to be at least 2 points (min and max),
	    and if there's only two, kneei will refer to the max */
	/* this no longer is guaranteed, since for kneeAbsolutes, the knee
	    may be the first min */
	if (kneei < simpleResultNum-1) {
	    (*upperParamValuePtr) = simpleResult[kneei+1].paramValue;
	    (*upperPerfPtr) = simpleResult[kneei+1].performance;
	} else {
	    (*upperParamValuePtr) = simpleResult[kneei].paramValue;
	    (*upperPerfPtr) = simpleResult[kneei].performance;
	}
    } else {
	/* this is a positive parameter curve */
	(*positivePtr) = 1;
	/* find the smallest possible knee point */
	for (kneei=0; kneei<simpleResultNum; kneei++) {
	    if (simpleResult[kneei].performance > kneePerf) {
		break;
	    }
	}
	(*upperParamValuePtr) = simpleResult[kneei].paramValue;
	(*upperPerfPtr) = simpleResult[kneei].performance;
	if (kneei>0) {
	    (*lowerParamValuePtr) = simpleResult[kneei-1].paramValue;
	    (*lowerPerfPtr) = simpleResult[kneei-1].performance;
	} else {
	    (*lowerParamValuePtr) = simpleResult[kneei].paramValue;
	    (*lowerPerfPtr) = simpleResult[kneei].performance;
	}
    }

    if (kneei<0 || kneei >= simpleResultNum) {
	printf("error in describeKnee: no knee was found\n");
	exit(1);
    }

    printf("knee was found at simpleResult %d, performance was %lf, %s was %lf\n",
	kneei, simpleResult[kneei].performance, convertParamStr(whichParam),
	simpleResult[kneei].paramValue);

    return(simpleResult[kneei].paramValue);
}


double run(specPtr, result, resultNumPtr, maxRunTime, fileServer, warmStartFlag,
	minRunTime, flushPossible)
    WLPARAM *specPtr;
    WLRESULT result[MAXRESULTNUM];
    int *resultNumPtr;
    int maxRunTime;
    char *fileServer;
    int warmStartFlag;
    int minRunTime;
    int flushPossible;
{
    /* execute several runs with the given workload parameters, varying the
	seed */
    /* returns average performance of this run */

    int i;
    int iter;
    int whichParam;
    AutoStat perfStat;
    char tmpFileName[200];
    FILE *tmpFilePtr;
    double reUseValue, reUseValueOld;
    int reUseValueSet;
    double reUseMatchThroughput;
    int reUseMatchThroughputNum;
    int numScriptIter;
    int firstRun;
    double tmp;
    static int skip=SKIPSECONDS;

    static int seed=0;

    unsigned int paramDiff();
    double roundReUse();

    if (seed == 0) {
	seed = specPtr->seed;
    }

    firstRun = 1;

    reUseMatchThroughput = reUseMatchThroughputNum = 0;

    /* find out how many runs before were attempted with these parameters */
    printf("looking for matches from previous runs\n");
    InitAutoStat(&perfStat, 1, 1);
    for (i=0; i<(*resultNumPtr); i++) {
	if (paramDiff(specPtr, &(result[i].target), NOPARAMETER, 0, 10.0)==0) {
	    printf("run %d matched\n", i);
	    if (result[i].perf.throughput >= 0) {
		AddAutoStat(&perfStat, result[i].perf.throughput);
	    } else {
		printf("but run %d had negative throughput (run didn't match target parameters)\n", i);
	    }
	}
	/* match runs to form an initial reUse guess */
	if (paramDiff(specPtr, &(result[i].target), NOPARAMETER, 0, .5)==0) {
	    if (result[i].perf.throughput >= 0) {
		reUseMatchThroughput += result[i].perf.throughput;
		reUseMatchThroughputNum++;
	    }
	}
	    
    }

    printf("%d matches were found; still need to do at most %d runs\n",
	NStat(&perfStat), ITERPERRUN_MAX-NStat(&perfStat));
    if (NStat(&perfStat) > 0) {
	/* reUseValue = roundReUse(reUseValue / NStat(&perfStat)); */
	reUseValueSet = 1;
    } else if (reUseMatchThroughputNum > 0) {
	/* no close matches, but I have a guess of the throughput of somewhat
	    close matches */
	reUseMatchThroughput /= reUseMatchThroughputNum;
	reUseValue = roundReUse(
		minRunTime * 1.2 / (specPtr->uniqueBytes / (1024*1024.0) /
			reUseMatchThroughput) );
	if (reUseValue < MINREUSE) {
	    reUseValue = MINREUSE;
	}
	reUseValueSet = 0;
    } else {
	reUseValue = MINREUSE;
	reUseValueSet = 0;
    }
    /* printf("reUseValue=%lf\n", reUseValue); */

    for (iter=NStat(&perfStat);
	iter < ITERPERRUN_MAX &&
	    (NStat(&perfStat) < ITERPERRUN_MIN ||
		ConfStat(&perfStat, CONFBOUND)/AveStat(&perfStat) > CONFINTERVAL); 
	iter++) {

	printf("iter=%d, current %lf%% confidence interval is %lf of mean (%lf)\n",
	    iter, CONFBOUND*100, (NStat(&perfStat)<2) ? -1:
	    ConfStat(&perfStat, CONFBOUND)/AveStat(&perfStat),
	    NStat(&perfStat)==0?-1:AveStat(&perfStat));
	/*
	printf("trying to make run %d\n", *resultNumPtr);
	*/

	seed++;
	specPtr->seed = seed;

	if (paramDiff(specPtr, &(result[(*resultNumPtr)].target),
		NOPARAMETER, 0, 100.0) != 0 ) {
	    /* next run in result array isn't an exact match, I must not be
		continuing an aborted run */
	    printf("not continuing\n");

	    do {
		if (skip>0) {
		    printf("skipping this run\n");
		}
		do {

		    if (reUseValueSet && NStat(&perfStat)>0) {
			/* re-estimate reUse value from the previous
			    throughputs */
			reUseValue = roundReUse(
			    minRunTime / (specPtr->uniqueBytes / (1024*1024.0) /
			    AveStat(&perfStat)) );
			if (reUseValue < MINREUSE) {
			    reUseValue = MINREUSE;
			}
			printf("re-estimating reUse at %lf\n", reUseValue);
		    }
		    if (reUseValue > MAXREUSE && warmStartFlag) {
			specPtr->reUse = MAXREUSE;
			numScriptIter =
			    MAX(1,ceil((double)reUseValue/specPtr->reUse));
		    } else if (warmStartFlag) {
			specPtr->reUse = reUseValue;
			numScriptIter = 1;
		    }
		    /* specPtr->reUse = reUseValue; */
		    if (warmStartFlag) {
			printf("numScriptIter=%d\n", numScriptIter);
		    }
		    runSpecWl(specPtr);

		    /* copy target parameters into result structure for future
			reference */
		    copyWlParam(specPtr, &(result[*resultNumPtr].target));

		    /* get the specWl result of the run */
		    /*
		    sprintf(tmpFileName, "cat %s/specWl.adaptWl",
			specPtr->dirName);
		    mySystem(tmpFileName);
		    */
		    sprintf(tmpFileName, "%s/specWl.adaptWl", specPtr->dirName);
		    tmpFilePtr = fopen(tmpFileName, "r");
		    if (tmpFilePtr == NULL) {
			perror("error in run: fopen");
			printf("tmpFileName=%s\n", tmpFileName);
			exit(1);
		    }
		    getSpecWlResultFile(&(result[*resultNumPtr].param),
			tmpFilePtr);
		    fclose(tmpFilePtr);

		    if (paramDiff(specPtr, &(result[*resultNumPtr].param),
			    NOPARAMETER, 1, 1.0) == 0 ) {
			/* this run matched the target parameters */

			if (firstRun || !warmStartFlag) {
			    /* only read in the data on the first run for a
				workload */
			    firstRun = 0;
			    /* flush the first time this workoad is run */
			    if (flushPossible) {
				flushClientServer(fileServer, specPtr->dirName);
			    }
			    runDoWl(specPtr->dirName, maxRunTime, fileServer,
				warmStartFlag, warmStartFlag?numScriptIter:1);
			} else {
			    runDoWl(specPtr->dirName, maxRunTime, fileServer,
				0, warmStartFlag?numScriptIter:1);
			}

			getDoWlResult(&(result[*resultNumPtr].perf),
				    specPtr->dirName);

			result[*resultNumPtr].param.cpuThink =
				result[*resultNumPtr].perf.cpuThink;

			if (warmStartFlag && reUseValueSet<=0) {
			    tmp = result[*resultNumPtr].perf.duration/1000.0;
			    printf("duration of doWl run is %lf seconds\n",
				tmp);
			    if ( reUseValueSet <= -2 || tmp >= minRunTime) {
				reUseValueSet = 1;
				reUseValue =
				    roundReUse( MAX (MINREUSE,
					reUseValue * minRunTime / tmp));
			    } else {
				reUseValueOld = reUseValue;
				reUseValue =
				    roundReUse( MAX (MINREUSE,
					reUseValue * minRunTime * 1.2 / tmp));
				if (reUseValue == reUseValueOld) {
				    reUseValueSet = 1;
				    printf("reUseValue didn't change\n");
				} else {
				    printf("new reUseValue = %lf\n",
					reUseValue);
				    reUseValueSet--;
				}
			    }
			}
		
		    } else {
			/* fill in the param structure with dummy values that
			    should never match anything */
		    
			for (whichParam=1; whichParam<=MAXWLPARAM;
				whichParam*=2) {
			    assignParamVal(&(result[*resultNumPtr].param),
				whichParam, 0.0);
			}
			result[*resultNumPtr].perf.throughput = -1000000.0;
			result[*resultNumPtr].perf.avResponseTime = 1000000.0;
		    }
		} while (warmStartFlag && reUseValueSet<=0 &&
			result[*resultNumPtr].perf.throughput >= 0);
		if (skip > 0) {
		    skip -=
			 (int) ceil(result[*resultNumPtr].perf.duration/1000.0);
		    if (skip<0) {
			/* so that I really do skip this run */
			skip = 0;
		    }
		} else {
		    skip = -1;
		}
	    } while (skip>=0);

	} else {
	    printf("continuing run, using result # %d\n", *resultNumPtr);
	    if (result[*resultNumPtr].param.reUse >= MINREUSE) {
		/* reUseValue = result[*resultNumPtr].target.reUse; */
		/* don't need to set reUseValue, since I'll be estimating
		    it from the current average throughput */
		reUseValueSet = 1;
	    }
	}


	if (result[*resultNumPtr].perf.throughput >= 0) {
	    /* run must have been ok */
	    AddAutoStat(&perfStat, result[*resultNumPtr].perf.throughput);
	}

	printResult(&(result[*resultNumPtr]), *resultNumPtr);

	(*resultNumPtr)++;
	if ( (*resultNumPtr) >= MAXRESULTNUM) {
	    printf("error: resultNum too large (%d)\n", (*resultNumPtr));
	    exit(1);
	}
    } /* end iter */
    /*
    printf("CONFBOUND*100 = %lf\n", CONFBOUND*100);
    printf("NStat(&perfStat) = %d\n", NStat(&perfStat));
    printf("AveStat(&perfStat) = %lf\n", AveStat(&perfStat));
    */
    printf("final %lf%% confidence interval is %lf of mean (%lf)\n",
	CONFBOUND*100, (NStat(&perfStat)<2) ? -1:
	ConfStat(&perfStat, CONFBOUND)/AveStat(&perfStat), AveStat(&perfStat));
    
    if (NStat(&perfStat) <= 0) {
	printf("error in run: no runs successfully completed, NStat(&perfStat)=%d\n", NStat(&perfStat));
	exit(1);
    }

    return(AveStat(&perfStat));
}

narrowRange(minSpecPtr, maxSpecPtr, kneeSpecPtr, result, resultNumPtr,
	maxRunTime, fileServer, warmStartFlag, minRunTime, flushPossible,
	numExplore)
    WLPARAM *minSpecPtr, *maxSpecPtr, *kneeSpecPtr;
    WLRESULT result[MAXRESULTNUM];
    int *resultNumPtr;
    int maxRunTime;
    char *fileServer;
    int warmStartFlag;
    int minRunTime;
    int flushPossible;
    int numExplore;
{
    /* narrow the range of several parameters */

    WLPARAM spec;
    double paramValue;
    double kneeParamValue;
    int whichParam;
    double diskPerf;
    double perf;
    double tmpParamValue, tmpParamValue1;

    double narrowRangeParam();
    double narrowRangeParamNew();
    double dynamicMinMax();
    double convertParamVal();
    char *convertParamStr();
    double run();

    copyWlParam(kneeSpecPtr, &spec);

    /* figure out max value for uniqueBytes */
    printf("figuring out max uniqueBytes\n");

    /* spec.readProb = maxSpecPtr->readProb; */
    spec.processNum = minSpecPtr->processNum;
/* #ifdef UNDEF */
    spec.seqProb = 1.0; /* allows specWl to create a workload with reUse = 1 */
    /* not using this method of determining disk performance anymore, since it
	didn't work on mako */
    if (flushPossible) {
	/* first, measure performance under cold start with no reUse */
	spec.reUse = 1;
	/* pick a reasonable value for uniqueBytes */
	assignParamVal(&spec, UNIQUEBYTES, 2.0 *
		    dynamicMinMax(UNIQUEBYTES, &spec, minSpecPtr, maxSpecPtr, 1));
	if (spec.uniqueBytes > maxSpecPtr->uniqueBytes) {
	    spec.uniqueBytes = maxSpecPtr->uniqueBytes;
	}
	/* run with cold start */
	diskPerf = run(&spec, result, resultNumPtr, maxRunTime, fileServer, 0,
			minRunTime, flushPossible);
    } else {
/* #endif */
	printf("can't flush, measuring at max uniqueBytes to get disk performance\n");
	assignParamVal(&spec, UNIQUEBYTES,
	    convertParamVal(maxSpecPtr, UNIQUEBYTES));
	diskPerf = run(&spec, result, resultNumPtr, maxRunTime, fileServer, 1,
			minRunTime, flushPossible);
/* #ifdef UNDEF */
    }
/* #endif */
    printf("disk performance with seqProb=%lf, readProb=%lf, sizeMean=%lf KB, uniqueBytes=%lf KB is %lf MB/s\n",
	convertParamVal(&spec, SEQPROB), convertParamVal(&spec, READPROB),
	convertParamVal(&spec, SIZEMEAN), convertParamVal(&spec, UNIQUEBYTES),
	diskPerf);

    /* second, explore uniqueBytes until performance reaches 120% of diskPerf */
/* #ifdef UNDEF */
    if (flushPossible) {
	spec.reUse = 4;
    }
/* #endif */
    assignParamVal(&spec, UNIQUEBYTES, 2.0 *
		dynamicMinMax(UNIQUEBYTES, &spec, minSpecPtr, maxSpecPtr, 1));
    if (spec.uniqueBytes > maxSpecPtr->uniqueBytes) {
	spec.uniqueBytes = maxSpecPtr->uniqueBytes;
    }
    do {
	spec.uniqueBytes *= 2;
	if (spec.uniqueBytes < maxSpecPtr->uniqueBytes) {
	    printf("spec.uniqueBytes=%lf KB\n",
		convertParamVal(&spec, UNIQUEBYTES));
	    perf = run(&spec, result, resultNumPtr, maxRunTime, fileServer,
			!flushPossible, minRunTime, flushPossible);
	}
    } while (spec.uniqueBytes < maxSpecPtr->uniqueBytes && perf > 1.2*diskPerf);

    assignParamVal(maxSpecPtr, UNIQUEBYTES,
	MIN(convertParamVal(&spec,UNIQUEBYTES) * 1.5,
	    convertParamVal(maxSpecPtr, UNIQUEBYTES)));
    printf("new max uniqueBytes = %lf KB\n",
	convertParamVal(maxSpecPtr, UNIQUEBYTES));
    
    /* max processNum */
    whichParam = PROCESSNUM;

    /* first test is to get at the parallelism in disk system
	(see adaptWl/notes) */
    copyWlParam(kneeSpecPtr, &spec);
    spec.uniqueBytes = maxSpecPtr->uniqueBytes;
    /*
    spec.processNum = maxSpecPtr->processNum; so that dynamicMinMax works
						    right
    assignParamVal(&spec, UNIQUEBYTES,
	    dynamicMinMax(UNIQUEBYTES, &spec, minSpecPtr, maxSpecPtr, 1));
    */

    paramValue = narrowRangeParamNew(whichParam, &spec, minSpecPtr, maxSpecPtr,
	result, resultNumPtr, maxRunTime, fileServer, &kneeParamValue,
	warmStartFlag, minRunTime, flushPossible);
    
    /*
    assignParamVal(maxSpecPtr, whichParam, paramValue);
    printf("new max %s = %lf\n", convertParamStr(whichParam),
	convertParamVal(maxSpecPtr, whichParam));
    */

    /* assign knee value to be min + 1/3 distance from min to max */
    /*
    assignParamVal(kneeSpecPtr, whichParam,
	(convertParamVal(maxSpecPtr,whichParam) +
	2 * convertParamVal(minSpecPtr,whichParam) )/3.0 );
    */
    assignParamVal(kneeSpecPtr, whichParam, kneeParamValue);

    printf("new knee for %s = %lf\n", convertParamStr(whichParam),
	convertParamVal(kneeSpecPtr, whichParam));

#ifdef UNDEF
    /* this test is to get at parallelism in the file cache, or anywhere else
	above the disk system */
    /*  this test is no longer done */
    spec.readProb = .5 * (maxSpecPtr->readProb + minSpecPtr->readProb);
    spec.reUse = MIN(maxSpecPtr->reUse, 5 * minSpecPtr->reUse);

    paramValue1 = narrowRangeParamNew(whichParam, &spec, minSpecPtr, maxSpecPtr,
	result, resultNumPtr, maxRunTime, fileServer, &kneeParamValue,
	warmStartFlag, minRunTime, flushPossible);
    assignParamVal(maxSpecPtr, whichParam,
	MIN(convertParamVal(maxSpecPtr, whichParam), 
	    MAX(paramValue, paramValue1)));
#endif /* UNDEF */
    
    /* max sizeMean */
    whichParam = SIZEMEAN; 

    copyWlParam(kneeSpecPtr, &spec);
    spec.uniqueBytes = maxSpecPtr->uniqueBytes;

    paramValue = narrowRangeParamNew(whichParam, &spec, minSpecPtr, maxSpecPtr,
	result, resultNumPtr, maxRunTime, fileServer, &kneeParamValue,
	warmStartFlag, minRunTime, flushPossible);

    /* get focal point for sizeMean at max uniqueBytes */
    /* assign knee value to be min + 1/4 distance from min to max */
    /*
    tmpParamValue = (convertParamVal(maxSpecPtr,whichParam) +
	3 * convertParamVal(minSpecPtr,whichParam) )/4.0;
    */
    tmpParamValue = kneeParamValue;
    assignParamVal(kneeSpecPtr, whichParam, tmpParamValue);
    printf("new knee for %s = %lf at max uniqueBytes\n",
	convertParamStr(whichParam), convertParamVal(kneeSpecPtr, whichParam));

#ifdef UNDEF
    /* get focal point for sizeMean at min uniqueBytes */
    spec.uniqueBytes = minSpecPtr->uniqueBytes;

    paramValue = narrowRangeParamNew(whichParam, &spec, minSpecPtr, maxSpecPtr,
	result, resultNumPtr, maxRunTime, fileServer, &kneeParamValue,
	warmStartFlag, minRunTime, flushPossible);

    tmpParamValue1 = (2.0*convertParamVal(minSpecPtr, whichParam) + paramValue)/
			3.0;
    printf("knee for %s = %lf at min uniqueBytes\n",
	convertParamStr(whichParam), tmpParamValue1);

    assignParamVal(kneeSpecPtr, whichParam,
	(tmpParamValue+tmpParamValue1)/2.0);
#endif
    printf("new knee for %s = %lf\n", convertParamStr(whichParam),
	convertParamVal(kneeSpecPtr, whichParam));

#ifdef UNDEF
    /* max reUse */
    /* this test is no longer done, since reUse doesn't matter */
    whichParam = REUSE;

    copyWlParam(kneeSpecPtr, &spec);

    spec.hitDepth = minSpecPtr->hitDepth;
    spec.readProb = .5 * (maxSpecPtr->readProb + minSpecPtr->readProb);
    spec.sizeMean = minSpecPtr->sizeMean;
    spec.processNum = minSpecPtr->processNum;
    spec.seqProb = minSpecPtr->seqProb;
    assignParamVal(&spec, UNIQUEBYTES,
	dynamicMinMax(UNIQUEBYTES, &spec, minSpecPtr, maxSpecPtr, 1));

    paramValue = narrowRangeParam(whichParam, &spec, minSpecPtr, maxSpecPtr,
	result, resultNumPtr, maxRunTime, fileServer, &kneeParamValue,
	warmStartFlag, minRunTime, flushPossible, numExplore);
    assignParamVal(maxSpecPtr, whichParam,
	MIN(convertParamVal(maxSpecPtr, whichParam), paramValue));
    printf("new max %s = %lf\n", convertParamStr(whichParam),
	convertParamVal(maxSpecPtr, whichParam));

    assignParamVal(kneeSpecPtr, whichParam, kneeParamValue);
    printf("new knee for %s = %lf\n", convertParamStr(whichParam),
	convertParamVal(kneeSpecPtr, whichParam));

    /* max uniqueBytes */
    /* this test has been done above */
    whichParam = UNIQUEBYTES;

    copyWlParam(kneeSpecPtr, &spec);

    /* this uses the current processNum, sizeMean */
    /* spec.sizeMean = minSpecPtr->sizeMean; */
    spec.reUse = MIN(maxSpecPtr->reUse, 5 * minSpecPtr->reUse);
    spec.hitDepth = minSpecPtr->hitDepth;
    spec.readProb = .5 * (maxSpecPtr->readProb + minSpecPtr->readProb);
    spec.seqProb = minSpecPtr->seqProb;

    paramValue = narrowRangeParam(whichParam, &spec, minSpecPtr, maxSpecPtr,
	result, resultNumPtr, maxRunTime, fileServer, &kneeParamValue,
	warmStartFlag, minRunTime, flushPossible, numExplore);
    assignParamVal(maxSpecPtr, whichParam,
	MIN(convertParamVal(maxSpecPtr, whichParam), paramValue));
    printf("new max %s = %lf\n", convertParamStr(whichParam),
	convertParamVal(maxSpecPtr, whichParam));

    assignParamVal(kneeSpecPtr, whichParam, kneeParamValue);
    printf("new knee for %s = %lf\n", convertParamStr(whichParam),
	convertParamVal(kneeSpecPtr, whichParam));
#endif /* UNDEF */
}

double narrowRangeParamNew(whichParam, specPtr, minSpecPtr, maxSpecPtr,
	result, resultNumPtr, maxRunTime, fileServer, kneeParamValuePtr,
	warmStartFlag, minRunTime, flushPossible)
    int whichParam;
    WLPARAM *specPtr;
    WLPARAM *minSpecPtr, *maxSpecPtr;
    WLRESULT result[MAXRESULTNUM];
    int *resultNumPtr;
    int maxRunTime;
    char *fileServer;
    double *kneeParamValuePtr;
    int warmStartFlag;
    int minRunTime;
    int flushPossible;
{
    /* try to narrow the ranges of one parameter */

    int lower;
    double paramValue, paramValueSave;
    double perf, maxPerf, perfSave;

    double convertParamVal();
    double convertParamSmallestMax();
    char *convertParamStr();
    double exploreParam();
    double round();
    double fractionDiff();
    double run();

    printf("trying to narrow the range of %s\n", convertParamStr(whichParam));
    printf("started out %lf-%lf\n", convertParamVal(minSpecPtr, whichParam),
	convertParamVal(maxSpecPtr, whichParam));

    if (!(whichParam & (SIZEMEAN|PROCESSNUM))) {
	printf("error: narrowRangeParam called for parameter %s\n",
		convertParamStr(whichParam));
	exit(1);
    }

    /* start at minimum, then keep doubling until performance is no longer
	improving (stop when two runs in a row don't improve max by at least
	5%) */

    perf = maxPerf = lower = 0;
    /* paramValue = convertParamVal(minSpecPtr, whichParam); */
    paramValue = MAX(convertParamVal(minSpecPtr, whichParam),
			convertParamSmallestMax(whichParam)/2.0/1.5);
	/* divide by 2 to give a reasonable starting max and give it a chance
	    to stop at the smallest max, divide by 1.5 to compensate for
	    the multiply by 1.5 at the end */

    do {
	maxPerf = MAX(perf, maxPerf);
	assignParamVal(specPtr, whichParam, paramValue);
	perf = run(specPtr, result, resultNumPtr, maxRunTime, fileServer, 1,
		minRunTime, flushPossible);
	if (maxPerf == 0 || perf/maxPerf > (1.0+FLATCRITERION) ) {
	    lower = 0;
	} else {
	    lower++;
	    printf("lower=%d\n", lower);
	    if (lower == 1) {
		paramValueSave = paramValue;
		perfSave = perf;
	    }
	}
	paramValue *= 2.0;
    } while (paramValue <= convertParamVal(maxSpecPtr, whichParam) &&
	lower <= 1);
    
    if (lower > 1) {
	if (perfSave > maxPerf) {
	    /* performance still increasing, but slowly */
	    paramValue = paramValueSave * 1.5;
	} else {
	    paramValue = paramValueSave;
	}
    }
    
    paramValue = MIN(paramValue, convertParamVal(maxSpecPtr, whichParam));

    assignParamVal(maxSpecPtr, whichParam, paramValue);
    printf("new max %s = %lf\n", convertParamStr(whichParam),
	convertParamVal(maxSpecPtr, whichParam));

    (*kneeParamValuePtr) = exploreParam(whichParam, specPtr, minSpecPtr,
	    maxSpecPtr, result, resultNumPtr, maxRunTime, fileServer,
	    .5, 0, BINARY, 2, warmStartFlag,
	    minRunTime, flushPossible);

/*
    assignParamVal(kneeSpecPtr, whichParam, *kneeParamValuePtr);
    printf("new knee for %s = %lf\n", convertParamStr(whichParam),
	convertParamVal(kneeSpecPtr, whichParam));
*/

    return(paramValue);
}

double narrowRangeParam(whichParam, specPtr, minSpecPtr, maxSpecPtr,
	result, resultNumPtr, maxRunTime, fileServer, kneeParamValuePtr,
	warmStartFlag, minRunTime, flushPossible, numExplore)
    int whichParam;
    WLPARAM *specPtr;
    WLPARAM *minSpecPtr, *maxSpecPtr;
    WLRESULT result[MAXRESULTNUM];
    int *resultNumPtr;
    int maxRunTime;
    char *fileServer;
    double *kneeParamValuePtr;
    int warmStartFlag;
    int minRunTime;
    int flushPossible;
    int numExplore;
{
    /* try to narrow the ranges of one parameter */

    double criterion;
    double paramValue;

    double convertParamVal();
    char *convertParamStr();
    double exploreParam();
    double round();

    printf("trying to narrow the range of %s\n", convertParamStr(whichParam));
    printf("started out %lf-%lf\n", convertParamVal(minSpecPtr, whichParam),
	convertParamVal(maxSpecPtr, whichParam));

    if (whichParam == UNIQUEBYTES) {
	/* since uniqueBytes is a negative parameter and we're trying to lower
	    the max, we're looking for the point at which performance is
	    close to the minimum performance.  For other parameters, we're
	    looking for the point at which performance is close to the max */
	criterion = FLATCRITERION;
    } else if (!(whichParam & POSITIVE_PARAM)) {
	printf("error: narrowRangeParam called for a non-positive parameter %s\n",
		convertParamStr(whichParam));
	exit(1);
    } else {
	criterion = 1.0-FLATCRITERION;
    }

    /* explore by: first exploring at a few regular intervals to get the
	real max, then a binary search */

    /* note that this needs to be *not* kneeAbsolute, since that wouldn't
	narrow the range of the max at all (at least for uniqueBytes) */
    paramValue = exploreParam(whichParam, specPtr, minSpecPtr, maxSpecPtr,
	    result, resultNumPtr, maxRunTime, fileServer, criterion, 0,
	    BINARY, (int) round(numExplore/2.0), warmStartFlag, minRunTime,
	    flushPossible);

    if (paramValue > convertParamVal(maxSpecPtr, whichParam) ) {
	printf("warning: in narrowRangeParam, exploreParam somehow returned a paramValue (%lf) larger than the max allowed (%lf)\n", paramValue,
	convertParamVal(maxSpecPtr, whichParam));
	paramValue = convertParamVal(maxSpecPtr, whichParam);
    }

    printf("narrowRangeParam trying to find a reasonable starting knee\n");
    *kneeParamValuePtr = exploreParam(whichParam, specPtr, minSpecPtr,
	    maxSpecPtr, result, resultNumPtr, maxRunTime, fileServer,
	    KNEEFRACTION, 1, BINARY /* INTERPOLATE */, 2, warmStartFlag,
	    minRunTime, flushPossible);
    return(paramValue);
}

double dynamicMinMax(whichParam, specPtr, minSpecPtr, maxSpecPtr, minFlag)
    int whichParam;
    WLPARAM *specPtr;
    WLPARAM *minSpecPtr;
    WLPARAM *maxSpecPtr;
    int minFlag;
{
    /* returns the minimum or maximum value allowed for whichParam based on
	the other parameter values (in specPtr) */
    double paramValue;
    int fileNum;

    unsigned int fileSizeFunc();
    int fileNumFunc();
    double convertParamVal();
    char *convertParamStr();

    fileNum = fileNumFunc(specPtr->processNum, specPtr->sharing);
    if (minFlag) {
	if (whichParam == UNIQUEBYTES) {
	    /*
	    printf("MIN_FILESIZE_OVER_SIZE = %lf\n", MIN_FILESIZE_OVER_SIZE);
	    printf("fileNum = %d\n", fileNum);
	    printf("specPtr->sizeMean=%lf\n",
		convertParamVal(specPtr, SIZEMEAN));
	    */
	    paramValue = MIN_FILESIZE_OVER_SIZE * (double) fileNum *
		convertParamVal(specPtr, SIZEMEAN);
	} else {
	    printf("error in dynamicMinMax: whichParam = %d, minFlag = %d\n",
		whichParam, minFlag);
	    exit(1);
	}
    } else {
	if (whichParam == SIZEMEAN) {
	    paramValue = convertParamVal(specPtr, UNIQUEBYTES) /
			    (double) fileNum / MIN_FILESIZE_OVER_SIZE;
	} else if (whichParam == PROCESSNUM) {
	    /* note: this assumes sharing of 0 */
	    paramValue = convertParamVal(specPtr, UNIQUEBYTES) /
			    convertParamVal(specPtr, SIZEMEAN) /
			    MIN_FILESIZE_OVER_SIZE;
	} else {
	    printf("error in dynamicMinMax: whichParam = %d, minFlag = %d\n",
		whichParam, minFlag);
	    exit(1);
	}
    }
    /*
    printf("first estimate of dynamicMinMax %s is %lf\n",
	convertParamStr(whichParam), paramValue);
    */
    if (paramValue < convertParamVal(minSpecPtr, whichParam)) {
	paramValue = convertParamVal(minSpecPtr, whichParam);
    }
    if (paramValue > convertParamVal(maxSpecPtr, whichParam)) {
	paramValue = convertParamVal(maxSpecPtr, whichParam);
    }
    /*
    printf("dynamicMinMax returning %lf\n", paramValue);
    */
    return(paramValue);
}

double roundReUse(reUse)
    double reUse;
{
    /* round reUse to nearest .5 */
    return( ceil(reUse*2)/2.0);
}
