/*
 * Converted from the G++ library.
 */

#include "stat.h"
#include <math.h>
#include <stdio.h>

#ifndef HUGE
#define HUGE HUGE_VAL
#endif

/*
 * t-distribution: given p-value and degrees of freedom, return t-value
 * adapted from Peizer & Pratt JASA, vol63, p1416
 */
double
tval(p, df)
    double p;
    int    df;
{
    double t;
    int positive = p >= 0.5;
    p = (positive)? 1.0 - p : p;
    if (p <= 0.0 || df == 0) {
	t = HUGE;
    } else if (p == 0.5) {
	t = 0.0;
    } else if (df == 1) {
	t = 1.0 / tan((p + p) * 1.57079633);
    } else if (df == 2) {
	t = sqrt(1.0 / ((p + p) * (1.0 - p)) - 2.0);
    } else {
	double ddf = df;
	double a = sqrt(log(1.0 / (p * p)));
	double aa = a * a;
	a = a - ((2.515517 + (0.802853 * a) + (0.010328 * aa)) /
		 (1.0 + (1.432788 * a) + (0.189269 * aa) +
		  (0.001308 * aa * a)));
	t = ddf - 0.666666667 + 1.0 / (10.0 * ddf);
	t = sqrt(ddf * (exp(a * a * (ddf - 0.833333333) / (t * t)) - 1.0));
    }
    return (positive ? t : -t);
}

void
InitStat(stat)
    Stat *stat;
{
    stat->n = 0;
    stat->sumX = 0.0;
    stat->sumX2 = 0.0;
    stat->min = HUGE;
    stat->max = -HUGE;
}

void
InitBatchStat(bstat, batchSize)
    BatchStat  *bstat;
    int		batchSize;
{
    bstat->batchSize = batchSize;
    bstat->batchN = 0;
    bstat->batchSumX = 0.0;
    InitStat(&bstat->stat);
}

void
InitAutoStat(astat, batchSize, autoDist)
    AutoStat  *astat;
    int		batchSize;
    int		autoDist;
{
    int		i;

    astat->batchSize = batchSize;
    astat->batchN = 0;
    astat->batchSumX = 0.0;
    if (autoDist+1 > MAX_AUTO_COV_DIST) {
	printf("ERROR: AutoDist too large.\n");
	abort();
    } else {
	astat->autoDist = autoDist + 1;
    }
    InitStat(&astat->stat);

    for (i = 0; i < astat->autoDist; i++) {
	astat->sumXY[i] = 0.0;
	astat->firstX[i] = 0.0;
	astat->lastX[i] = 0.0;
    }

}

#ifndef MIN
#define MIN(a,b) ( (a) < (b) ? (a) : (b) )
#endif  MIN

#ifndef MAX
#define MAX(a,b) ( (a) > (b) ? (a) : (b) )
#endif  MAX

void
AddAutoStat(astat, x)
    AutoStat  *astat;
    double x;
{
    int		i;

    astat->batchN += 1;
    astat->batchSumX += x;
    if (astat->batchN >= astat->batchSize) {
	double 	batchX = astat->batchSumX/astat->batchN;

	if (NStat(astat) < astat->autoDist) {
	    astat->firstX[NStat(astat)] = batchX;
	}

	AddStat(&astat->stat, batchX);

	for (i = astat->autoDist-1; i > 0; i--) {
	    astat->lastX[i] = astat->lastX[i-1];
	}
	astat->lastX[0] = batchX;

	for (i = 0; i < MIN(NStat(astat), astat->autoDist); i++) {
	    astat->sumXY[i] += astat->lastX[i] * batchX;
	}

	astat->batchN = 0;
	astat->batchSumX = 0.0;
    }
}


void
AddStat(stat, x)
    Stat  *stat;
    double x;
{
    stat->n += 1;
    stat->sumX += x;
    stat->sumX2 += x * x;
    if (x < stat->min) {
	stat->min = x;
    }
    if (x > stat->max) {
	stat->max = x;
    }
}

void
AddBatchStat(bstat, x)
    BatchStat  *bstat;
    double x;
{
    bstat->batchN += 1;
    bstat->batchSumX += x;
    if (bstat->batchN >= bstat->batchSize) {
	AddStat(&bstat->stat, bstat->batchSumX/bstat->batchN);
	bstat->batchN = 0;
	bstat->batchSumX = 0.0;
    }
}

int
NStat(stat)
    Stat *stat;
{
    return (stat->n);
}

double
AveStat(stat)
    Stat *stat;
{
    if (stat->n==0) {
	return(HUGE);
    } else {
	return (stat->sumX / stat->n);
    }
}

double
VarMeanAutoStat(astat)
    AutoStat *astat;
{
    int		i;
    double	cov;
    double	firstSumX, lastSumX;
    int		autoDist;
    int		n;

    cov = 0.0;
    n = NStat(astat);
    firstSumX = astat->stat.sumX;
    lastSumX = astat->stat.sumX;
    autoDist = MIN(n-1, astat->autoDist);
    for (i = 1; i < autoDist; i++) {
	firstSumX -= astat->lastX[autoDist-i];
	lastSumX -= astat->firstX[i-1];
	cov += (1-((double)i)/n) *
		(astat->sumXY[i] - (n-i)*firstSumX/(n-i)*lastSumX/(n-i)) /
		(n-1-i);
    }
    return (VarStat(astat) + 2*cov)/n;
}

void
PrintVarMeanAutoStat(astat)
    AutoStat *astat;
{
    int		i;
    double	covi;
    double	firstSumX, lastSumX;
    int		autoDist;
    int		n;

    covi = 0.0;
    n = NStat(astat);
    firstSumX = astat->stat.sumX;
    lastSumX = astat->stat.sumX;
    autoDist = MIN(n-1, astat->autoDist);
    for (i = 1; i < autoDist; i++) {
	firstSumX -= astat->lastX[autoDist-i];
	lastSumX -= astat->firstX[i-1];
	covi = (1-((double)i)/n) *
		(astat->sumXY[i] - (n-i)*firstSumX/(n-i)*lastSumX/(n-i)) /
		(n-1-i);
	printf("%6.2f", covi/VarStat(astat));
    }
    printf("\n");
    printf("MEAN:%g VAR:%g INDSE:%g AUTOSE:%g N:%d\n",
	    AveStat(astat), VarStat(astat), SEStat(astat), SEAutoStat(astat),
	    NStat(astat));
}

double
VarStat(stat)
    Stat *stat;
{
    if (stat->n == 1) {
	/* printf("VarStat returning HUGE since n=1\n"); */
	return(HUGE);
    }
    return ((stat->sumX2 - ((stat->sumX*stat->sumX)/stat->n)) / (stat->n - 1));
}

double
SDStat(stat)
    Stat *stat;
{
    return sqrt(VarStat(stat));
}

double
SEAutoStat(astat)
    AutoStat *astat;
{
    return sqrt(VarMeanAutoStat(astat));
}


double
SEStat(stat)
    Stat *stat;
{
    /* printf("called SEStat, stat->=%d\n", stat->n); */
    return sqrt(VarStat(stat)/(double)stat->n);
}

double
ConfAutoStat(astat, p_value)
    AutoStat  *astat;
    double p_value;
{
    double t = tval((1.0 + p_value) * 0.5, NStat(astat));
    if (t == HUGE) {
	return t;
    } else {
	return (t * SEAutoStat(astat));
    }
}

double
ConfStat(stat, p_value)
    Stat  *stat;
    double p_value;
{
    double t = tval((1.0 + p_value) * 0.5, stat->n);
    if (t == HUGE) {
	return t;
    } else {
	return (t * SEStat(stat));
    }
}

void
AddStatStat(stat1, stat2)
    Stat  *stat1;
    Stat  *stat2;
{
    stat1->n += stat2->n;
    stat1->sumX += stat2->sumX;
    stat1->sumX2 += stat2->sumX2;
    if (stat2->min < stat1->min) {
	stat1->min = stat2->min;
    }
    if (stat2->max > stat1->max) {
	stat1->max = stat2->max;
    }
}
