/* This program is intended to take the definition of a workload, as specified
    by specWl and run it on the local computer

    doWl -d dirName -m maxRunTime -w -mi minRunTime
	dirName is where doWl looks to find parent.cmd
	maxRunTime is the maximum running time in seconds doWl is allowed.
	    If it hasn't finished by then, it exits with an error
	-w means warm start (read in all the data before starting the program)
	minRunTime makes the child processes repeat the script if they
	    finish before minRunTime (in seconds).  This only applies under
	    warm start, since we can't repeat the script if we're not warm
	    starting without changing reUse
*/

#include <stdio.h>
#include <sys/types.h>
#include <sys/file.h>
#include <fcntl.h>
#include <strings.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <sys/wait.h>
#include <math.h>
#include <sys/stat.h>
#include "doWl.h"

#define WARMSTARTBUFSIZE (1024*1024)

int latBucket[NUMLATBUCKET];

doWl(argc, argv, perfResultPtr, printFlag)
    int argc;
    char *argv[];
    WLPERF *perfResultPtr;
    int printFlag;
{
    int i;
    int status;

    /* workload parameters */
    char *dirName;
    int maxRunTime;
    int numScriptIter;
    int warmStartFlag;
    int processNum;
    int filesPerChild; /* how many files are accessed per child */

    /* command file info */
    FILE *tmpFilePtr;
    char tmpName[FILENAMEMAX];

    /* communication between parent and children */
    int childNum;
    FILE *resultFilePtr;
#ifdef SPRITE
    union wait waitStatus;
#else /* SPRITE */
    int waitStatus;
#endif /* SPRITE */

    /* performance results */
    double sumSize = 0;
    double sumSizeChild;
    double latency;
    double latencyChild;
    double throughput;
    struct timeval tStart, tEnd;
    /*
    double sumLatency;
    */
    double sum2Latency;
    double sumReadTime, sumWriteTime;
    double readTimeChild, writeTimeChild;
    int numIO;
    int numIOChild;
    int numReadChild, numWriteChild;
    int numRead, numWrite;
    int latBucketEntry;
    int latencyFreq;
    double averageLatency;

    /* function declarations */
    void getParamDoWl();

    for (i=0; i<argc; i++) {
	printf("%s ", argv[i]);
    }
    printf("\n");

    gethostname(tmpName, FILENAMEMAX);
    printf("host is %s\n", tmpName);
    /* return; */

    /* initialize latBucket */
    for (i=0; i<NUMLATBUCKET; i++) {
	latBucket[i] = 0;
    }

    getParamDoWl(argc, argv, &dirName, &maxRunTime, &warmStartFlag,
	&numScriptIter);

    if (printFlag == 2) {
	printf("directory %s\n", dirName);
	printf("maximum running time %d seconds\n", maxRunTime);
	if (warmStartFlag) {
	    printf("warm starting\n");
	}
	printf("numScriptIter = %d\n", numScriptIter);
    }

    /* read in the number of processes from the parent command file */
    sprintf(tmpName, "%s/%s.cmd", dirName, PARENTFILE);
    tmpFilePtr = fopen(tmpName, "r");
    if (tmpFilePtr == NULL) {
	perror("error: fopen");
	exit(1);
    }

    fscanf(tmpFilePtr, "processNum %d\n", &processNum);
    fscanf(tmpFilePtr, "filesPerChild %d\n", &filesPerChild);
    fclose(tmpFilePtr);

    if (processNum > PROCMAX) {
	printf("too many processes (%d)\n", processNum);
	exit(1);
    }

    if (filesPerChild > FILESPERCHILDMAX) {
	printf("too many files per child (%d)\n", filesPerChild);
	exit(1);
    }

    if (printFlag) {
	printf("%d processes\n", processNum);
	printf("%d files per child\n", filesPerChild);
    }

    if (warmStartFlag) {
	/* if there's more than one file per child, then all the children
	    are sharing all the files, and I only need to read in child 0's
	    data files to warm start */

	warmStart(dirName, (filesPerChild>1) ? 1:processNum, filesPerChild);
    }

    /* flush all output */
    fflush(stdout);
    fflush(stderr);
    gettimeofday(&tStart, (struct timezone *) NULL);
    /* ---------------- official start of run for parent --------------- */
    for (childNum = 0; childNum < processNum; childNum++) {
	status = fork();
	if (status < 0) {
	    perror("error: fork");
	    exit(1);
	} else if (!status) {
	    break;
	}
    }
    if (childNum == processNum) {
	if (printFlag == 2) {
	    printf("parent is alive\n");
	}
	for (i=0; i<processNum; i++) {
	    status = wait(&waitStatus);
	    if (status < 0) {
		perror("error: wait");
		exit(1);
	    }
#ifdef SPRITE
	    if (waitStatus.w_retcode) {
		printf("error: parent saw a child exit with error status %d\n",
		    waitStatus.w_retcode);
		exit(1);
	    }
#else /* SPRITE */
	    if (waitStatus) {
		printf("error: parent saw a child exit with error status 0x%x\n",
		    waitStatus);
		exit(1);
	    }
#endif /* SPRITE */
	    if (printFlag) {
		printf("parent saw a child finish (%d of %d)\n", i+1,
		    processNum);
	    }
	}

	/* ---------------- official end of run for parent --------------- */
	gettimeofday(&tEnd, (struct timezone *) NULL);
	if (printFlag == 2) {
	    printf("all children finished\n");
	}

	latency = 1000.0 * (tEnd.tv_sec - tStart.tv_sec) +
		    (tEnd.tv_usec - tStart.tv_usec) / 1000.0;

	/* read back all the results */

	/* sumLatency = sum2Latency = 0.0; */
	sum2Latency = 0.0;
	numIO = 0;
	numRead = numWrite = 0;
	sumReadTime = sumWriteTime = 0;

	for (childNum=0; childNum<processNum; childNum++) {
	    numIOChild = 0;
	    
	    /* get results from children */
	    sprintf(tmpName, "%s/%s.%d.result", dirName, CHILDFILE, childNum);
	    resultFilePtr = fopen(tmpName, "r");
	    if (resultFilePtr == NULL) {
		perror("error: fopen");
		exit(1);
	    }

	    fscanf(resultFilePtr, "%lf bytes %lf ms\n",
			&sumSizeChild, &latencyChild);
	    if (printFlag == 2) {
		printf("parent got %lf KB in %lf ms from child %d\n",
		    sumSizeChild/1024.0, latencyChild, childNum);
	    }
	    
	    if (latencyChild > latency) {
		printf("error in parent, latencyChild = %lf, latency = %lf\n",
		    latencyChild, latency);
		exit(1);
	    }
	    sumSize += sumSizeChild;

	    /* get read and write info from children */
	    fscanf(resultFilePtr, "total time for %d reads %lf ms\n",
		    &numReadChild, &readTimeChild);
	    fscanf(resultFilePtr, "total time for %d writes %lf ms\n",
		    &numWriteChild, &writeTimeChild);
	    numRead += numReadChild;
	    numWrite += numWriteChild;
	    sumReadTime += readTimeChild;
	    sumWriteTime += writeTimeChild;

	    while ( (status = fscanf(resultFilePtr, "%d %d\n",
		&latBucketEntry, &latencyFreq)) == 2) {
		if (latencyFreq <= 0) {
		    printf("error in parent while reading child %d: latencyFreq = %d\n", childNum, latencyFreq);
		    exit(1);
		}
		latBucket[latBucketEntry] += latencyFreq;
		/*
		sumLatency += LATBUCKETQUANTA * latBucketEntry * latencyFreq;
		*/
		sum2Latency += LATBUCKETQUANTA * (double)LATBUCKETQUANTA *
			latBucketEntry * latBucketEntry * (double)latencyFreq;
		numIO += latencyFreq;
		numIOChild += latencyFreq;
		if (numIOChild > numReadChild + numWriteChild) {
		    printf("error: number of I/O according to latency buckets (%d) is greater than %d + %d, latBucketEntry=%d, child %d\n", numIOChild, numReadChild, numWriteChild, latBucketEntry, childNum);
		    exit(1);
		}
	    }
	}

	/* print final performance results */
	throughput = sumSize/1024.0/1024.0 / (latency/1000.0);
	if (printFlag == 2) {
	    printf("sumSize = %lf KB, %d I/O's\n", sumSize/1024.0, numIO);
	    printf("average size %lf KB\n", sumSize/1024.0/numIO);
	}
	if (printFlag) {
	    printf("run time %lf seconds\n", latency/1000.0);
	    printf("average time for reads %lf ms\n",
		numRead ? sumReadTime/numRead:0);
	    printf("average time for writes %lf ms\n",
		numWrite ? sumWriteTime/numWrite:0);
	}
	averageLatency = (sumReadTime + sumWriteTime) / numIO;
	if (sum2Latency/numIO <= averageLatency*averageLatency) {
	    sum2Latency = averageLatency*averageLatency*numIO;
	}
	if (printFlag) {
	    printf("throughput %lf MB/s\n", throughput);
	    printf("avResponseTime %lf ms\n", averageLatency);
	    printf("%lf%% time spent in I/O\n",
		100.0 * (sumReadTime+sumWriteTime)/latency/(double)processNum);
	}
	if (printFlag == 2) {
	    printf("stDev of latency %lf ms\n",
		sqrt(sum2Latency/numIO - averageLatency * averageLatency));
	    printf("latency characteristic\n");
	    for (i=0; i<NUMLATBUCKET; i++) {
		if (latBucket[i]) {
		    printf("%d %d\n", i*LATBUCKETQUANTA, latBucket[i]);
		}
	    }
	}

	/* store results for adaptWl */
	/*
	perfResultPtr->throughput = throughput;
	perfResultPtr->avResponseTime = averageLatency;
	perfResultPtr->cpuThink = -1.0;
	perfResultPtr->duration = latency;
	*/

	/* print result for adaptWl */
	sprintf(tmpName, "%s/doWl.adaptWl", dirName);
	/* printf("opening file %s\n", tmpName); */
	tmpFilePtr = fopen(tmpName, "w");
	if (tmpFilePtr == NULL) {
	    perror("error in doWl: fopen");
	    exit(1);
	}

	fprintf(tmpFilePtr, "throughput %lf\n", throughput);
	fprintf(tmpFilePtr, "avResponseTime %lf\n", averageLatency);
	fprintf(tmpFilePtr, "cpuThink %lf\n", -1.0);
	fprintf(tmpFilePtr, "duration %lf\n", latency);
	return(0);
    } else {
	if (printFlag == 2) {
	    printf("child %d is alive with pid %x\n", childNum, getpid());
	}
	child(childNum, filesPerChild, dirName, maxRunTime, printFlag,
	    numScriptIter);
	exit(0);
    }
}

child(childNum, filesPerChild, dirName, maxRunTime, printFlag, numScriptIter)
    int childNum;
    int filesPerChild;
    char *dirName;
    int maxRunTime;
    int printFlag;
    int numScriptIter;
{
    int i, status;
    char command;
    int file;
    unsigned int size;
    unsigned int start;
    double waitTime;

    struct timeval tStart1, tEnd1;

    int iter; /* iteration within the script */
    int scriptIter=0; /* script iteration */
    double sumSize=0;
    double sumReadTime = 0;
    double sumWriteTime = 0;
    int numRead = 0;
    int numWrite = 0;
    unsigned int currentPos;

    char fileName[FILENAMEMAX];
    char dataFileName[FILENAMEMAX];
    int dataFd[FILESPERCHILDMAX];

    unsigned int fileSize;
    struct stat statBuf;

    double tmpSumLatency=0;

    FILE *cmdFilePtr;

    char *buf;
    unsigned int currentBufSize;

    struct timeval tStart, tEnd;
    struct timeval tStartIO, tEndIO;
    double latency; /* latency of this I/O in ms */
    double childRunTime; /* current running time of this child in ms */
    int latBucketEntry;
    int maxLatBucketEntry = -1;

    int skipNumber=0;

    void *malloc();
    FILE *openCommand();
    double round();

    cmdFilePtr = openCommand(dirName, childNum, 0);

    /* open the data file(s) for this child */
    for (i=0; i<filesPerChild; i++) {
	fscanf(cmdFilePtr, "%s %u\n", fileName, &fileSize);
	skipNumber++;

	/* now data files are specified with full pathnames, so I can use
	    absolute pathnames for symbolically linked files if I want to */
	sprintf(dataFileName, "%s", fileName);
	if (printFlag == 2) {
	    printf("%d: data file %s\n", childNum, dataFileName);
	}

	/* check to make sure fileSize is correct (this should be taken outo
	    later if I decide to have adaptWl only create one set of data
	    files for a bunch of runs */
	/*
	if (stat(dataFileName, &statBuf) < 0) {
	    perror("error stat");
	    exit(1);
	}
	if (S_ISREG(statBuf.st_mode) && fileSize != statBuf.st_size) {
	    printf("error in child %d: doWl got file of size %u instead of %u\n",
		childNum, statBuf.st_size, size);
	    exit(1);
	}
	*/

	dataFd[i] = open(dataFileName, O_RDWR, 0666);
	printf("dataFileName = %s\n", dataFileName);

	if (dataFd[i] < 0) {
	    perror("error: open");
	    printf("error: dataFileName = %s\n", dataFileName);
	    exit(1);
	}

	if (printFlag == 2) {
	    printf("%d: data file opened\n", childNum);
	}
    }

    /* ----------------- run officially begins for child ----------------- */
    gettimeofday(&tStart, (struct timezone *) NULL);

    currentBufSize = 0;
    currentPos = 0;
    iter = 0;

    for (scriptIter=0; scriptIter<numScriptIter; scriptIter++) {
	if (scriptIter > 0) {
	    /* close and re-open cmd file */
	    fclose(cmdFilePtr);
	    cmdFilePtr = openCommand(dirName, childNum, skipNumber);
	}
    /* repeat entire cmd script */

    while ( (status =
		    fscanf(cmdFilePtr, "%c %d %u %u %lf\n",
			&command, &file, &size, &start, &waitTime)) == 5) {
	iter++;
	
#ifdef DEBUG
	printf("%d: command %c file %d size %u start %u waitTime %lf\n",
	    childNum, command, file, size, start, waitTime);
#endif

	/* wait the appropriate amount of time */
	mySleep(waitTime);

	/* make sure there's adequate buffer space */
	if (size > currentBufSize) {
	    if (currentBufSize > 0) {
		free(buf);
	    }
	    buf = (char *) malloc(size);
	    if (buf == NULL) {
		printf("error in child %d malloc\n", childNum);
		perror("error: malloc");
		exit(1);
	    }
	    putGarbage(buf, size);
	    currentBufSize = size;
	}
	
	/* ------------ timing for this I/O begins here --------------- */
	gettimeofday(&tStartIO, (struct timezone *) NULL);

	/* position pointer in file */
	/* ??? can't try to save this seek, since other processes may have
	    repositioned the file pointer
	if (start != currentPos) {
	*/
	     status = lseek(dataFd[file], start, L_SET);
	     if (status < 0) {
		perror("error: lseek");
		exit(1);
	     }
	     if (status != start) {
		printf("error child %d: lseek didn't go to right place (%d)\n",
		    childNum, status);
		exit(1);
	     }
	/* ???
	}
	*/

	if (command == 'r') {
	    status = read(dataFd[file], buf, (int) size);
	    if (status < 0) {
		perror("error: read");
		printf("dataFd[file] = %d, buf = 0x%x, size = %d, start=%d\n",
		    dataFd[file], buf, (int) size, start);
		exit(1);
	    }
	    if (status != size) {
		printf("error child %d: read only read %d bytes, wanted %u\n",
		    childNum, status, size);
		exit(1);
	    }
	    buf[0] = buf[0]+1;
	} else if (command == 'w') {
	    status = write(dataFd[file], buf, (int) size);
	    if (status < 0) {
		perror("error: write");
		printf("dataFd[file] = %d, buf = 0x%x, size = %d, start=%d\n",
		    dataFd[file], buf, (int) size, start);
		exit(1);
	    }
	    if (status != size) {
		printf("error child %d: write only wrote %d bytes, wanted %u\n",
		    status, size);
		exit(1);
	    }
	} else {
	    printf("error child %d: command (%c) not r or w\n",
		childNum, command);
	    exit(1);
	}

	gettimeofday(&tEndIO, (struct timezone *) NULL);
	/* ------------ timing for this I/O ends here --------------- */

	currentPos = start + size;
	sumSize += (double)size;
	latency = 1000.0 * (tEndIO.tv_sec - tStartIO.tv_sec) +
		    (tEndIO.tv_usec - tStartIO.tv_usec) / 1000.0;
	/* printf("%d: latency %lf ms\n", childNum, latency); */

	latBucketEntry = (int) round(latency/LATBUCKETQUANTA);
	if (latBucketEntry >= NUMLATBUCKET) {
	    printf("warning in %d: latency %lf ms too large for latency buckets\n", 
		childNum, latency);
	    printf("warning in %d: tEndIO=%d, %d, tStartIO=%d, %d\n", childNum,
		tEndIO.tv_sec, tEndIO.tv_usec, tStartIO.tv_sec,
		tStartIO.tv_usec);
	    /*
	    exit(1);
	    */
	    /* put it in the maximum latBucket */
	    latBucketEntry = NUMLATBUCKET-1;
	}

	latBucket[latBucketEntry]++;
	if (maxLatBucketEntry < latBucketEntry) {
	    maxLatBucketEntry = latBucketEntry;
	}

	/* update the read and write buckets */
	if (command == 'r') {
	    numRead++;
	    sumReadTime += latency;
	} else if (command == 'w') {
	    numWrite++;
	    sumWriteTime += latency;
	}

	/* print out average latency for this batch of requests */
	/*
	tmpSumLatency += latency;
	if ( !((numRead+numWrite)%100)) {
	    if (numRead+numWrite>0) {
		printf("average latency for requests %d-%d = %lf ms\n",
		    numRead+numWrite-100, numRead+numWrite-1,
		    tmpSumLatency/100.0);
	    }
	    tmpSumLatency = 0;
	}
	*/

	/* check for exceeding maxRunTime */
	gettimeofday(&tEnd, (struct timezone *) NULL);
	childRunTime = 1000.0 * (tEnd.tv_sec - tStart.tv_sec) +
		    (tEnd.tv_usec - tStart.tv_usec) / 1000.0;
	if (childRunTime/1000.0 > maxRunTime) {
	    printf("error: child %d exceeded the maxRunTime of %d seconds\n",
		childNum, maxRunTime);
	    exit(1);
	}
    } /* end of running script once */

    if (status != EOF && status < 0) {
	perror("error: fscanf");
	printf("error: status = %d\n", status);
	exit(1);
    }

    if (status != EOF) {
	printf("error: child %d only got %d data items from the command file, iter=%d\n",
	    childNum, status, iter);
	exit(1);
    }

    /* determine whether or not to repeat the command script */
    /*
    printf("child %d finished script iteration %d after %lf seconds\n",
	childNum, scriptIter, childRunTime/1000.0);
    */

    } /* end repeat entire command script */

    for (i=0; i<filesPerChild; i++) {
	/* ??? flush buffer cache (this is temporary) */
	/*
	gettimeofday(&tStart1, (struct timezone *) NULL);
	status = fsync(dataFd[i]);
	if (status < 0) {
	    perror("error: fsync");
	    exit(1);
	}
	gettimeofday(&tEnd1, (struct timezone *) NULL);
	printf("time to do fsync: %lf ms\n",
		1000.0 * (tEnd1.tv_sec-tStart1.tv_sec) +
		(tEnd1.tv_usec-tStart1.tv_usec) / 1000.0);
	*/

	close(dataFd[i]);
    }
    free(buf);

    /* ----------------- run officially ends for child ----------------- */

    gettimeofday(&tEnd, (struct timezone *) NULL);

    fclose(cmdFilePtr);

    if (printFlag == 2) {
	printf("%d: finished workload\n", childNum);
    }

    /* print results for parent to read */
    sprintf(fileName, "%s/%s.%d.result", dirName, CHILDFILE, childNum);
    cmdFilePtr = fopen(fileName, "w");
    if (cmdFilePtr == NULL) {
	perror("error: fopen");
	exit(1);
    }

    /* first print throughput */
    latency = 1000.0 * (tEnd.tv_sec - tStart.tv_sec) +
		(tEnd.tv_usec - tStart.tv_usec) / 1000.0;
    fprintf(cmdFilePtr, "%lf bytes %lf ms\n", sumSize, latency);
    fprintf(cmdFilePtr, "total time for %d reads %lf ms\n", numRead,
	    sumReadTime);
    fprintf(cmdFilePtr, "total time for %d writes %lf ms\n", numWrite,
	    sumWriteTime);

    /* now print latency buckets */
    for (i=0; i<=maxLatBucketEntry; i++) {
	if (latBucket[i]) {
	    fprintf(cmdFilePtr, "%d %d\n", i, latBucket[i]);
	}
    }
}

putGarbage(buf, size)
    char *buf;
    unsigned int size;
{
    /* puts size bytes of random data into the buffer buf */
    int i,j;

    long randomCompress();

    if (buf == NULL) {
	printf("error in putGarbage: buf is NULL\n");
    }

    for (i=0; i<size/sizeof(long); i++) {
	*( (long *)buf + i) = randomCompress();
    }

    for (j=0; j<size%sizeof(long); j++) {
	buf[i*sizeof(long) + j] = (char) randomCompress();
    }
}

long randomCompress()
{
    static int old=0;
    int decide;

    long myRandom();

    decide=myRandom()%4;

    if (decide < 2) {
	return(old);
    } else if (decide == 2) {
	return((old=(myRandom()%(1024*256)) ));
    } else {
	return((old=(myRandom()%64) ));
    }
}

warmStart(dirName, processNum, filesPerChild)
    char *dirName;
    int processNum;
    int filesPerChild;
{
    int childNum, i;
    unsigned fileSize;

    char fileName[FILENAMEMAX];
    char dataFileName[FILENAMEMAX];
    int dataFd;
    FILE *cmdFilePtr;

    unsigned int bytesLeft, bytes;
    int status;
    char *buf;

    FILE *openCommand();
    void *malloc();

    buf = malloc(WARMSTARTBUFSIZE);
    if (buf == NULL) {
	perror("malloc");
	printf("error in malloc of warm start buf\n");
	exit(1);
    }

    for (childNum=0; childNum<processNum; childNum++) {
	printf("warm starting child %d\n", childNum);

	/* open the command file */
	cmdFilePtr = openCommand(dirName, childNum, 0);

	/* open the data file(s) for this child */
	for (i=0; i<filesPerChild; i++) {
	    fscanf(cmdFilePtr, "%s %u\n", fileName, &fileSize);

	    sprintf(dataFileName, "%s", fileName);

	    printf("reading file %s\n", dataFileName);

	    dataFd = open(dataFileName, O_RDWR, 0666);

	    if (dataFd < 0) {
		perror("error: open");
		printf("error: dataFileName = %s\n", dataFileName);
		exit(1);
	    }

	    /* now warm start by reading in all the data of this file */
	    for (bytesLeft=fileSize; bytesLeft>0; ) {
		bytes = MIN(WARMSTARTBUFSIZE, bytesLeft);

		status = read(dataFd, buf, (int) bytes);
		if (status != bytes) {
		    printf("error: read status=%d, should be %d\n", status,
			bytes);
		    if (status < 0) {
			perror("read");
		    }
		    printf("dataFd = %d, buf = 0x%x, bytes = %d\n",
			dataFd, buf, (int) bytes);
		    exit(1);
		}
		/* to prevent gaming */
		buf[0] = buf[0]+1;

		bytesLeft -= bytes;
	    }

	    /* make sure all dirty blocks are flushed to disk */
	    /*
	    printf("fsync\n");
	    fsync(dataFd);
	    */

	    close(dataFd);

	}
    }

    free(buf);
}

FILE *openCommand(dirName, childNum, skipNumber)
    char *dirName;
    int childNum;
    int skipNumber; /* how many lines to skip after opening */
{
    char fileName[FILENAMEMAX];
    FILE *cmdFilePtr;
    int i;

    /* open the command file */
    sprintf(fileName, "%s/%s.%d.cmd", dirName, CHILDFILE, childNum);
    /* printf("opening cmd file %s for child %d\n", fileName, childNum); */
    cmdFilePtr = fopen(fileName, "r");
    if (cmdFilePtr == NULL) {
	perror("error: fopen");
	exit(1);
    }

    /* skip the first "skipNumber" of lines (for repeating the command script
	without reopening the data files */

    /* printf("skipping %d lines in beginning of %s\n", skipNumber, fileName);
	*/
    for (i=0; i<skipNumber; i++) {
	fscanf(cmdFilePtr, "%*[^\n]\n");
    }
    return(cmdFilePtr);
}
