/* routines to implement an address 2-3 tree for calculating the
    locality of a trace */
#include <stdio.h>
#include "tree.h"

#define LEAVESFORVIADDRESS 5

int addressNodeId=0; /* ok to have automatically initialized */
unsigned int lastLeaf;
int numLeavesPrintedAddress=0; /* ok to have automatically initialized */

char *malloc();
long myRandom();

TWOTHREENODE *insertAddress(rootPtr, newNodePtr)
    TWOTHREENODE *rootPtr;
    TWOTHREENODE *newNodePtr;
{
    TWOTHREENODE *tmpPtr, *tmpPtr1;
    unsigned int minStart;

    TWOTHREENODE *insertAddressSub();

    if (rootPtr == NULL) {
	/* the root is empty */
	/* set rootPtr to the newNode */
	/*
	printf("root is empty\n");
	printf("setting root node to be a leaf (#%d)\n", newNodePtr->id);
	*/
	newNodePtr->parent = NULL;
	return (newNodePtr);
    }

    if (rootPtr->left == NULL) {
	/* root is a leaf */

	/*
	printf("root is a leaf (#%d)\n", rootPtr->id);
	*/

	/* make a new root which points to the two leaves (the old root
	    node and the newNode) */
	tmpPtr = (TWOTHREENODE *) malloc(sizeof(TWOTHREENODE));
	tmpPtr->id = addressNodeId++;
	if (newNodePtr->start < rootPtr->start) {
	    tmpPtr->left = newNodePtr;
	    tmpPtr->middle = rootPtr;
	    tmpPtr->minMiddle = rootPtr->start;
	} else {
	    tmpPtr->left = rootPtr;
	    tmpPtr->middle = newNodePtr;
	    tmpPtr->minMiddle = newNodePtr->start;
	}
	tmpPtr->left->parent = tmpPtr->middle->parent = tmpPtr;
	tmpPtr->parent = NULL;
	tmpPtr->right = NULL;
	/*
	printf("set root to be a branch (#%d) pointing to %d (#%d) and %d (#%d)\n",
	    tmpPtr->id, tmpPtr->left->start, tmpPtr->left->id, tmpPtr->middle->start, tmpPtr->middle->id);
	*/
	return(tmpPtr);
    }

    /* root is normal (a parent of 2 or 3 nodes, which may be leaves) */
    /*
    printf("root is a normal branch\n");
    */
    tmpPtr1 = insertAddressSub(rootPtr, newNodePtr, &minStart);
    /*
    printf("\tinsertAddressSub returned to insertAddress\n");
    */

    if (tmpPtr1 != NULL) {
	/*
	printf("\treturned a new node with minStart %d (#%d)\n",
	    minStart, tmpPtr1->id);
	*/
	/* create new root which points to old root and tmpPtr */
	tmpPtr = (TWOTHREENODE *) malloc(sizeof(TWOTHREENODE));
	tmpPtr->id = addressNodeId++;

	/* I don't need to figure out which side to put the new node, since
	    the new node insertAddressSub returns always goes to the right of
	    the node insertAddressSub was called with */
	tmpPtr->left = rootPtr;
	tmpPtr->middle = tmpPtr1;
	tmpPtr->minMiddle = minStart;
	tmpPtr->parent = NULL;
	tmpPtr->right = NULL;
	tmpPtr->left->parent = tmpPtr->middle->parent = tmpPtr;
	/*
	printf("creating new root (#%d)\n", tmpPtr->id);
	*/
	return (tmpPtr);
    }

    /* no new root */
    /*
    printf("returned NULL, no new root\n");
    printf("root now points to #%d (minMiddle %d), #%d (minMiddle %d)",
	rootPtr->left->id, rootPtr->left->minMiddle, rootPtr->middle->id,
	rootPtr->middle->minMiddle);
    */
    if (rootPtr->right != NULL) {
	/*
        printf(", #%d (minMiddle %d)", rootPtr->right->id, rootPtr->right->minMiddle);
	*/
    }
    /*
    printf("\n");
    */
    return(rootPtr);
}

TWOTHREENODE *insertAddressSub(nodePtr, newNodePtr, minStartPtr)
    TWOTHREENODE *nodePtr;
    TWOTHREENODE *newNodePtr;
    unsigned int *minStartPtr;
{
    TWOTHREENODE *newPtr;
    TWOTHREENODE *tmpPtr;
    int child;
    unsigned int saveMin;

    /* if insertAddressSub returns a new node, it always goes to the right
	of node */

    /*
    printf("called insertAddressSub on node %d\n", nodePtr->id);
    */

    if (nodePtr->left == NULL) {
	/* node is a leaf */
	/* make sure node returned goes to the right of node */
	if (newNodePtr->start < nodePtr->start) {
	    swapContents(newNodePtr, nodePtr);
	}

	/* return newNode */
	*minStartPtr = newNodePtr->start;
	return(newNodePtr);
    }

    /* node is a branch */

    /* which child to follow */
    if (newNodePtr->start < nodePtr->minMiddle) {
	/* follow left branch */
	child = 1;
	tmpPtr = insertAddressSub(nodePtr->left, newNodePtr, minStartPtr);
    } else if (nodePtr->right == NULL ||
	    newNodePtr->start < nodePtr->minRight) {
	/* follow middle branch */
	child = 2;
	tmpPtr = insertAddressSub(nodePtr->middle, newNodePtr, minStartPtr);
    } else {
	/* follow right branch */
	child = 3;
	tmpPtr = insertAddressSub(nodePtr->right, newNodePtr, minStartPtr);
    }

    if (tmpPtr == NULL) {
	/*
	printf("\tsub returned to sub with no new node\n");
	*/
	/* no new children need to be added to node */
	/* don't have to worry about changing the minMiddle, minRight values
	    of node, since they will never change.  E.g. if the new start
	    was less than minMiddle, you would have followed the left branch,
	    not the middle branch */
	return(NULL);
    }

    /* need to add a new child to node */
    /*
    printf("\tsub returned to sub with a new node to add on (#%d) with minStart %d\n",
	tmpPtr->id, *minStartPtr);
    */
    if (nodePtr->right == NULL) {
	/* node only has 2 children */
	/*
	printf("\t\tnode has two children--adding the third\n");
	*/
	if (child == 1) {
	    /* followed the left branch */
	    nodePtr->right = nodePtr->middle;
	    nodePtr->minRight = nodePtr->minMiddle;
	    nodePtr->middle = tmpPtr;
	    nodePtr->minMiddle = *minStartPtr;
	} else if (child == 2) {
	    /* followed middle branch */
	    nodePtr->right = tmpPtr;
	    nodePtr->minRight = *minStartPtr;
	} else {
	    printf("error: took right branch, but only had 2 children\n");
	    exit(1);
	}
	nodePtr->left->parent = nodePtr->middle->parent =
	    nodePtr->right->parent = nodePtr;
	return(NULL);
    }

    /* node already has 3 children */

    /* create new node to pass up to caller of insertAddressSub */
    newPtr = (TWOTHREENODE *) malloc(sizeof(TWOTHREENODE));
    newPtr->id = addressNodeId++;

    /*
    printf("\t\tnode already has 3 children\n");
    printf("\t\tcreated new node #%d\n", newPtr->id);
    */

    if (child == 3) {
	/* right child and tmpPtr become children of new node */
	newPtr->left = nodePtr->right;
	newPtr->middle = tmpPtr;
	newPtr->minMiddle = *minStartPtr;
	newPtr->right = NULL;
	*minStartPtr = nodePtr->minRight;
	nodePtr->right = NULL;
    } else {
	/* child is left or middle */

	/* move right child of node to middle of the new node */
	newPtr->middle = nodePtr->right;
	newPtr->minMiddle = nodePtr->minRight;
	newPtr->right = NULL;
	nodePtr->right = NULL;

	if (child == 2) {
	    /* followed the middle branch */

	    /* connect returned node to left branch of new node */
	    newPtr->left = tmpPtr;
	    /* don't have to change minStart */
	} else {
	    /* followed left branch */

	    /*
	    printf("(followed left branch)\n");
	    */

	    /* node's middle child becomes left branch of new node */
	    newPtr->left = nodePtr->middle;
	    saveMin = *minStartPtr;
	    *minStartPtr = nodePtr->minMiddle;

	    /* returned node becomes middle branch of node */
	    nodePtr->middle = tmpPtr;
	    nodePtr->minMiddle = saveMin;
	    /*
	    printf("now node #%d has minMiddle %d\n",
		nodePtr->id, nodePtr->minMiddle);
	    */
	}
    }
    newPtr->left->parent = newPtr->middle->parent = newPtr;
    nodePtr->left->parent = nodePtr->middle->parent = nodePtr;
    return(newPtr);
}

TWOTHREENODE *deleteAddress(rootPtr, start)
    TWOTHREENODE *rootPtr;
    unsigned int start; /* starting address of node to be deleted */
{
    int onlyOne; /* holds status from deleteAddressSub call */
    unsigned int unUsed;
    TWOTHREENODE *newRootPtr;

    /* returns the new root */
    if (rootPtr == NULL) {
	/* root is empty */
	printf("error: called delete on an empty tree\n");
	exit(1);
    }

    if (rootPtr->left == NULL) {
	/* root is a leaf */
	if (rootPtr->start == start) {
	    /* found the match */
	    free((char *)rootPtr);
	    return(NULL);
	} else {
	    printf("no match found for start = %d (root is a leaf)\n", start);
	    exit(1);
	}
    }

    /* root is a normal branch */
    /* printf("root is a normal branch\n"); */
    onlyOne = deleteAddressSub(rootPtr, start, 0, &unUsed); /* ??? the 0 is questionable */

    /* printf("sub returned to deleteAddress with onlyOne=%d\n", onlyOne); */
    if (onlyOne) {
	/* root only has one child left */
	/* printf("root only has one child left--making that child the root\n"); */

	/* free the old root */
	newRootPtr = rootPtr->left;
	free((char *)rootPtr);

	newRootPtr->parent = NULL;
	return(newRootPtr);
    }

    /* printf("leaving the root unchanged\n"); */
    return(rootPtr);
}

int deleteAddressSub(nodePtr, start, minFromAbove, minFromChildPtr)
    TWOTHREENODE *nodePtr;
    unsigned int start;
    unsigned int minFromAbove;
    unsigned int *minFromChildPtr;
{
    int onlyOne; /* holds status from deleteAddressSub call */
    unsigned int minFromChildSub;
    int child; /* which child was followed */

    /*
    printf("called sub on #%d, min for this tree %d\n",
	nodePtr->id, minFromAbove);
    */

    if (nodePtr->left->left == NULL) {
	/* children of nodePtr are leaves */
	/* search for start among those leaves */
	/* printf("children of node are leaves\n"); */
	if (start == nodePtr->left->start) {
	    /* matched on left child */
	    free((char *)nodePtr->left);
	    nodePtr->minMiddle = nodePtr->minRight;
	    nodePtr->left = nodePtr->middle;
	    nodePtr->middle = nodePtr->right;
	    nodePtr->right = NULL;
	} else if (start == nodePtr->middle->start) {
	    /* matched on middle child */
	    free((char *)nodePtr->middle);
	    nodePtr->minMiddle = nodePtr->minRight;
	    nodePtr->middle = nodePtr->right;
	    nodePtr->right = NULL;
	} else if (nodePtr->right != NULL && start == nodePtr->right->start) {
	    /* matched on right child */
	    free((char *)nodePtr->right);
	    nodePtr->right = NULL;
	} else {
	    printf("couldn't find a match for start = %d\n", start);
	    exit(1);
	}

	/* return the minimum child */
	*minFromChildPtr = nodePtr->left->start;

	/* if node only has one child, return TRUE */
	return( (nodePtr->middle == NULL) );
    }

    /* children of nodePtr are branches */

    /* printf("children of node are branches\n"); */
    /* figure out which of the children of nodePtr to follow */
    if (start < nodePtr->minMiddle) {
	/* follow left branch */
	child = 1;
	onlyOne = deleteAddressSub(nodePtr->left, start, minFromAbove,
		    &minFromChildSub);
    } else if (nodePtr->right == NULL || start < nodePtr->minRight) {
	/* follow middle branch */
	child = 2;
	onlyOne = deleteAddressSub(nodePtr->middle, start, nodePtr->minMiddle,
		&minFromChildSub);
    } else {
	/* follow right branch */
	child = 3;
	onlyOne = deleteAddressSub(nodePtr->right, start, nodePtr->minRight,
		    &minFromChildSub);
    }

    /*
    printf("sub returned to sub (nodePtr=#%d, with onlyOne=%d, minFromChildSub=%d\n",
	nodePtr->id, onlyOne, minFromChildSub);
    */
	

    if (onlyOne) {
	if (child == 1) {
	    /* followed the left branch */
	    /* printf("fixing up left branch\n"); */
	    *minFromChildPtr = minFromChildSub;
	    if (nodePtr->middle->right != NULL) {
		/* middle child of node has 3 children */
		/* make left child of the middle child be the middle child
		    of the left */
		nodePtr->left->middle = nodePtr->middle->left;
		nodePtr->left->minMiddle = nodePtr->minMiddle;

		nodePtr->minMiddle = nodePtr->middle->minMiddle;

		nodePtr->middle->minMiddle = nodePtr->middle->minRight;
		nodePtr->middle->left = nodePtr->middle->middle;
		nodePtr->middle->middle = nodePtr->middle->right;
		nodePtr->middle->right = NULL;
		
		nodePtr->left->middle->parent = nodePtr->left;
		return(0); /* still has enough children */
	    } else {
		/* middle child only has 2 children */
		/* transfer 2 children of middle branch over to left branch */
		nodePtr->left->middle = nodePtr->middle->left;
		nodePtr->left->right = nodePtr->middle->middle;

		nodePtr->left->minMiddle = nodePtr->minMiddle;
		nodePtr->left->minRight = nodePtr->middle->minMiddle;

		/* free middle child */
		free((char *)nodePtr->middle);

		/* move right child (if there is one) to the middle child */
		nodePtr->middle = nodePtr->right;
		nodePtr->minMiddle = nodePtr->minRight;
		nodePtr->right = NULL;

		nodePtr->left->middle->parent = nodePtr->left->right->parent =
		    nodePtr->left;
		return( (nodePtr->middle == NULL) ); /* only one child left */
	    }
	} else if (child == 2) {
	    /* followed the middle branch */
	    /* printf("fixing up middle branch\n"); */
	    *minFromChildPtr = minFromAbove;
	    if (nodePtr->left->right != NULL) {
		/* left child of node has 3 children */
		/* printf("grabbing a child from the left neighbor\n"); */
		/* make right child of left node be left child of middle */
		nodePtr->middle->right = nodePtr->middle->middle;
		nodePtr->middle->middle = nodePtr->middle->left;

		nodePtr->middle->left = nodePtr->left->right;
		nodePtr->middle->minMiddle = minFromChildSub;
		/* *minFromChildPtr = nodePtr->left->minRight; */
		nodePtr->minMiddle = nodePtr->left->minRight;
		nodePtr->left->right = NULL;

		nodePtr->middle->left->parent = nodePtr->middle;
		return(0);
	    } else if (nodePtr->right != NULL &&
		    nodePtr->right->right != NULL) {
		/* right child of node has 3 children */
		/* printf("grabbing a child from the right neighbor\n"); */
		/* make left child of right node be middle child of middle */
		nodePtr->middle->middle = nodePtr->right->left;
		
		nodePtr->right->left = nodePtr->right->middle;
		nodePtr->right->middle = nodePtr->right->right;
		nodePtr->right->right = NULL;

		nodePtr->minMiddle = minFromChildSub;
		nodePtr->middle->minMiddle = nodePtr->minRight;
		nodePtr->minRight = nodePtr->right->minMiddle;
		nodePtr->right->minMiddle = nodePtr->right->minRight;

		nodePtr->middle->middle->parent = nodePtr->middle;
		return(0);
	    } else {
		/* no neighbors of the middle child have 3 children */
		/* add the only child of the middle child to the left child */
		/* printf("moving only child of middle to left neighbor\n"); */
		nodePtr->left->right = nodePtr->middle->left;
		nodePtr->left->minRight = minFromChildSub;

		/* free the old middle child */
		free((char *)nodePtr->middle);

		nodePtr->middle = nodePtr->right;
		nodePtr->minMiddle = nodePtr->minRight;
		nodePtr->right = NULL;

		nodePtr->left->right->parent = nodePtr->left;

		return( (nodePtr->middle == NULL) );
	    }
	} else if (child == 3) {
	    /* followed the right branch */
	    /* printf("fixing up right branch\n"); */
	    *minFromChildPtr = minFromAbove;
	    if (nodePtr->middle->right != NULL) {
		/* middle child has 3 children */
		/* make right child of middle node be the left child of the
		    right node */
		nodePtr->right->middle = nodePtr->right->left;
		nodePtr->right->minMiddle = minFromChildSub;
		nodePtr->right->left = nodePtr->middle->right;

		nodePtr->minRight = nodePtr->middle->minRight;
		nodePtr->middle->right = NULL;

		nodePtr->right->left->parent = nodePtr->right;

		return(0);

	    } else {
		/* middle child has 2 children */
		/* move only child of right node to the middle node */
		nodePtr->middle->right = nodePtr->right->left;
		nodePtr->middle->minRight = minFromChildSub;

		/* free right child */
		free((char *)nodePtr->right);
		nodePtr->right = NULL;

		nodePtr->middle->right->parent = nodePtr->middle;
		return(0); /* certainly has 2 children left */
	    }
	} else {
	    printf("error: child = %d\n", child);
	    exit(1);
	}
    } else { /* onlyOne = 0 */
	/* *minFromChildPtr = minFromChildSub; */
	if (child == 1) {
	    *minFromChildPtr = minFromChildSub;
	} else if (child == 2) {
	    /*
	    printf("changing minMiddle of #%d to %d\n", nodePtr->id,
		minFromChildSub);
	    */
	    nodePtr->minMiddle = minFromChildSub;
	    *minFromChildPtr = minFromAbove;
	} else if (child == 3) {
	    nodePtr->minRight = minFromChildSub;
	    /*
	    printf("changing minRight of #%d to %d\n", nodePtr->id,
		minFromChildSub);
	    */
	    *minFromChildPtr = minFromAbove;
	}
	return(0);
    }
    printf("error: hit bottom of deleteAddressSub\n");
    exit(1);
    return(0);
}

swapContents(node1Ptr, node2Ptr)
    TWOTHREENODE *node1Ptr;
    TWOTHREENODE *node2Ptr;
{
    /* swap the contents of node1 and node2 */
    TWOTHREENODE tmpNode;

    /*
    printf("swapping contents of node #%d and #%d\n", node1Ptr->id,
	node2Ptr->id);
    */

    /* copy node1 to tmpNode */
    copyContents(node1Ptr, &tmpNode);

    /* copy node2 to node1 */
    copyContents(node2Ptr, node1Ptr);

    /* copy tmpNode to node2 */
    copyContents(&tmpNode, node2Ptr);

    /* fix up the lru pointers (make the lru nodes corresponding to node1 and
	node2 point to the correct one) */
    node1Ptr->lruPtr->addressPtr = node1Ptr;
    node2Ptr->lruPtr->addressPtr = node2Ptr;

}

copyContents(node1Ptr, node2Ptr)
    TWOTHREENODE *node1Ptr;
    TWOTHREENODE *node2Ptr;
{
    /* copy the contents of node1 to node2 */
    node2Ptr->left = node1Ptr->left;
    node2Ptr->middle = node1Ptr->middle;
    node2Ptr->right = node1Ptr->right;
    node2Ptr->minMiddle = node1Ptr->minMiddle;
    node2Ptr->minRight = node1Ptr->minRight;
    node2Ptr->start = node1Ptr->start;
    node2Ptr->end = node1Ptr->end;
    node2Ptr->lruPtr = node1Ptr->lruPtr;
    node2Ptr->parent = node1Ptr->parent;
    node2Ptr->id = node1Ptr->id;

    return;
}

printLeaves(nodePtr, printAll)
    TWOTHREENODE *nodePtr;
    int printAll;
{
    if (nodePtr == NULL) {
	printf("empty tree (error if not root)\n");
	return;
    }
    if (nodePtr->left == NULL) {
	/* node is a leaf */
	printf("%d-%d, (#%d)", nodePtr->start, nodePtr->end, nodePtr->id);
	if (printAll) {
	    printf(", points to lru node #%d", nodePtr->lruPtr->id);
	}
	printf("\t");
	if (!(numLeavesPrintedAddress%LEAVESFORVIADDRESS)) {
	    printf("\n");
	    numLeavesPrintedAddress=0;
	}
	numLeavesPrintedAddress++;
    } else {
	/* node is a branch */
	if (printAll) {
	    printf("node #%d->#%d, #%d (min %d)",
		nodePtr->id, nodePtr->left->id, nodePtr->middle->id,
		nodePtr->minMiddle);
	    if (nodePtr->right != NULL) {
		printf(", #%d (min %d)", nodePtr->right->id, nodePtr->minRight);
	    }
	    printf("\t");
	}
	printLeaves(nodePtr->left, printAll);
	printLeaves(nodePtr->middle, printAll);
	if (nodePtr->right != NULL) {
	    printLeaves(nodePtr->right, printAll);
	}
    }
}

checkAddressRoot(rootPtr, numLeavesCorrect)
    TWOTHREENODE *rootPtr;
    int numLeavesCorrect;
{
    unsigned int min;
    int depth;
    int numLeaves;

    numLeaves = 0;

    if (rootPtr != NULL) {
	lastLeaf = 0;
	numLeaves = checkAddress(rootPtr, &min, &depth);
	/*
	printf("\nmin=%d, depth=%d\n", min, depth);
	*/
	if (rootPtr->parent != NULL) {
	    printf("parent pointer of root is not NULL\n");
	    exit(1);
	}
    } else {
	/*
	printf("empty tree\n");
	*/
    }
    if (numLeaves != numLeavesCorrect) {
	printf("wrong number of leaves %d (should be %d)\n",
	    numLeaves, numLeavesCorrect);
	exit(1);
    }
}

checkAddress(nodePtr, minPtr, depthPtr)
    TWOTHREENODE *nodePtr;
    unsigned int *minPtr;
    int *depthPtr;
{
    /* check for:
	right ordering of leaves (min -> max)
	minimum of middle subtree is indeed the minimum
	minimum of right subtree is indeed the minimum
	depth of left, middle, and right subtrees are all equal
	*** parent pointer points back to right place ***
	pass back the number of leaves in this tree
    */

    unsigned int minLeft;
    int depthLeft, depthMiddle, depthRight;
    unsigned int minMiddle;
    unsigned int minRight;
    int numLeaves;

    numLeaves = 0;

    if (nodePtr == NULL) {
	printf("checkAddress got passed a NULL pointer\n");
	printf("this might mean an empty root\n");
	exit(1);
    }

    if (nodePtr->left == NULL) {
	/* node is a leaf */
	numLeaves = 1;

	/* check ordering of leaves */
	if (nodePtr->start < lastLeaf) {
	    printf("error: leaves are ordered wrong\n");
	    printf("node #%d\n", nodePtr->id);
	    exit(1);
	}
	/*
	printf("%d-%d (#%d)\t", nodePtr->start, nodePtr->end, nodePtr->id);
	*/
	/* this forces the series of starting-ending address extents to
	    be non-overlapping */
	lastLeaf = nodePtr->end;
	*minPtr = nodePtr->start;
	*depthPtr = 1;

	/* check to make sure the lruPtr points to a lru node with the
	    same starting and ending address */
	/* note that if checkAddress is called before the lru tree is
	    updated, this will give a false alarm */
	if (nodePtr->start != nodePtr->lruPtr->start ||
		nodePtr->end != nodePtr->lruPtr->end) {
	    printf("address node (#%d) starting and ending address [%d, %d] but lru node (#%d) starting and ending address [%d, %d]\n",
		nodePtr->id, nodePtr->start, nodePtr->end,
		nodePtr->lruPtr->id, nodePtr->lruPtr->start,
		nodePtr->lruPtr->end);
	    exit(1);
	}
    } else {
	/* node is a branch */
	if (nodePtr->left->parent != nodePtr) {
	    printf("left branch doesn't point back to parent, parent node #%d\n", nodePtr->id);
	    exit(1);
	}
	numLeaves+=checkAddress(nodePtr->left, &minLeft, &depthLeft);
	if (nodePtr->middle->parent != nodePtr) {
	    printf("middle branch doesn't point back to parent, parent node #%d\n", nodePtr->id);
	    exit(1);
	}
	numLeaves+=checkAddress(nodePtr->middle, &minMiddle, &depthMiddle);
	if (minMiddle != nodePtr->minMiddle) {
	    printf("minMiddle doesn't match the minimum leaf of the middle branch (#%d)\n", nodePtr->id);
	    exit(1);
	}

	if (depthLeft != depthMiddle) {
	    printf("depth of left branch is %d, depth of middle branch is %d, node #%d\n", depthLeft, depthMiddle);
	    exit(1);
	}
	if (nodePtr->right != NULL) {
	    if (nodePtr->right->parent != nodePtr) {
		printf("right branch doesn't point back to parent, parent node #%d\n", nodePtr->id);
		exit(1);
	    }
	    numLeaves+=checkAddress(nodePtr->right, &minRight, &depthRight);
	    if (minRight != nodePtr->minRight) {
		printf("nodePtr->minRight at node #%d doesn't match the minimum leaf of the right branch\n", nodePtr->id);
		printf("nodePtr->minRight = %d, minimum leaf of right branch = %d\n",
		    nodePtr->minRight, minRight);
		exit(1);
	    }
	    if (depthLeft != depthRight) {
		printf("depth of left branch is %d, depth of right branch is %d, node #%d\n", depthLeft, depthRight);
		exit(1);
	    }
	}
	*minPtr = minLeft;
	*depthPtr = depthLeft + 1;
    }
    return(numLeaves);
}

findMissAddressSub(startPtr, endPtr, maxAddress, rootPtr)
    unsigned int *startPtr;
    unsigned int *endPtr;
    unsigned int maxAddress;
    TWOTHREENODE *rootPtr;
{
    /* randomly walk the address tree.  When you arrive at a leaf, search
	for its nearest left neighbor.
	returns the starting and ending address of an extent that's not used.
	If not able to find a miss address, return an extent with end < start
    */

    TWOTHREENODE *nodePtr, *nodeLeftPtr, *nodeRightPtr;
    int i;
    TWOTHREENODE *findNeighborAddress();

    nodePtr = rootPtr;

    if (nodePtr == NULL) {
	/* empty tree */
	/* return the entire file as an extent */
	*startPtr = 0;
	*endPtr = maxAddress;
	return;
    }

    while (nodePtr->left != NULL) {
	if (nodePtr->right == NULL) {
	    i = myRandom()%2;
	} else {
	    i = myRandom()%3;
	}
	if (i==0) {
	    /* take left branch */
	    nodePtr = nodePtr->left;
	} else if (i==1) {
	    /* take middle branch */
	    nodePtr = nodePtr->middle;
	} else {
	    /* take right branch */
	    nodePtr = nodePtr->right;
	}
    }

    /* test for hitting the beginning of the file */
    /*
    if (nodePtr->start == 0) {
	*startPtr = 0;
	*endPtr = 1;
	return;
    }
    */
    /*
    printf("got to node #%d [%u, %u]\n", nodePtr->id, nodePtr->start,
	nodePtr->end);
    */

    /* now find the nearest left or right neighbor (choose which neighbor
	randomly) */
    nodeLeftPtr = findNeighborAddress(nodePtr, 1);
    nodeRightPtr = findNeighborAddress(nodePtr, 0);

    /*
    if (nodeLeftPtr == NULL) {
	printf("no left neighbor\n");
    }
    if (nodeRightPtr == NULL) {
	printf("no right neighbor\n");
    }
    */
    if (nodeLeftPtr == NULL && nodeRightPtr == NULL) {
	/* only one leaf, choose which way to go evenly */
	if (myRandom()%2 && nodePtr->start != 0) {
	    /* go left */
	    *startPtr = 0;
	    *endPtr = nodePtr->start - 1;
	} else if (nodePtr->end < maxAddress) {
	    /* go right */
	    *startPtr = nodePtr->end + 1;
	    *endPtr = maxAddress;
	} else {
	    /* only one extent, and it touches the entire file */
	    *startPtr = 0;
	    *endPtr = maxAddress;
	}
	return;
    }

    if (nodeLeftPtr == NULL) {
	/* at the left-most edge of the tree */
	/* with probability 2/3, choose the extent [0, x] */
	if ((myRandom()%3 || nodePtr->end >= maxAddress) && nodePtr->start != 0) {
	    /* go left */
	    *startPtr = 0;
	    *endPtr = nodePtr->start - 1;
	} else if (nodePtr->end < maxAddress) {
	    /* go right */
	    *startPtr = nodePtr->end + 1;
	    *endPtr = nodeRightPtr->start - 1;
	} else {
	    *startPtr = 0;
	    *endPtr = 1;
	}
	return;
    }

    if (nodeRightPtr == NULL) {
	/* at the right-most edge of the tree */
	/* with probability 2/3, choose the extent [x, maxAddress] */
	if ((myRandom()%3 || nodePtr->start == 0) && nodePtr->end < maxAddress) {
	    /* go right */
	    *startPtr = nodePtr->end + 1;
	    *endPtr = maxAddress;
	} else if (nodePtr->start > 0) {
	    /* go left */
	    *startPtr = nodeLeftPtr->end + 1;
	    *endPtr = nodePtr->start - 1;
	} else {
	    *startPtr = 0;
	    *endPtr = 1;
	}
	return;
    }

    /* we have both a right and left neighbor to choose from */

    /* printf("have both neighbors, choose randomly\n"); */

    if (myRandom()%2) {
	/* go left */
	*startPtr = nodeLeftPtr->end + 1;
	*endPtr = nodePtr->start - 1;
    } else {
	/* go right */
	*startPtr = nodePtr->end + 1;
	*endPtr = nodeRightPtr->start - 1;
    }
    return;
}

TWOTHREENODE *findNeighborAddress(nodePtr, leftFlag)
    TWOTHREENODE *nodePtr;
    int leftFlag; /* searching left or not (ie. right) */
{
    /* search for this node's nearest left or right neighbor */
    /* all comments are as if leftFlag = 1.  if you're searching for the
	right neighbor, switch the direction of most comments */

    /* if successful, return the neighboring node,
       if unsuccessful, return NULL */

    int child;

    /* check to make sure this node is not garbage */
    if (nodePtr == NULL) {
	printf("error: findNeighborAddress is NULL\n");
	exit(1);
    }

    /* check to make sure this node is a leaf */
    if (nodePtr->left != NULL) {
	printf("error: findNeighborAddress got passed a non-leaf node #%d\n", nodePtr->id);
	exit(1);
    }

    /* first go up the tree.  Look for going up a non-left branch.  Stop if
	you hit the root (means that the first node was the left-most node */
    child = leftFlag?1:3;
    while (nodePtr->parent != NULL &&
	( (leftFlag && child == 1) ||
	  (!leftFlag && (child == 3 || (child == 2 && nodePtr->right == NULL)))
	)   )
    {
	if (nodePtr->parent->left == nodePtr) {
	    child = 1;
	} else if (nodePtr->parent->middle == nodePtr) {
	    child = 2;
	} else if (nodePtr->parent->right == nodePtr) {
	    child = 3;
	} else {
	    printf("error in findNeighborAddress: node #%d has no parent that points back to itself\n", nodePtr->id);
	    exit(1);
	}
	
	nodePtr = nodePtr->parent;
    }

    if (nodePtr->parent == NULL &&
	( (leftFlag && child == 1) ||
	  (!leftFlag && (child == 3 || (child == 2 && nodePtr->right == NULL)))
	  ) ) {
	/* reached the root without finding a node with a branch to the left 
	    of what I came up */
	return(NULL);
    }

    /*
    printf("got to a node with a branch in the correct direction\n");
    printf("node #%d points to #%d, #%d", nodePtr->id, nodePtr->left->id,
	nodePtr->middle->id);
    if (nodePtr->right != NULL) {
	printf(", #%d", nodePtr->right->id);
    }
    printf("\n");
    */

    /* got to a node which has a branch to the left of the branch we came up */
    if (leftFlag) {
	if (child == 2) {
	    nodePtr = nodePtr->left;
	} else if (child == 3) {
	    nodePtr = nodePtr->middle;
	} else {
	    printf("error in findNeighborAddress: child=%d\n", child);
	    exit(1);
	}
    } else {
	/* go down one to the right */
	if (child == 1) {
	    nodePtr = nodePtr->middle;
	} else if (child == 2) {
	    nodePtr = nodePtr->right;
	    if (nodePtr == NULL) {
		printf("error in findNeighborAddress, nodePtr became NULL\n");
		exit(1);
	    }
	} else {
	    printf("error in findNeighborAddress: child=%d\n", child);
	    exit(1);
	}
    }

    /* now go down the tree, staying to the right as much as possible */
    while (nodePtr->left != NULL) {
	if (leftFlag) {
	    if (nodePtr->right != NULL) {
		nodePtr = nodePtr->right;
	    } else {
		nodePtr = nodePtr->middle;
	    }
	} else {
	    /* keep left as much as possible */
	    nodePtr = nodePtr->left;
	}
    }
    return(nodePtr);
}

freeAddress(nodePtr)
    TWOTHREENODE *nodePtr;
{
    if (nodePtr == NULL) {
	return;
    }
    if (nodePtr->left != NULL) {
	/* node is a branch */

	/* first free all nodes underneath this node */
	freeAddress(nodePtr->left);
	freeAddress(nodePtr->middle);
	if (nodePtr->right != NULL) {
	    freeAddress(nodePtr->right);
	}
    }
    /* now free the node itself */
    free((char *)nodePtr);
}
