/* implement an lru 2-3 tree for calculating the locality of a trace */
#include <stdio.h>
#include "tree.h"

#define HITTRIES 100
#define HITFITGOODNESS .5
#define LEAVESFORVILRU 5

int lruNodeId=0; /* ok to have automatically initialized */
int numLeavesPrintedLru=0; /* ok to have automatically initialized */

char *malloc();

TWOTHREENODELRU *insertLru(rootPtr, newNodePtr, bytesAhead)
    TWOTHREENODELRU *rootPtr;
    TWOTHREENODELRU *newNodePtr;
    unsigned int bytesAhead;
{
    TWOTHREENODELRU *tmpPtr, *tmpPtr1;

    TWOTHREENODELRU *insertLruSub();
    unsigned int bytes();

    if (rootPtr == NULL) {
	/* the root is empty */
	if (bytesAhead > 0) {
	    printf("inserted on an empty tree with bytesAhead = %d\n",
		bytesAhead);
	    exit(1);
	}
	/* set rootPtr to the newNode */
	/*
	printf("root is empty\n");
	printf("setting root node to be a leaf (#%d)\n", newNodePtr->id);
	*/
	newNodePtr->parent = NULL;
	return (newNodePtr);
    }

    if (rootPtr->left == NULL) {
	/* root is a leaf */

	/* printf("root is a leaf (#%d)\n", rootPtr->id); */

	/* make a new root which points to the two leaves (the old root
	    node and the newNode) */
	tmpPtr = (TWOTHREENODELRU *) malloc(sizeof(TWOTHREENODELRU));
	tmpPtr->id = lruNodeId++;
	if (bytesAhead == 0) {
	    tmpPtr->left = newNodePtr;
	    tmpPtr->middle = rootPtr;
	} else {
	    tmpPtr->left = rootPtr;
	    tmpPtr->middle = newNodePtr;
	}
	tmpPtr->bytesLeft = bytes(tmpPtr->left);
	tmpPtr->bytesMiddle = bytes(tmpPtr->middle);

	tmpPtr->left->parent = tmpPtr->middle->parent = tmpPtr;
	tmpPtr->parent = NULL;
	tmpPtr->right = NULL;
	/*
	printf("set root to be a branch (#%d) pointing to %d (#%d) and %d (#%d)\n",
	    tmpPtr->id, tmpPtr->left->start, tmpPtr->left->id, tmpPtr->middle->start, tmpPtr->middle->id);
	*/
	return(tmpPtr);
    }

    /* root is normal (a parent of 2 or 3 nodes, which may be leaves) */
    /* printf("root is a normal branch\n"); */
    tmpPtr1 = insertLruSub(rootPtr, newNodePtr, bytesAhead);
    /* printf("\tinsertLruSub returned to insertLru\n"); */

    if (tmpPtr1 != NULL) {
	/* printf("\treturned a new node (#%d)\n", tmpPtr1->id); */
	/* create new root which points to old root and tmpPtr */
	tmpPtr = (TWOTHREENODELRU *) malloc(sizeof(TWOTHREENODELRU));
	tmpPtr->id = lruNodeId++;

	/* I don't need to figure out which side to put the new node, since
	    the new node insertLruSub returns always goes to the right of
	    the node insertLruSub was called with */
	tmpPtr->left = rootPtr;
	tmpPtr->middle = tmpPtr1;
	tmpPtr->bytesLeft = bytes(tmpPtr->left);
	tmpPtr->bytesMiddle = bytes(tmpPtr->middle);
	tmpPtr->parent = NULL;
	tmpPtr->right = NULL;
	tmpPtr->left->parent = tmpPtr->middle->parent = tmpPtr;
	/* printf("creating new root (#%d)\n", tmpPtr->id); */
	return (tmpPtr);
    }

    /* no new root */
    /*
    printf("returned NULL, no new root\n");
    printf("root now points to #%d (bytes %d), #%d (bytes %d)",
	rootPtr->left->id, rootPtr->bytesLeft, rootPtr->middle->id,
	rootPtr->bytesMiddle);
    if (rootPtr->right != NULL) {
        printf(", #%d (bytes %d)", rootPtr->right->id, rootPtr->bytesRight);
    }
    printf("\n");
    */
    return(rootPtr);
}

TWOTHREENODELRU *insertLruSub(nodePtr, newNodePtr, bytesAhead)
    TWOTHREENODELRU *nodePtr;
    TWOTHREENODELRU *newNodePtr;
    unsigned int bytesAhead;
{
    TWOTHREENODELRU *newPtr;
    TWOTHREENODELRU *tmpPtr;
    int child;

    unsigned int bytes();

    /* if insertLruSub returns a new node, it always goes to the right
	of node */

    /*
    printf("called insertLruSub on node %d with bytesAhead %d\n", nodePtr->id,
		bytesAhead);
    */

    if (nodePtr->left == NULL) {
	/* node is a leaf */
	/* make sure node returned goes to the right of node */

	/* check to make sure the new node will go completely to one side
	    of the old node */
	if (bytesAhead != 0 && bytesAhead != bytes(nodePtr)) {
	    printf("error: new request would split a leaf (#%d), bytesAhead = %d\n",
		nodePtr->id, bytesAhead);
	    exit(1);
	}
	if (bytesAhead == 0) {
	    swapContentsLru(newNodePtr, nodePtr);
	}

	/* return newNode */
	/* *minStartPtr = newNodePtr->start; */
	return(newNodePtr);
    }

    /* node is a branch */

    /* which child to follow */
    if (bytesAhead <= nodePtr->bytesLeft) {
	/* follow left branch */
	child = 1;
	tmpPtr = insertLruSub(nodePtr->left, newNodePtr, bytesAhead);
    } else if (nodePtr->right == NULL ||
	    bytesAhead <= nodePtr->bytesLeft + nodePtr->bytesMiddle) {
	/* follow middle branch */
	child = 2;
	tmpPtr = insertLruSub(nodePtr->middle, newNodePtr,
			    bytesAhead - nodePtr->bytesLeft);
    } else {
	/* follow right branch */
	child = 3;
	tmpPtr = insertLruSub(nodePtr->right, newNodePtr,
		    bytesAhead - nodePtr->bytesLeft - nodePtr->bytesMiddle);
    }

    if (tmpPtr == NULL) {
	/*
	printf("\tsub returned to sub with no new node\n");
	*/
	/* no new children need to be added to node */
	/* don't have to worry about changing the minMiddle, minRight values
	    of node, since they will never change.  E.g. if the new start
	    was less than minMiddle, you would have followed the left branch,
	    not the middle branch */

	/* fix the byte totals */
	fixBytes(nodePtr);
	return(NULL);
    }

    /* need to add a new child to node */
    /*
    printf("\tsub returned to sub with a new node to add on (#%d)\n",
	tmpPtr->id);
    */
    if (nodePtr->right == NULL) {
	/* node only has 2 children */
	/* printf("\t\tnode has two children--adding the third\n"); */
	if (child == 1) {
	    /* followed the left branch */
	    nodePtr->right = nodePtr->middle;
	    nodePtr->middle = tmpPtr;
	} else if (child == 2) {
	    /* followed middle branch */
	    nodePtr->right = tmpPtr;
	} else {
	    printf("error: took right branch, but only had 2 children\n");
	    exit(1);
	}
	fixBytes(nodePtr);
	nodePtr->left->parent = nodePtr->middle->parent =
	    nodePtr->right->parent = nodePtr;
	return(NULL);
    }

    /* node already has 3 children */

    /* create new node to pass up to caller of insertLruSub */
    newPtr = (TWOTHREENODELRU *) malloc(sizeof(TWOTHREENODELRU));
    newPtr->id = lruNodeId++;

    /*
    printf("\t\tnode already has 3 children\n");
    printf("\t\tcreated new node #%d\n", newPtr->id);
    */

    if (child == 3) {
	/* right child and tmpPtr become children of new node */
	newPtr->left = nodePtr->right;
	newPtr->middle = tmpPtr;
	newPtr->right = NULL;
	nodePtr->right = NULL;
    } else {
	/* child is left or middle */

	/* move right child of node to middle of the new node */
	newPtr->middle = nodePtr->right;
	newPtr->right = NULL;
	nodePtr->right = NULL;

	if (child == 2) {
	    /* followed the middle branch */

	    /* connect returned node to left branch of new node */
	    newPtr->left = tmpPtr;
	} else {
	    /* followed left branch */

	    /*
	    printf("(followed left branch)\n");
	    */

	    /* node's middle child becomes left branch of new node */
	    newPtr->left = nodePtr->middle;

	    /* returned node becomes middle branch of node */
	    nodePtr->middle = tmpPtr;
	}
    }
    fixBytes(nodePtr);
    fixBytes(newPtr);
    newPtr->left->parent = newPtr->middle->parent = newPtr;
    nodePtr->left->parent = nodePtr->middle->parent = nodePtr;
    return(newPtr);
}

TWOTHREENODELRU *deleteLru(rootPtr, bytesAhead)
    TWOTHREENODELRU *rootPtr;
    unsigned int bytesAhead;
{
    int onlyOne; /* holds status from deleteLruSub call */
    TWOTHREENODELRU *newRootPtr;

    /* returns the new root */
    if (rootPtr == NULL) {
	/* root is empty */
	printf("error: called delete on an empty tree\n");
	exit(1);
    }

    if (rootPtr->left == NULL) {
	/* root is a leaf */
	if (bytesAhead == 0) {
	    /* found the match */
	    return(NULL);
	} else {
	    printf("tried to delete entry at bytesAhead %d, but tree was a leaf\n", bytesAhead);
	    exit(1);
	}
    }

    /* root is a normal branch */
    /* printf("root is a normal branch\n"); */
    onlyOne = deleteLruSub(rootPtr, bytesAhead);

    /* printf("sub returned to deleteLru with onlyOne=%d\n", onlyOne); */
    if (onlyOne) {
	/* root only has one child left */
	/* 
	printf("root only has one child left--making that child the root\n");
	*/

	/* free the old root */
	newRootPtr = rootPtr->left;
	free((char *)rootPtr);

	newRootPtr->parent = NULL;
	return(newRootPtr);
    }

    /* printf("leaving the root unchanged\n"); */
    return(rootPtr);
}

int deleteLruSub(nodePtr, bytesAhead)
    TWOTHREENODELRU *nodePtr;
    unsigned int bytesAhead;
{
    int onlyOne; /* holds status from deleteLruSub call */
    int child; /* which child was followed */

    /*
    printf("called sub on #%d with bytesAhead %d\n", nodePtr->id, bytesAhead);
    */

    if (nodePtr->left->left == NULL) {
	/* children of nodePtr are leaves */
	/* search for start among those leaves */
	/* printf("children of node are leaves\n"); */
	if (bytesAhead == 0) {
	    /* matched on left child */
	    free((char *)nodePtr->left);
	    nodePtr->left = nodePtr->middle;
	    nodePtr->middle = nodePtr->right;
	    nodePtr->right = NULL;
	} else if (bytesAhead == nodePtr->bytesLeft) {
	    /* matched on middle child */
	    free((char *)nodePtr->middle);
	    nodePtr->middle = nodePtr->right;
	    nodePtr->right = NULL;
	} else if (nodePtr->right != NULL &&
		bytesAhead == nodePtr->bytesLeft + nodePtr->bytesMiddle) {
	    /* matched on right child */
	    free((char *)nodePtr->right);
	    nodePtr->right = NULL;
	} else {
	    printf("couldn't find a match for bytesAhead = %d\n", bytesAhead);
	    exit(1);
	}
	fixBytes(nodePtr);

	/* if node only has one child, return TRUE */
	return( (nodePtr->middle == NULL) );
    }

    /* children of nodePtr are branches */

    /*
    printf("children of node are branches\n");
    */
    /* figure out which of the children of nodePtr to follow */
    if (bytesAhead < nodePtr->bytesLeft) {
	/* follow left branch */
	child = 1;
	onlyOne = deleteLruSub(nodePtr->left, bytesAhead);
    } else if (nodePtr->right == NULL ||
	    bytesAhead < nodePtr->bytesLeft + nodePtr->bytesMiddle) {
	/* follow middle branch */
	child = 2;
	onlyOne = deleteLruSub(nodePtr->middle,
		bytesAhead - nodePtr->bytesLeft );
    } else {
	/* follow right branch */
	child = 3;
	onlyOne = deleteLruSub(nodePtr->right,
		    bytesAhead - nodePtr->bytesLeft - nodePtr->bytesMiddle);
    }

    /*
    printf("sub returned to sub (nodePtr=#%d, with onlyOne=%d\n",
	nodePtr->id, onlyOne);
    */

    if (onlyOne) {
	if (child == 1) {
	    /* followed the left branch */
	    /* printf("fixing up left branch\n"); */
	    if (nodePtr->middle->right != NULL) {
		/* middle child of node has 3 children */
		/* make left child of the middle child be the middle child
		    of the left */
		nodePtr->left->middle = nodePtr->middle->left;

		nodePtr->middle->left = nodePtr->middle->middle;
		nodePtr->middle->middle = nodePtr->middle->right;
		nodePtr->middle->right = NULL;
		
		nodePtr->left->middle->parent = nodePtr->left;

		fixBytes(nodePtr->left);
		fixBytes(nodePtr->middle);
		fixBytes(nodePtr);

		return(0); /* still has enough children */
	    } else {
		/* middle child only has 2 children */
		/* transfer 2 children of middle branch over to left branch */
		nodePtr->left->middle = nodePtr->middle->left;
		nodePtr->left->right = nodePtr->middle->middle;

		/* free middle child */
		free((char *)nodePtr->middle);

		/* move right child (if there is one) to the middle child */
		nodePtr->middle = nodePtr->right;
		nodePtr->right = NULL;

		nodePtr->left->middle->parent = nodePtr->left->right->parent =
		    nodePtr->left;

		fixBytes(nodePtr->left);
		fixBytes(nodePtr->middle);
		fixBytes(nodePtr);
		return( (nodePtr->middle == NULL) ); /* only one child left */
	    }
	} else if (child == 2) {
	    /* followed the middle branch */
	    /* printf("fixing up middle branch\n"); */
	    if (nodePtr->left->right != NULL) {
		/* left child of node has 3 children */
		/* printf("grabbing a child from the left neighbor\n"); */
		/* make right child of left node be left child of middle */
		nodePtr->middle->right = nodePtr->middle->middle;
		nodePtr->middle->middle = nodePtr->middle->left;

		nodePtr->middle->left = nodePtr->left->right;
		nodePtr->left->right = NULL;

		nodePtr->middle->left->parent = nodePtr->middle;
		fixBytes(nodePtr->left);
		fixBytes(nodePtr->middle);
		fixBytes(nodePtr);
		return(0);
	    } else if (nodePtr->right != NULL &&
		    nodePtr->right->right != NULL) {
		/* right child of node has 3 children */
		/* printf("grabbing a child from the right neighbor\n"); */
		/* make left child of right node be middle child of middle */
		nodePtr->middle->middle = nodePtr->right->left;
		
		nodePtr->right->left = nodePtr->right->middle;
		nodePtr->right->middle = nodePtr->right->right;
		nodePtr->right->right = NULL;

		nodePtr->middle->middle->parent = nodePtr->middle;

		fixBytes(nodePtr->middle);
		fixBytes(nodePtr->right);
		fixBytes(nodePtr);
		return(0);
	    } else {
		/* no neighbors of the middle child have 3 children */
		/* add the only child of the middle child to the left child */
		/* printf("moving only child of middle to left neighbor\n"); */
		nodePtr->left->right = nodePtr->middle->left;

		/* free the old middle child */
		free((char *)nodePtr->middle);

		nodePtr->middle = nodePtr->right;
		nodePtr->right = NULL;

		nodePtr->left->right->parent = nodePtr->left;
		fixBytes(nodePtr->left);
		fixBytes(nodePtr);

		return( (nodePtr->middle == NULL) );
	    }
	} else if (child == 3) {
	    /* followed the right branch */
	    /* printf("fixing up right branch\n"); */
	    if (nodePtr->middle->right != NULL) {
		/* middle child has 3 children */
		/* make right child of middle node be the left child of the
		    right node */
		nodePtr->right->middle = nodePtr->right->left;
		nodePtr->right->left = nodePtr->middle->right;

		nodePtr->middle->right = NULL;

		nodePtr->right->left->parent = nodePtr->right;

		fixBytes(nodePtr->middle);
		fixBytes(nodePtr->right);
		fixBytes(nodePtr);

		return(0);

	    } else {
		/* middle child has 2 children */
		/* move only child of right node to the middle node */
		nodePtr->middle->right = nodePtr->right->left;

		/* free right child */
		free((char *)nodePtr->right);

		nodePtr->right = NULL;

		nodePtr->middle->right->parent = nodePtr->middle;

		fixBytes(nodePtr->middle);
		fixBytes(nodePtr);
		return(0); /* certainly has 2 children left */
	    }
	} else {
	    printf("error: child = %d\n", child);
	    exit(1);
	}
    } else { /* onlyOne = 0 */
	fixBytes(nodePtr);
	return(0);
    }
    printf("error: reached bottom of deleteLruSub\n");
    exit(1);
    return(0);
}

swapContentsLru(node1Ptr, node2Ptr)
    TWOTHREENODELRU *node1Ptr;
    TWOTHREENODELRU *node2Ptr;
{
    /* swap the contents of node1 and node2 */
    TWOTHREENODELRU tmpNode;

    /*
    printf("swapping contents of node #%d and #%d\n", node1Ptr->id,
	node2Ptr->id);
    */

    /* copy node1 to tmpNode */
    copyContentsLru(node1Ptr, &tmpNode);

    /* copy node2 to node1 */
    copyContentsLru(node2Ptr, node1Ptr);

    /* copy tmpNode to node2 */
    copyContentsLru(&tmpNode, node2Ptr);

    /* fix up the address pointers (make the address nodes corresponding to
	node1 and node2 point to the other one) */
    node1Ptr->addressPtr->lruPtr = node1Ptr;
    node2Ptr->addressPtr->lruPtr = node2Ptr;

}

copyContentsLru(node1Ptr, node2Ptr)
    TWOTHREENODELRU *node1Ptr;
    TWOTHREENODELRU *node2Ptr;
{
    /* copy the contents of node1 to node2 */
    node2Ptr->left = node1Ptr->left;
    node2Ptr->middle = node1Ptr->middle;
    node2Ptr->right = node1Ptr->right;
    node2Ptr->bytesLeft = node1Ptr->bytesLeft;
    node2Ptr->bytesMiddle = node1Ptr->bytesMiddle;
    node2Ptr->bytesRight = node1Ptr->bytesRight;
    node2Ptr->start = node1Ptr->start;
    node2Ptr->end = node1Ptr->end;
    node2Ptr->file = node1Ptr->file;
    node2Ptr->addressPtr = node1Ptr->addressPtr;
    node2Ptr->parent = node1Ptr->parent;
    node2Ptr->id = node1Ptr->id;

    return;
}

printLeavesLru(nodePtr, printAll)
    TWOTHREENODELRU *nodePtr;
    int printAll;
{
    if (nodePtr == NULL) {
	printf("empty tree (error if not root)\n");
	return;
    }
    if (nodePtr->left == NULL) {
	/* node is a leaf */
	printf("%d-%d:%d (#%d)\t", nodePtr->start, nodePtr->end, nodePtr->file,
		nodePtr->id);
	if (!(numLeavesPrintedLru%LEAVESFORVILRU)) {
	    printf("\n");
	    numLeavesPrintedLru = 0;
	}
	numLeavesPrintedLru++;
    } else {
	/* node is a branch */
	if (printAll) {
	    printf("node #%d->#%d (bytes %d), #%d (bytes %d)",
		nodePtr->id, nodePtr->left->id, nodePtr->bytesLeft,
		nodePtr->middle->id, nodePtr->bytesMiddle);
	    if (nodePtr->right != NULL) {
		printf(", #%d (bytes %d)", nodePtr->right->id,
		    nodePtr->bytesRight);
	    }
	    printf("\t");
	}
	printLeavesLru(nodePtr->left, printAll);
	printLeavesLru(nodePtr->middle, printAll);
	if (nodePtr->right != NULL) {
	    printLeavesLru(nodePtr->right, printAll);
	}
    }
}

checkLruRoot(rootPtr, numLeavesCorrect, maxFileNumSeen)
    TWOTHREENODELRU *rootPtr;
    int numLeavesCorrect;
    int maxFileNumSeen;
{
    int depth;
    int numLeaves;
    unsigned int bytes;

    numLeaves = 0;

    if (rootPtr != NULL) {
	numLeaves = checkLru(rootPtr, &bytes, &depth, maxFileNumSeen);
	/*
	printf("\ndepth=%d\n", depth);
	*/
	if (rootPtr->parent != NULL) {
	    printf("parent pointer of root is not NULL\n");
	    exit(1);
	}
    } else {
	/*
	printf("empty tree\n");
	*/
    }
    if (numLeaves != numLeavesCorrect) {
	printf("wrong number of leaves %d (should be %d)\n",
	    numLeaves, numLeavesCorrect);
	exit(1);
    }
}

int checkLru(nodePtr, bytesPtr, depthPtr, maxFileNumSeen)
    TWOTHREENODELRU *nodePtr;
    unsigned int *bytesPtr;
    int *depthPtr;
    int maxFileNumSeen;
{
    /* check for:
	byte totals for each subtree is correct
	depth of left, middle, and right subtrees are all equal
	parent pointer points back to right place
	pass back the number of leaves in this tree
    */

    int depthLeft;
    int depthMiddle;
    int depthRight;
    int numLeaves;

    unsigned int bytes; /* temporary variable for when I do the recursive
			    call */

    numLeaves = 0;

    if (nodePtr == NULL) {
	printf("checkLru got passed a NULL pointer\n");
	printf("this might mean an empty root\n");
	exit(1);
    }

    if (nodePtr->left == NULL) {
	/* node is a leaf */
	numLeaves = 1;

	/*
	printf("%d-%d (#%d)\t", nodePtr->start, nodePtr->end, nodePtr->id);
	*/
	*depthPtr = 1;
	*bytesPtr = nodePtr->end - nodePtr->start + 1;

	/* check to make sure the address leaf this points at has the
	    same starting and ending addresses */
	if (nodePtr->start != nodePtr->addressPtr->start ||
	    nodePtr->end != nodePtr->addressPtr->end) {
	    printf("error: extent for lru leaf #%d is [%d, %d], but extent for corresponding address leaf #%d is [%d, %d]\n",
		nodePtr->id, nodePtr->start, nodePtr->end,
		nodePtr->addressPtr->id, nodePtr->addressPtr->start,
		nodePtr->addressPtr->end);
	    exit(1);
	}
	if (nodePtr->file > maxFileNumSeen) {
	    printf("error: file number for node #%d too large (%d), max seen is %d\n",
		nodePtr->id, nodePtr->file, maxFileNumSeen);
	    exit(1);
	}
    } else {
	/* node is a branch */
	if (nodePtr->bytesLeft == 0) {
	    printf("bytesLeft = %d at node #%d\n", nodePtr->bytesLeft,
		    nodePtr->id);
	    exit(1);
	}

	if (nodePtr->bytesMiddle == 0) {
	    printf("bytesMiddle = %d at node #%d\n", nodePtr->bytesMiddle,
		    nodePtr->id);
	    exit(1);
	}

	if (nodePtr->left->parent != nodePtr) {
	    printf("left branch doesn't point back to parent, parent node #%d\n", nodePtr->id);
	    exit(1);
	}
	numLeaves+=checkLru(nodePtr->left, &bytes, &depthLeft, maxFileNumSeen);
	(*bytesPtr) = bytes;
	if (bytes != nodePtr->bytesLeft) {
	    printf("bytesLeft (%d) doesn't match the number of bytes summed up(%d) (#%d)\n", nodePtr->bytesLeft, bytes, nodePtr->id);
	    exit(1);
	}
	if (nodePtr->middle->parent != nodePtr) {
	    printf("middle branch doesn't point back to parent, parent node #%d\n", nodePtr->id);
	    exit(1);
	}
	numLeaves+=checkLru(nodePtr->middle, &bytes, &depthMiddle, maxFileNumSeen);
	(*bytesPtr)+=bytes;
	if (bytes != nodePtr->bytesMiddle) {
	    printf("bytesMiddle (%d) doesn't match the number of bytes summed up(%d) (#%d)\n", nodePtr->bytesMiddle, bytes, nodePtr->id);
	    exit(1);
	}

	if (depthLeft != depthMiddle) {
	    printf("depth of left branch is %d, depth of middle branch is %d, node #%d\n", depthLeft, depthMiddle);
	    exit(1);
	}
	if (nodePtr->right != NULL) {
	    if (nodePtr->right->parent != nodePtr) {
		printf("right branch doesn't point back to parent, parent node #%d\n", nodePtr->id);
		exit(1);
	    }
	    numLeaves+=checkLru(nodePtr->right, &bytes, &depthRight, maxFileNumSeen);
	    (*bytesPtr)+=bytes;
	    if (bytes != nodePtr->bytesRight) {
		printf("bytesRight (%d) doesn't match the number of bytes summed up(%d) (#%d)\n", nodePtr->bytesRight, bytes, nodePtr->id);
		exit(1);
	    }
	    if (depthLeft != depthRight) {
		printf("depth of left branch is %d, depth of right branch is %d, node #%d\n", depthLeft, depthRight);
		exit(1);
	    }
	}
	*depthPtr = depthLeft + 1;
    }
    return(numLeaves);
}

unsigned int bytes(nodePtr)
    TWOTHREENODELRU *nodePtr;
{
    /* return the number of bytes under this node */
    unsigned int total;

    if (nodePtr == NULL) {
	printf("error: nodePtr = NULL in the function bytes\n");
	exit(1);
    }

    if (nodePtr->left == NULL) {
	/* this is a leaf */
	total = nodePtr->end - nodePtr->start + 1;
    } else {
	/* this is a branch */
	total = nodePtr->bytesLeft;
	if (nodePtr->middle != NULL) {
	    total += nodePtr->bytesMiddle;
	}
	if (nodePtr->right != NULL) {
	    total += nodePtr->bytesRight;
	}
    }

    return(total);
}

fixBytes(nodePtr)
    TWOTHREENODELRU *nodePtr;
{
    /* fix the byte totals at node */
    unsigned int bytes();

    if (nodePtr != NULL) {
	nodePtr->bytesLeft = bytes(nodePtr->left);
	if (nodePtr->middle != NULL) {
	    nodePtr->bytesMiddle = bytes(nodePtr->middle);
	}
	if (nodePtr->right != NULL) {
	    nodePtr->bytesRight = bytes(nodePtr->right);
	}
	/*
	printf("fixing byte totals at node #%d to %d, %d",
	    nodePtr->id, nodePtr->bytesLeft, nodePtr->bytesMiddle);
	if (nodePtr->right != NULL) {
	    printf(", %d", nodePtr->bytesRight);
	}
	printf("\n");
	*/
    }
    return;
}

unsigned int changeByte(leafPtr, delta)
    TWOTHREENODELRU *leafPtr;
    unsigned int delta;
{
    /* returns the number of bytes ahead of the leaf specified */
    /* updates the byte totals on the way from the leaf to the root (subtracts
	delta from all nodes on the way) */

    TWOTHREENODELRU *nodePtr;

    int child;
    unsigned int bytesAhead;

    /*
    printf("called changeByte on node #%d\n", leafPtr->id);
    */

    child = 0;
    bytesAhead = 0;

    if (leafPtr->left != NULL) {
	printf("error: changeByte was passed a non-leaf (#%d)\n",
	    leafPtr->id);
	exit(1);
    }

    nodePtr = leafPtr;

    while (nodePtr != NULL) {
	/*
	printf("changeByte: child=%d\n", child);
	*/
	if (nodePtr->left != NULL) {
	    /* this is a branch--needs to be updated */
	    if (child == 1) {
		/* came from left child */
		nodePtr->bytesLeft -= delta;
	    } else if (child == 2) {
		/* came from middle child */
		nodePtr->bytesMiddle -= delta;
		bytesAhead += nodePtr->bytesLeft;
	    } else if (child == 3) {
		/* came from right child */
		nodePtr->bytesRight -= delta;
		bytesAhead += nodePtr->bytesLeft + nodePtr->bytesMiddle;
	    } else if (child == 0) {
		printf("error: child = %d\n", child);
		exit(1);
	    }
	}
	if (nodePtr->parent == NULL) {
	    /* we've reached the root */
	} else if (nodePtr->parent->left == nodePtr) {
	    child = 1;
	} else if (nodePtr->parent->middle == nodePtr) {
	    child = 2;
	} else if (nodePtr->parent->right == nodePtr) {
	    child = 3;
	} else {
	    printf("error: in changeByte, node #%d has no parent that it descends from\n", nodePtr->id);
	    exit(1);
	}
	/*
	printf("changeByte: bytesAhead=%d\n", bytesAhead);
	*/
	nodePtr = nodePtr->parent;
    }
    return(bytesAhead);
}

findHitAddressSub(startPtr, endPtr, file, size, bytesAhead, rootPtr)
    unsigned int *startPtr;
    unsigned int *endPtr;
    int file;
    unsigned int size;
    unsigned int bytesAhead;
    TWOTHREENODELRU *rootPtr;
{
    /* walk the lru tree, trying to get to the lruDist specified.
	When you arrive at the leaf, see if the file is right.  If so, return.
	If not, search right, then left (alternating) looking for an
	extent that belongs to the right file.  If the search goes on too
	long (more than HITTRIES times), or runs into one of the edges,
	return an extent with end < start.
    */

    TWOTHREENODELRU *nodePtr;
    TWOTHREENODELRU *nodeRightPtr, *nodeLeftPtr;
    TWOTHREENODELRU *nodeSavePtr;
    unsigned int bytesAheadRem; /* remaining lru distance (bytes ahead) */
    int i;
    double hitEvalSave;
    double leftEval, rightEval;

    TWOTHREENODELRU *findNeighborLru();
    double hitEval();

    nodePtr = rootPtr;
    bytesAheadRem = bytesAhead;

    if (nodePtr == NULL) {
	/* empty tree */
	/* return failure */
	*startPtr = 1;
	*endPtr = 0;
	return;
    }

    if (bytesAheadRem >
	1.1 * (nodePtr->bytesLeft + nodePtr->bytesMiddle +
	    ((nodePtr->right==NULL)?0:nodePtr->bytesRight)) )
    {
	/* lru tree is too small to accomodate this request (out of the
	    tree + 10% */
#ifdef DEBUG
	printf("lru tree is too small (%u) to accomodate this request\n",
	    nodePtr->bytesLeft + nodePtr->bytesMiddle +
	    ((nodePtr->right==NULL)?0:nodePtr->bytesRight));
#endif
	/* return failure */
	*startPtr = 1;
	*endPtr = 0;
	return;
    }

    while (nodePtr->left != NULL) {
	if (bytesAheadRem < nodePtr->bytesLeft) {
	    nodePtr = nodePtr->left;
	} else if (nodePtr->right == NULL ||
		    bytesAheadRem < nodePtr->bytesLeft + nodePtr->bytesMiddle) {
	    nodePtr = nodePtr->middle;
	    bytesAheadRem -= nodePtr->bytesLeft;
	} else {
	    nodePtr = nodePtr->right;
	    bytesAheadRem -= nodePtr->bytesLeft + nodePtr->bytesMiddle;
	}
    }

    /* now node is a leaf.  Now find the nearest extent of this file */
#ifdef DEBUG
    printf("first leaf arrived at was #%d:%d, [%u, %u]\n",
	nodePtr->id, nodePtr->file, nodePtr->start, nodePtr->end);
#endif

    nodeRightPtr = nodePtr;
    nodeLeftPtr = nodePtr;
    nodeSavePtr = nodePtr;
    hitEvalSave = hitEval(nodePtr, file, size);
    for (i=0; i<HITTRIES && hitEvalSave<HITFITGOODNESS; i++) {

	/* search for the neighbors of node */
	if (nodeRightPtr != NULL) {
	    nodeRightPtr = findNeighborLru(nodeRightPtr, 0);
	}
	if (nodeLeftPtr != NULL) {
	    nodeLeftPtr = findNeighborLru(nodeLeftPtr, 1);
	}

	/*
	if (nodeLeftPtr != NULL) {
	    printf("left neighbor: #%d:%d [%u, %u]\n", nodeLeftPtr->id,
		nodeLeftPtr->file, nodeLeftPtr->start, nodeLeftPtr->end);
	}
	if (nodeRightPtr != NULL) {
	    printf("right neighbor: #%d:%d [%u, %u]\n", nodeRightPtr->id,
		nodeRightPtr->file, nodeRightPtr->start, nodeRightPtr->end);
	}
	*/

	if (nodeRightPtr == NULL && nodeLeftPtr == NULL) {
	    /* break out of this loop */
	    break;
	}

	leftEval = hitEval(nodeLeftPtr, file, size);
	rightEval = hitEval(nodeRightPtr, file, size);

	if (hitEvalSave < leftEval) {
	    hitEvalSave = leftEval;
	    nodeSavePtr = nodeLeftPtr;
	}

	if (hitEvalSave < rightEval) {
	    hitEvalSave = rightEval;
	    nodeSavePtr = nodeRightPtr;
	}
    }

    if (hitEvalSave > 0) {
	*startPtr = nodeSavePtr->start;
	*endPtr = nodeSavePtr->end;
    } else {
#ifdef DEBUG
	printf("couldn't find a big enough hit in the vicinity, switching this to a miss\n");
#endif
	*startPtr = 1;
	*endPtr = 0;
    }

    return;
}

TWOTHREENODELRU *findNeighborLru(nodePtr, leftFlag)
    TWOTHREENODELRU *nodePtr;
    int leftFlag; /* searching left or not (ie. right) */
{
    /* search for this node's nearest left neighbor */
    /* all comments are as if leftFlag = 1.  if you're searching for the
	right neighbor, switch the direction of most comments */

    /* if successful, return the neighboring node,
       if unsuccessful, return NULL */

    int child;

    /* check to make sure this node is not garbage */
    if (nodePtr == NULL) {
	printf("error: findNeighborLru called with NULL\n");
	exit(1);
    }

    /* check to make sure this node is a leaf */
    if (nodePtr->left != NULL) {
	printf("error: findNeighborLru got passed a non-leaf node #%d\n", nodePtr->id);
	exit(1);
    }

    /* first go up the tree.  Look for going up a non-left branch.  Stop if
	you hit the root (means that the first node was the left-most node */
    child = leftFlag?1:3;
    while (nodePtr->parent != NULL &&
	( (leftFlag && child == 1) ||
	  (!leftFlag && (child == 3 || (child == 2 && nodePtr->right == NULL)))
	)   )
    {
	if (nodePtr->parent->left == nodePtr) {
	    child = 1;
	} else if (nodePtr->parent->middle == nodePtr) {
	    child = 2;
	} else if (nodePtr->parent->right == nodePtr) {
	    child = 3;
	} else {
	    printf("error in findNeighborLru: node #%d has no parent that points back to itself\n", nodePtr->id);
	    exit(1);
	}
	
	nodePtr = nodePtr->parent;
    }

    if (nodePtr->parent == NULL &&
	( (leftFlag && child == 1) ||
	  (!leftFlag && (child == 3 || (child == 2 && nodePtr->right == NULL)))
	) ) {
	/* reached the root without finding a node with a branch to the left 
	    of what I came up */
	return(NULL);
    }

    /* got to a node which has a branch to the left of the branch we came up */
    if (leftFlag) {
	if (child == 2) {
	    nodePtr = nodePtr->left;
	} else if (child == 3) {
	    nodePtr = nodePtr->middle;
	} else {
	    printf("error in findNeighborLru: child=%d\n", child);
	    exit(1);
	}
    } else {
	/* go down one to the right */
	if (child == 1) {
	    nodePtr = nodePtr->middle;
	} else if (child == 2) {
	    nodePtr = nodePtr->right;
	    if (nodePtr == NULL) {
		printf("error in findNeighborLru, nodePtr became NULL\n");
		exit(1);
	    }
	} else {
	    printf("error in findNeighborLru: child=%d\n", child);
	    exit(1);
	}
    }

    /* now go down the tree, staying to the right as much as possible */
    while (nodePtr->left != NULL) {
	if (leftFlag) {
	    if (nodePtr->right != NULL) {
		nodePtr = nodePtr->right;
	    } else {
		nodePtr = nodePtr->middle;
	    }
	} else {
	    /* keep left as much as possible */
	    nodePtr = nodePtr->left;
	}
    }
    return(nodePtr);
}

double hitEval(nodePtr, file, size)
    TWOTHREENODELRU *nodePtr;
    int file;
    unsigned int size;
{
    /* evaluate how good a hit this node would be for this size */
    if (nodePtr == NULL) {
	return(-1.0);
    }

    if (nodePtr->left != NULL) {
	printf("error: hitEval got passed a branch #%d\n", nodePtr->id);
	exit(1);
    }

    if (nodePtr->file != file) {
	return(-1.0);
    }

    /* return the fraction of size that fits in this extent */
    return ( (nodePtr->end - nodePtr->start + 1) / (double)size);
}


freeLru(nodePtr)
    TWOTHREENODELRU *nodePtr;
{
    if (nodePtr == NULL) {
	return;
    }
    if (nodePtr->left != NULL) {
	/* node is a branch */

	/* first free all nodes underneath this node */
	freeLru(nodePtr->left);
	freeLru(nodePtr->middle);
	if (nodePtr->right != NULL) {
	    freeLru(nodePtr->right);
	}
    }
    /* now free the node itself */
    free((char *)nodePtr);
}
