#include <stdio.h>
#include "specWl.h"
#include "math.h"

#ifdef SPRITE
#include <bstring.h>
#endif /* SPRITE */

#define MIN_UNIQUEBYTE_INCR (10 * 1024) /* 10 KB */
#define MIN_REUSE_INCR .5
#define MIN_HITDEPTH_INCR .1
#define MIN_READPROB_INCR .1
#define MIN_SIZEMEAN_INCR (1 * 1024) /* 1 KB */
#define MIN_SIZECVAR_INCR .2
#define MIN_PROCESSNUM_INCR 1
#define MIN_SHARING_INCR .2
#define MIN_CPUTHINK_INCR 5
#define MIN_SEQPROB_INCR .1
#define MIN_ALIGNQUANTA_INCR (1 * 1024) /* 1 KB */
#define MIN_ALIGNPROB_INCR .2

unsigned int paramDiff(spec1Ptr, spec2Ptr, exceptParam, printFlag, exactness)
    WLPARAM *spec1Ptr, *spec2Ptr;
    int exceptParam;
    int printFlag;
    double exactness;
{
    /* return a bit vector of the parameters that don't match fairly closely

	The definition of close varies for each parameter

	Don't check the parameters in the exceptParam flag list

	spec2 should be the one you're verifying to see if it matches against
	    spec1
	
	The higher the exactness, the more closely the parameters must match.
	    exactness = 1.0 is the standard.
    */

    unsigned int misMatch;
    
    double fractionDiff();

    misMatch = 0;

    /* ??? not checking hitDepth */
    exceptParam |= HITDEPTH;
    /* not checking sizeCVar */
    exceptParam |= SIZECVAR;

    if (!(exceptParam & UNIQUEBYTES)) {
	if (fractionDiff((double)spec1Ptr->uniqueBytes,
	    (double)spec2Ptr->uniqueBytes) > .1/exactness) {
	    if (printFlag) {
		printf("uniqueBytes off: %lf vs. %lf KB\n",
		    spec1Ptr->uniqueBytes/1024.0, spec2Ptr->uniqueBytes/1024.0);
	    }
	    misMatch |= UNIQUEBYTES;
	}
    }

    /* only checking to make sure reUse is not 1 (cold start for figuring out
	max uniqueBytes */
    if (!(exceptParam & REUSE)) {
	/*
	if (fractionDiff(spec1Ptr->reUse, spec2Ptr->reUse) > .15/exactness) {}
	*/
	if ( (spec1Ptr->reUse < 1.2) ^ (spec2Ptr->reUse < 1.2) ) {
	    if (printFlag) {
		printf("reUse off: %lf vs. %lf\n", spec1Ptr->reUse,
		    spec2Ptr->reUse);
	    }
	    misMatch |= REUSE;
	}
    }

    if (!(exceptParam & HITDEPTH)) {
	/* hitDepth is only checked if the fraction of hits is significant */
	if (spec1Ptr->reUse > 1.5 && spec2Ptr->reUse > 1.5) {
	    /* hitDepth is checked on the absolute difference */
	    if (fabs(spec1Ptr->hitDepth - spec2Ptr->hitDepth) > .2/exactness) {
		if (printFlag) {
		    printf("hitDepth off: %lf vs. %lf\n", spec1Ptr->hitDepth,
			spec2Ptr->hitDepth);
		}
		misMatch |= HITDEPTH;
	    }
	}
    }

    if (!(exceptParam & READPROB)) {
	/* readProb is checked on the absolute difference */
	if (fabs(spec1Ptr->readProb - spec2Ptr->readProb) > .1/exactness) {
	    if (printFlag) {
		printf("readFrac off: %lf vs. %lf\n", spec1Ptr->readProb,
		    spec2Ptr->readProb);
	    }
	    misMatch |= READPROB;
	}
    }

    if (!(exceptParam & SIZEMEAN)) {
	if (fractionDiff((double)spec1Ptr->sizeMean,
	    (double)spec2Ptr->sizeMean) > .1/exactness) {
	    if (printFlag) {
		printf("sizeMean off: %lf vs. %lf KB\n",
		    spec1Ptr->sizeMean/1024.0, spec2Ptr->sizeMean/1024.0);
	    }
	    misMatch |= SIZEMEAN;
	}
    }

    /* checking for sizeCVar is very lax */

    if (!(exceptParam & SIZECVAR)) {
	/* sizeCVar is checked on the absolute difference */
	if (fabs(spec1Ptr->sizeCVar - spec2Ptr->sizeCVar) > .5/exactness) {
	    if (printFlag) {
		printf("sizeCVar off: %lf vs. %lf\n", spec1Ptr->sizeCVar,
		    spec2Ptr->sizeCVar);
	    }
	    misMatch |= SIZECVAR;
	}
    }

    if (!(exceptParam & PROCESSNUM)) {
	if (spec1Ptr->processNum != spec2Ptr->processNum) {
	    if (printFlag) {
		printf("processNum off: %d vs. %d\n", spec1Ptr->processNum,
		    spec2Ptr->processNum);
	    }
	    misMatch |= PROCESSNUM;
	}
    }


    if (!(exceptParam & SHARING)) {
	/* checking for sharing is very lax */

	if (spec1Ptr->processNum>1 && spec2Ptr->processNum>1 &&
	    fabs(spec1Ptr->sharing - spec2Ptr->sharing) > .5/exactness) {
	    if (printFlag) {
		printf("sharing off: %lf vs. %lf\n", spec1Ptr->sharing,
		    spec2Ptr->sharing);
	    }
	    misMatch |= SHARING;
	}
    }

    /* don't bother checking for cpuThink */


    if (!(exceptParam & SEQPROB)) {
	/* seqProb is checked on absolute difference */
	if (fabs(spec1Ptr->seqProb - spec2Ptr->seqProb) > .1/exactness) {
	    if (printFlag) {
		printf("seqFrac off: %lf vs. %lf\n", spec1Ptr->seqProb,
		    spec2Ptr->seqProb);
	    }
	    misMatch |= SEQPROB;
	}
    }

    /* don't bother checking for alignQuanta */
    /* don't bother checking for alignProb */

    return(misMatch);
}

double convertParamVal(specPtr, whichParam)
    WLPARAM *specPtr;
    int whichParam;
{
    /* converts a WLPARAM into one of the values of the structure (depending
	on whichParam) */
    if (whichParam == UNIQUEBYTES) {
	return (specPtr->uniqueBytes/1024.0);
    } else if (whichParam == REUSE) {
	return (specPtr->reUse);
    } else if (whichParam == HITDEPTH) {
	return (specPtr->hitDepth);
    } else if (whichParam == READPROB) {
	return (specPtr->readProb);
    } else if (whichParam == SIZEMEAN) {
	return (specPtr->sizeMean/1024.0);
    } else if (whichParam == SIZECVAR) {
	return (specPtr->sizeCVar);
    } else if (whichParam == PROCESSNUM) {
	return ((double) specPtr->processNum);
    } else if (whichParam == SHARING) {
	return (specPtr->sharing);
    } else if (whichParam == CPUTHINK) {
	return (specPtr->cpuThink);
    } else if (whichParam == SEQPROB) {
	return (specPtr->seqProb);
    } else if (whichParam == ALIGNQUANTA) {
	return (specPtr->alignQuanta/1024.0);
    } else if (whichParam == ALIGNPROB) {
	return (specPtr->alignProb);
    } else {
	printf("error in convertParamVal: whichParam = %d\n", whichParam);
	exit(1);
    }
    /* should never get here */
    return(-1);
}

double convertParamSmallestMax(whichParam)
    int whichParam;
{
    /* returns the smallest maximum value adaptWl will allow */
    /* should only be called from narrowRangeParamNew */

    char *convertParamStr();

    if (whichParam == SIZEMEAN) {
	return (64.0); /* smallest is 64 KB */
    } else if (whichParam == PROCESSNUM) {
	return ((double) 3.0); /* smallest is 3 processes */
    } else {
	printf("error in convertParamVal: whichParam = %d (%s)\n", whichParam,
	    convertParamStr(whichParam));
	exit(1);
    }
    /* should never get here */
    return(-1);
}

char *convertParamStr(whichParam)
    int whichParam;
{
    static char *name[] = {
        "uniqueBytes (KB)",	/* 0 */
	"reUse",		/* 1 */
	"hitDepth",		/* 2 */
	"readFrac",		/* 3 */
	"sizeMean (KB)",	/* 4 */
	"sizeCVar",		/* 5 */
	"processNum",		/* 6 */
	"sharing",		/* 7 */
	"cpuThink (ms)",	/* 8 */
	"seqFrac",		/* 9 */
	"alignQuanta",		/* 10 */
	"alignFrac",		/* 11 */
    };
    if (whichParam == UNIQUEBYTES) {
	return (name[0]);
    } else if (whichParam == REUSE) {
	return (name[1]);
    } else if (whichParam == HITDEPTH) {
	return (name[2]);
    } else if (whichParam == READPROB) {
	return (name[3]);
    } else if (whichParam == SIZEMEAN) {
	return (name[4]);
    } else if (whichParam == SIZECVAR) {
	return (name[5]);
    } else if (whichParam == PROCESSNUM) {
	return (name[6]);
    } else if (whichParam == SHARING) {
	return (name[7]);
    } else if (whichParam == CPUTHINK) {
	return (name[8]);
    } else if (whichParam == SEQPROB) {
	return (name[9]);
    } else if (whichParam == ALIGNQUANTA) {
	return (name[10]);
    } else if (whichParam == ALIGNPROB) {
	return (name[11]);
    } else {
	printf("error in convertParamStr: whichParam = %d\n", whichParam);
	exit(1);
    }
    /* should never get here */
    return("\0");
}

double convertParamMinIncr(whichParam)
    int whichParam;
{
    if (whichParam == UNIQUEBYTES) {
	return (MIN_UNIQUEBYTE_INCR/1024.0);
    } else if (whichParam == REUSE) {
	return (MIN_REUSE_INCR);
    } else if (whichParam == HITDEPTH) {
	return (MIN_HITDEPTH_INCR);
    } else if (whichParam == READPROB) {
	return (MIN_READPROB_INCR);
    } else if (whichParam == SIZEMEAN) {
	return (MIN_SIZEMEAN_INCR/1024.0);
    } else if (whichParam == SIZECVAR) {
	return (MIN_SIZECVAR_INCR);
    } else if (whichParam == PROCESSNUM) {
	return ((double) MIN_PROCESSNUM_INCR);
    } else if (whichParam == SHARING) {
	return (MIN_SHARING_INCR);
    } else if (whichParam == CPUTHINK) {
	return (MIN_CPUTHINK_INCR);
    } else if (whichParam == SEQPROB) {
	return (MIN_SEQPROB_INCR);
    } else if (whichParam == ALIGNQUANTA) {
	return (MIN_ALIGNQUANTA_INCR/1024.0);
    } else if (whichParam == ALIGNPROB) {
	return (MIN_ALIGNPROB_INCR);
    } else {
	printf("error in convertParamIncr: whichParam = %d\n", whichParam);
	exit(1);
    }
    /* should never get here */
    return(-1);
}

assignParamVal(specPtr, whichParam, value)
    WLPARAM *specPtr;
    int whichParam;
    double value;
{
    double round();

    if (whichParam == UNIQUEBYTES) {
	specPtr->uniqueBytes = (unsigned int) round(value * 1024.0);
    } else if (whichParam == REUSE) {
	specPtr->reUse = value;
    } else if (whichParam == HITDEPTH) {
	specPtr->hitDepth = value;
    } else if (whichParam == READPROB) {
	specPtr->readProb = value;
    } else if (whichParam == SIZEMEAN) {
	specPtr->sizeMean = (unsigned int) round(value * 1024.0);
    } else if (whichParam == SIZECVAR) {
	specPtr->sizeCVar = value;
    } else if (whichParam == PROCESSNUM) {
	specPtr->processNum = (int) round(value);
	if (specPtr->processNum<1) {
	    specPtr->processNum = 1;
	}
    } else if (whichParam == SHARING) {
	specPtr->sharing = value;
    } else if (whichParam == CPUTHINK) {
	specPtr->cpuThink = value;
    } else if (whichParam == SEQPROB) {
	specPtr->seqProb = value;
    } else if (whichParam == ALIGNQUANTA) {
	specPtr->alignQuanta = (unsigned int) round(value * 1024.0);
    } else if (whichParam == ALIGNPROB) {
	specPtr->alignProb = value;
    } else {
	printf("error in assignParamVal: whichParam = %d\n", whichParam);
	exit(1);
    }
}

double convertParamFlat(whichParam, minParamValue, maxParamValue)
    int whichParam;
    double minParamValue;
    double maxParamValue;
{
    /* return the value of the parameter when the performance curve is flat */

    double paramValue;

    double convertParamVal();
    char *convertParamStr();

    if (whichParam & POSITIVE_PARAM) {
	/* this is a positive parameter */
	/* paramValue = convertParamVal(minSpecPtr, whichParam); */
	paramValue = minParamValue;
    } else if (whichParam & NEGATIVE_PARAM) {
	/* this is a negative parameter */
	/* paramValue = convertParamVal(maxSpecPtr, whichParam); */
	paramValue = maxParamValue;
    } else {
	/* this isn't generally positive or negative */
	/* return the halfway point between min and max */
	/* paramValue = .5 * (convertParamVal(minSpecPtr, whichParam) +
		      convertParamVal(maxSpecPtr, whichParam)); */
	paramValue = .5 * (minParamValue + maxParamValue);
    }

    printf("convertParamFlat called for %s, returning %lf\n",
	convertParamStr(whichParam), paramValue);
    return(paramValue);

    /* maybe for flat curves, I should keep the same knee value (at least
	 for uniqueBytes).  For now, I'll return the max uniqueByte value */

/* this was the old way of doing it.  I might use this in the future if
    the max and min values are too hard to achieve
    if (whichParam == UNIQUEBYTES) {
	return(convertParamVal(maxSpecPtr, whichParam));
    } else if (whichParam == REUSE) {
	return(2.0);
    } else if (whichParam == HITDEPTH) {
	return(.75);
    } else if (whichParam == READPROB) {
	return(.5);
    } else if (whichParam == SIZEMEAN) {
	return(4.0);
    } else if (whichParam == SIZECVAR) {
	return(1.0);
    } else if (whichParam == PROCESSNUM) {
	return(1.0);
    } else if (whichParam == SHARING) {
	return(0.0);
    } else if (whichParam == CPUTHINK) {
	return(0.0);
    } else if (whichParam == SEQPROB) {
	return(0.0);
    } else if (whichParam == ALIGNQUANTA) {
	return(4.0);
    } else if (whichParam == ALIGNPROB) {
	return(.9);
    } else {
	printf("error in convertParamFlat: whichParam = %d\n", whichParam);
	exit(1);
    }
*/
}

int convertParamIndex(whichParam)
    int whichParam;
{
    int i, paramIndex;

    for (i=1, paramIndex=0; i<=MAXWLPARAM && i!=whichParam; i*=2, paramIndex++);
    if (i>MAXWLPARAM) {
	printf("error in convertParamIndex: whichParam=%d\n", whichParam);
	exit(1);
    }
    return(paramIndex);
}

copyWlParam(srcSpecPtr, destSpecPtr)
    WLPARAM *srcSpecPtr;
    WLPARAM *destSpecPtr;
{
    void bcopy();
    double convertParamVal();
    bcopy((char *) srcSpecPtr, (char *) destSpecPtr, sizeof(WLPARAM));
    /*
    int whichParam;
    for (whichParam = 1; whichParam<=MAXWLPARAM; whichParam*=2) {
	assignParamVal(destSpecPtr, whichParam,
	    convertParamVal(srcSpecPtr, whichParam));
    }
    */
}

double fractionDiff(a1, a2)
    double a1, a2;
{
    if (a1 == 0) {
	return(1.0);
    }
    return(fabs((a1-a2)/a1));
}

int inRange(specPtr, minSpecPtr, maxSpecPtr)
    WLPARAM *specPtr;
    WLPARAM *minSpecPtr;
    WLPARAM *maxSpecPtr;
{
    /* returns 1 if spec is between minSpec and maxSpec, 0 otherwise */
    int whichParam;

    char *convertParamStr();
    double convertParamVal();

    for (whichParam=1; whichParam<=MAXWLPARAM; whichParam*=2) {
	if ((minSpecPtr != NULL &&
		convertParamVal(specPtr, whichParam) <
		    convertParamVal(minSpecPtr,whichParam)) ||
	   (maxSpecPtr != NULL && convertParamVal(specPtr, whichParam) >
		convertParamVal(maxSpecPtr,whichParam))) {
	    printf("run not in range, %s=%lf not in [%lf, %lf]\n",
		convertParamStr(whichParam),
		convertParamVal(specPtr,whichParam),
		convertParamVal(minSpecPtr,whichParam),
		convertParamVal(maxSpecPtr,whichParam));
	    return(0);
	}
    }
    return(1);
}
