/* specWl -d dirName -u uniqueBytes -re reUse -h hitDepth -r readProb -s sizeMean
-sc sizeCVar -p processNum -sh sharing -c cpuThink -q seqProb -se seed
   e.g. specWl -d /r4/pmchen/test -ca 1000 -re 3.5 -h .5 -s 4 -sc 1.0 -p 3
		    -sh .75 -c 10 -q .3 -se 7
    
    This program specifies the workload to run (run by doWl).  It generates
	the command files for the parent of doWl (to tell how many children to
	fork off), the command files for the children of doWl to run, 
	and the data file for the children of doWl to access.

    dirName is the directory in which the data and command files will be created
    uniqueBytes is the number of unique bytes transferred in this run in KB
	(double), gets converted to bytes (unsigned int)
    reUse is the average number of times a byte is accessed (should be >= 1)
	(double)
    hitDepth is the average distance in the LRU stack that a request hits at
	(doesn't count compulsory misses).  This is a fraction of the
	uniqueBytes. (double)
    readProb is the probability that a request is a read (between 0 and 1)
	(double)
    sizeMean is the mean request size in KB (double), gets converted to
	    bytes (unsigned int)
    sizeCVar is the coefficient of variation of request size (double)
    processNum is the number of concurrent processes (integer)
    sharing is the amount of sharing between processes (see ~bench/step1/notes)
	(double)
    cpuThink is the cpu think time in ms (double)
    seqProb is the probability (between 0 and 1) that the next request a
	process issues is the next consecutive block (same size as the
	last request). (double)
    alignQuanta is the block size of the system--most (90%) of the requests
	should start on a block boundary; the size should also be a multiple of
	this block size
    seed is the random seed (integer)

   compilation notes:
    flags:
	DEBUG: used for debugging
	UNDEF: should never be defined
*/

#include <stdio.h>
#include <sys/types.h>
#include <string.h>
/* #include <strings.h> */
#include <fcntl.h>
#include <sys/file.h>
#include <sys/uio.h>
/* #include <status.h> */
/* #include <sys/time.h> */
#include <math.h>
/* #include <sys/resource.h> */
/* #include <sprite.h> */
#include <errno.h>
#include <signal.h>
#include "specWl.h"
#ifdef SPRITE
#include <bstring.h>
#endif /* SPRITE */

#define MAX(a,b) ( ( (a) > (b) ) ? (a) : (b)  )
#define MIN(a,b) ( ( (a) < (b) ) ? (a) : (b)  )

specWl(argc, argv, paramResultPtr, printFlag)
    int argc;
    char *argv[];
    WLPARAM *paramResultPtr;
    int printFlag;
{
    int i;
    int status;

    /* workload parameters */
    WLPARAM spec;
    WLPARAM target;

    int fileNum; /* number of files */
    unsigned int fileSize; /* size of each file */
    int process;
    int readFlag;
    unsigned int size;
    int file;
    unsigned int start;
    double waitTime;

    unsigned int lastStart[PROCMAX]; /* last IO location (for sequentiality
					    purposes) */
    int lastFile[PROCMAX]; /* last I/O file (for sequentiality purposes) */
    unsigned int lastSize[PROCMAX]; /* last IO size (for sequentiality
					    purposes) */

    /* stability variables */
    int iter;

    /* statistics declarations */
    double sumSize; /* bytes transferred so far */
    double sum2Size; /* sum of squares of bytes transferred */
    double avSize, sDevSize;
    double averageHitDepth; /* average lru distance for hits in bytes */
    unsigned int lruDepth; /* total number of bytes in the lru stack */
    int numHit;
    int numMiss;
    int numHitAttempt;
    int numMissAttempt;
    int numRead;

    int numHitAttemptSave;
    int numHitSave;
    int hitFlag;
    int seqFlag;
    int hitAttempt;
    int successfulLoc;

    double tmpHitDepth;

    int numAligned;
    int numSeq;

    /* feedback */
    double fractionDone;
    int sizeAtMin; /* flag to getSize to tell it when alignment should be
			a lower priority than a small size */

    /* communication with adaptWl */
    char tmpFileName[100];
    FILE *tmpFile;

    /* function declarations */
    void * malloc();
    void typedistr();
    void sizedistr();
    unsigned int getSize();
    unsigned int getLocRandom();
    void getParamSpecWl();
    double expon();
    double adjustTarget();
    int fileNumFunc();
    unsigned int fileSizeFunc();
    void bcopy();
    int processFileNum();
    double round();
    int prob();

    /* start program execution */
    for (i=0; i<argc; i++) printf("%s ",argv[i]); printf("\n");
    fflush(stdout);
    
#ifdef DEBUG
    printf("DEBUG\n");
#endif
#ifdef UNDEF
    printf("ERROR: UNDEF\n");
    exit(1);
#endif

    /* get the command line arguments */
    getParamSpecWl(argc, argv, &spec);

    if (printFlag) {
	printf("dirName %s\nuniqueBytes %lf KB\nreUse %lf\nhitDepth %lf\n",
	    spec.dirName, spec.uniqueBytes/1024.0, spec.reUse, spec.hitDepth);
	printf("readProb %lf\nsizeMean %lf KB\nsizeCVar %lf\nprocessNum %d\n",
	    spec.readProb, spec.sizeMean/1024.0, spec.sizeCVar,
	    spec.processNum);
	printf("sharing %lf\ncpuThink %lf ms\nseqProb %lf\nalignQuanta %u\n",
	    spec.sharing, spec.cpuThink, spec.seqProb, spec.alignQuanta);
	printf("alignProb %lf\nseed %d\n", spec.alignProb, spec.seed);
    }

    if (spec.processNum > PROCMAX) {
	printf("spec.processNum=%d, too many processes\n", spec.processNum);
	exit(1);
    }

    /* initialize random number generator */
    mySrandom(spec.seed);

    /* figure out sharing */
    fileNum = fileNumFunc(spec.processNum, spec.sharing);
    fileSize = fileSizeFunc(spec.uniqueBytes, fileNum);

    if (fileNum > FILENUMMAX) {
	printf("too many files (%d)\n", fileNum);
	exit(1);
    }

    if (printFlag) {
	printf("%d files of %lf KB\n", fileNum, fileSize/1024.0);
    }

    if (spec.sharing > 1) {
	printf("sharing all %d files between %d processes\n", fileNum,
	    spec.processNum);
    }

    /* initialize locality data structures */
    initLoc();
    averageHitDepth = 0.0;
    lruDepth = 0;
    numHit = numMiss = numHitAttempt = numMissAttempt = 0;
    numRead = 0;
    numAligned = 0;

    /* initialize other data structures */
    sumSize = sum2Size = 0.0;

    initChildParent(&spec, fileNum, fileSize);

#ifdef DEBUG
    printf("setting starting values for sequentiality\n");
#endif
    /* set starting values for sequentiality */
    for (process=0; process<spec.processNum; process++) {
	lastSize[process] = 0;
	lastStart[process] = 0;
	lastFile[process] = processFileNumUnique(process,fileNum, spec.sharing);
    }
    numHitAttempt = numMissAttempt = 0;
    numSeq = 0;
    sizeAtMin = 0;

    /* initialize the target workload to be equal to the specified workload */
    /*
    bcopy((char *) &spec, (char *) &target, sizeof(WLPARAM));
    */
    copyWlParam(&spec, &target);

    process = spec.processNum - 1; /* which process gets this I/O */

    /* main loop */

    for (iter=0; sumSize < spec.uniqueBytes * spec.reUse && iter<MAXITER; iter++) {
#ifdef DEBUG
	printf("\n");
#endif
	
	/* which process should issue the request */
	/* for now, just go round robin.  Maybe later, I'll try to
	    estimate the time for each I/O and guess the order that
	    it would occur when running doWl */
	process = (process+1)%spec.processNum;

	/* cpu think time */
	waitTime = expon(spec.cpuThink);

	/* read or write */
	readFlag = prob(target.readProb);
	if (readFlag) {
	    numRead++;
	}

	/* request size */
	size = getSize(target.sizeMean, target.sizeCVar, fileSize,
		spec.alignQuanta, target.alignProb, sizeAtMin);
#ifdef DEBUG
	printf("size = %lf KB\n", size/1024.0);
#endif

	/* get the request location */
	for (successfulLoc=0; successfulLoc<5; successfulLoc++) {

	    /* which file */
	    file = processFileNum(process,fileNum, spec.sharing);

	    seqFlag = prob(target.seqProb);
#ifdef DEBUG
	    printf("successfulLoc=%d, seqFlag=%d\n", successfulLoc, seqFlag);
#endif
	    /* sequential or random? */
	    if (seqFlag) {
		/* sequential */
#ifdef DEBUG
		printf("trying sequential\n");
#endif

		/* see if the miss ratio can afford another sequential access */
		start = lastStart[process] + lastSize[process];
		file = lastFile[process];
		if (start + size - 1 >= fileSize) {
		    start = 0;
		    file = (file + 1) % fileNum;
		}
		status = describeLoc(file, start, size, &tmpHitDepth);
		if (successfulLoc<4 && status && target.reUse<= 1.3) {
		    /* this sequential access would be a hit and we need a
			    miss */
		    seqFlag = 0;
		} else {
		    successfulLoc = 5;
#ifdef DEBUG
		    printf("start=%u\n", start);
#endif
		    numSeq++;
		    hitAttempt = 2;
		}
	    }
	    if (!seqFlag && target.seqProb < 1) {
		/* only get a random address if the target seqProb allows it */
		/* random */
#ifdef DEBUG
		printf("trying random\n");
#endif
		numHitSave = numHit;
		numHitAttemptSave = numHitAttempt;
		start = getLocRandom(target.reUse,
			target.hitDepth*(spec.uniqueBytes-1), size, fileSize,
			spec.uniqueBytes, file, spec.alignQuanta,
			target.alignProb, &numHitAttempt, &numMissAttempt);

		/* see if the miss ratio can afford this access */
		status = describeLoc(file, start, size, &tmpHitDepth);
		if (successfulLoc>=4 || !status || target.reUse > 1.3) {
		    successfulLoc = 5;
		    hitAttempt = (numHitAttempt != numHitAttemptSave);
		}
	    }
	}

	if (start > fileSize-1 || start + size - 1 > fileSize-1) {
	    printf("error: address too large, extent [%u, %u], maxAddress %u\n",
		start, start+size-1, fileSize-1);
	    exit(1);
	}

	/* add the I/O command to the list for this process */
	addIO(process, readFlag? 'r':'w', (spec.sharing<=1)?0:file, size,
	    start, waitTime);

#ifdef DEBUG
	printf("%d--process %d: %c size %.3lf start %u wait %lf\n", iter, process, readFlag? 'r':'w', size/1024.0, start, waitTime);
#endif

	maintainLoc(file, start, size, &averageHitDepth, &numHit, &numMiss,
		    &lruDepth);
	
	hitFlag = (numHit != numHitSave);

#ifdef DEBUG
	if (hitAttempt != 2 && hitAttempt != hitFlag) {
	    if (hitAttempt) {
		printf("tried to hit, but missed\n");
	    } else {
		printf("tried to miss, but hit\n");
	    }
	}
#endif

	lastStart[process] = start;
	lastSize[process] = size;

	if (!(start%spec.alignQuanta)) {
	    numAligned++;
	}
	sumSize+=size;
	sum2Size += (double)size*size;

	/*
	printf("printing address leaves for file %d\n", file);
	printLeavesTree(file, 0);
	printf("\n");
	printf("printing lru leaves\n");
	printLeavesLruTree(0);
	printf("\n");
	*/

	/* feedback mechanism */
	if (!(iter%FEEDBACKINTERVAL)) {
#ifdef DEBUG
	    printf("iter=%d, modifying target.hitDepth, target.reUse, target.sizeMean, target.readProb\n", iter);
#endif

	    fractionDone = sumSize / (spec.uniqueBytes*spec.reUse);

#ifdef DEBUG
	    printf("fractionDone = %lf\n", fractionDone);
#endif

	    if (fractionDone > 0 && fractionDone < 1) {
		/* adjust target.hitDepth */
		if (numHit>0) {
		    /* printf("current averageHitDepth = %lf KB for %d hits (%lf of uniqueBytes)\n",
			averageHitDepth/1024.0, numHit,
			averageHitDepth / (double)spec.uniqueBytes);
		*/
		    target.hitDepth = adjustTarget("hitDepth", spec.hitDepth,
			averageHitDepth/spec.uniqueBytes, fractionDone, 0.0,
			1.0, 1.0);
		} else {
#ifdef DEBUG
		    printf("no hits so far\n");
#endif
		}

		/* adjust target.reUse */
		if (lruDepth>0) {
		    target.reUse= adjustTarget("reUse", spec.reUse,
			sumSize/(double)lruDepth, fractionDone, 1.0,
			target.reUse*3.0, 2.0);
		    /* ??? this is a hack to get the final reUse more
			in line with what is requested */
		    /*
		    target.reUse = MAX(1.0, target.reUse*.9);
		    */
		} else {
#ifdef DEBUG
		    printf("no misses so far\n");
#endif
		}

		/* adjust target.sizeMean */
#ifdef DEBUG
		printf("current sizeMean = %lf KB\n",
			    sumSize/((double)numMiss+numHit)/1024.0);
#endif
		if (spec.sizeMean < -1 * fractionDone / (1-fractionDone) *
		    OVERSHOOT * (spec.sizeMean - sumSize/((double)numMiss+numHit))) {
#ifdef DEBUG
		    printf("sizeMean would have gone negative\n");
#endif
		    target.sizeMean = 0;
		    sizeAtMin = 1;
		} else {
		    target.sizeMean = spec.sizeMean +
			    fractionDone / (1-fractionDone) * OVERSHOOT *
			    (spec.sizeMean - sumSize/((double)numMiss+numHit));
		    sizeAtMin = 0;
		}

		/* don't allow target.sizeMean to vary too much away from
			spec.sizeMean */
		if (target.sizeMean < spec.sizeMean / 3.0) {
		    target.sizeMean = spec.sizeMean / 3.0;
		}
		if (target.sizeMean > spec.sizeMean * 3.0) {
		    target.sizeMean = spec.sizeMean * 3.0;
		}
		if (target.sizeMean > fileSize) {
		    target.sizeMean = fileSize;
		}
#ifdef DEBUG
		printf("new target.sizeMean = %lf KB\n",
		    target.sizeMean/1024.0);
#endif

		target.readProb = adjustTarget("readProb", spec.readProb,
		    numRead/((double)numMiss+numHit), fractionDone, 0.0, 1.0,
		    1.0);

		target.alignProb = adjustTarget("alignProb", spec.alignProb,
		    numAligned/((double)numMiss+numHit), fractionDone,0.0,1.0,
		    1.0);

		target.seqProb = adjustTarget("seqProb", spec.seqProb,
			numSeq/((double)numMiss+numHit), fractionDone,0.0,1.0,
			1.0);

	    } else {
#ifdef DEBUG
		printf("too early or too late to make adjustments\n");
#endif
	    }

	}

    } /* end of main loop */

#ifdef DEBUG
    printf("end of main loop\n");
#endif

    /* flush any remaining I/O buffers */
    for (i=0; i<spec.processNum; i++) {
	printBuf(i);
    }

    /* print the final address 2-3 trees and lru tree */
    /*
    for (file=0; file<fileNum; file++) {
	printf("printing address leaves for file %d\n", file);
	printLeavesTree(file, 0);
	printf("\n");
    }
    printf("printing lru leaves\n");
    printLeavesLruTree(0);
    printf("\n");
    */

    checkLoc();
    freeLoc();


    if (printFlag) {
	printf("hits = %d, misses = %d, averageHitDepth = %lf, lruDepth = %u\n",
	    numHit, numMiss, averageHitDepth, lruDepth);

	printf("number of attempted hits=%d, number of attempted misses=%d\n",
	    numHitAttempt, numMissAttempt);
    }

    avSize = sumSize / (numHit + numMiss);
    sDevSize = sqrt(sum2Size/(numHit + numMiss) - avSize*avSize);

    if (printFlag) {
	printf("avSize = %lf, sDevSize = %lf\n", avSize, sDevSize);

	printf("sumSize = %lf KB, average size = %lf KB, coef of var = %lf\n",
	    sumSize/1024.0, avSize/1024.0, sDevSize/avSize);

	printf("fraction reads %lf\n",
	    (double)numRead/((double)numMiss+numHit));

	printf("total LRU depth %lf KB\n", lruDepth/1024.0);
	printf("average LRU depth for hits %lf KB (%lf of uniqueBytes, %lf of lruDepth)\n",
		averageHitDepth/1024.0, averageHitDepth/spec.uniqueBytes,
		averageHitDepth/lruDepth);
	printf("fraction compulsory misses %lf\n",
	    (double)numMiss/(numMiss+numHit));

	printf("fraction aligned %lf to %lf KB\n",
	    (double)numAligned/(numMiss+numHit), spec.alignQuanta / 1024.0);
	
	printf("fraction sequential %lf\n", (double)numSeq/(numMiss+numHit));

	printf("\n");

	printLoc(numHit, lruDepth);
    }

    /* print out the achieved workload values for adaptWl */
    paramResultPtr->uniqueBytes = lruDepth;
    paramResultPtr->reUse = sumSize/lruDepth;
    paramResultPtr->hitDepth = averageHitDepth/lruDepth;
    paramResultPtr->readProb = (double)numRead/(numMiss+numHit);
    paramResultPtr->sizeMean = (unsigned int) round(avSize);
    paramResultPtr->sizeCVar = sDevSize/avSize;
    paramResultPtr->processNum = spec.processNum;
    if (spec.sharing <= 1) {
	paramResultPtr->sharing = (spec.processNum<=1)?
		0: (double)(spec.processNum - fileNum) / (spec.processNum-1);
    } else {
	paramResultPtr->sharing = spec.sharing;
    }
    paramResultPtr->cpuThink = -1; /* not defined 'til you run doWl */
    paramResultPtr->seqProb = (double)numSeq/(numMiss+numHit);
    paramResultPtr->alignQuanta = spec.alignQuanta;
    paramResultPtr->alignProb = (double)numAligned/(numMiss+numHit);
    
    sprintf(tmpFileName, "%s/specWl.adaptWl", spec.dirName);

    /*
    printf("opening %s\n", tmpFileName);
    */
    tmpFile = fopen(tmpFileName, "w");
    if (tmpFile == NULL) {
	perror("error in specWl: fopen");
	exit(1);
    }
    printWlParamFile(paramResultPtr, tmpFile);

    /*
    fprintf(tmpFile,"uniqueBytes %u\n", lruDepth);
    fprintf(tmpFile,"reUse %lf\n", sumSize/lruDepth);
    fprintf(tmpFile,"hitDepth %lf\n", averageHitDepth/lruDepth);
    fprintf(tmpFile,"readFrac %lf\n", (double)numRead/(numMiss+numHit));
    fprintf(tmpFile,"sizeMean %u\n", (unsigned int) round(avSize));
    fprintf(tmpFile,"sizeCVar %lf\n", sDevSize/avSize);
    fprintf(tmpFile,"processNum %d\n", spec.processNum);
    fprintf(tmpFile,"sharing %lf\n", (spec.processNum<=1)?
	    0: (double)(spec.processNum - fileNum) / (spec.processNum-1));
    fprintf(tmpFile,"cpuThink %lf\n", -1.0); not defined 'til you run doWl
    fprintf(tmpFile,"seqFrac %lf\n", (double)numSeq/(numMiss+numHit));
    fprintf(tmpFile,"alignQuanta %u\n", spec.alignQuanta);
    fprintf(tmpFile,"alignFrac %lf\n", (double)numAligned/(numMiss+numHit));
    */

    return(0);
}
