/* implement a 2-3 tree for calculating the locality of a trace */
#include <stdio.h>
#include <math.h>
#include "tree.h"

#define MAXFILENUM 10000
#define LRUFREQQUANTUM (100 * 1024) /* 100 KB */
#define MAXLRUNUM 25000 /* max number of lru depth buckets */

int mallocSpace;

int numLeaves[MAXFILENUM];
int numLeavesLru;
int maxFileNumSeen;

int lruDistFreq[MAXLRUNUM];

extern int lruNodeId;
extern int addressNodeId;

TWOTHREENODE *(addressRootPtr[MAXFILENUM]);
TWOTHREENODELRU *lruRootPtr; /* only one lru tree (one global lru
					tree for all files) */

void *malloc();

initLoc()
{
    /* initializes the locality trees */

    int file;

    numLeavesLru = 0;
    maxFileNumSeen = 0;

    /* initialize roots */
    for (file=0; file<MAXFILENUM; file++) {
	addressRootPtr[file] = NULL;
	numLeaves[file] = 0;
	/*
	checkAddressRoot(addressRootPtr[file], numLeaves[file]);
	*/
    }
    lruRootPtr = NULL;
    /*
    checkLruRoot(lruRootPtr, numLeavesLru, maxFileNumSeen);
    */

}

maintainLoc(file, start, length, averageHitDistPtr, numHitPtr, numMissPtr,
		lruDepthPtr)
    int file;
    unsigned int start;
    unsigned int length;
    double *averageHitDistPtr;
    int *numHitPtr;
    int *numMissPtr;
    unsigned int *lruDepthPtr;
{
    /* maintains the locality and address trees
	result is the average distance for hits after this new request
	also modifies numHit, numMiss, lruDepth

        *** make sure to initialize numHit, numMiss, and lruDepth in whatever
		routine calls maintainLoc ***
    */
    unsigned int end;

    unsigned int bytesMatched;
    unsigned int bytesAhead;
    unsigned int lastBytesMatched;
    unsigned int lastBytesAhead;

    double totalBytesMatched, totalDistMatched;

    int lruFreqEntry;

    TWOTHREENODE *newNodePtr;
    TWOTHREENODELRU *lruNewNodePtr;

    int status;

    TWOTHREENODE *insertAddress();
    TWOTHREENODE *deleteAddress();
    TWOTHREENODELRU *insertLru();
    TWOTHREENODELRU *deleteLru();
    int doSearch();
    double round();

    if (file >= MAXFILENUM) {
	printf("error: file number %d too large for array\n", file);
	exit(1);
    }
    end = start + length - 1;
    /*
    printf("\n---- file %d: %d - %d ----\n", file, start, end);
    */

    if (file > maxFileNumSeen) {
	maxFileNumSeen = file;
    }

    /* search for this extent in the address 2-3 tree */
    totalBytesMatched = totalDistMatched = 0;
    lastBytesMatched = lastBytesAhead = 0;
    do {
	status = doSearch(addressRootPtr[file], &(addressRootPtr[file]),
		    &lruRootPtr, start, end, file, &bytesMatched,
		    &bytesAhead);
	if (status) {
	    /*
	    printf("bytesMatched=%d, bytesAhead=%d\n", bytesMatched,
		bytesAhead);
	    */
	    /* note that this isn't exactly correct, since the lru tree
		was modified by the last match.  This last match might have
		taken some bytes out of the lru tree, thus causing
		bytesAhead to be underestimated.  This is hard to fix,
		since there might be many matched and it would tough to
		determine how many requests that already matched are ahead
		of the one that just matched. */

	    /* I'll do the best I can, by fixing up the new match according
		to if the *last* match was before it in the lru tree */
	    if (lastBytesAhead <= bytesAhead) {
		bytesAhead += lastBytesMatched;
		/*
		printf("patching up the lru distance according to the first part of the match\n");
		*/
	    }
	    lastBytesAhead = bytesAhead;
	    lastBytesMatched = bytesMatched;

	    /*
	    printf("matched %d bytes in file %d with %d bytes ahead\n",
		bytesMatched, file, bytesAhead);
	    */
	    
	    totalBytesMatched += bytesMatched;
	    totalDistMatched += bytesMatched * ((double)bytesAhead + bytesMatched/2);
	    /*
	    printf("printing leaves of address tree for file %d\n", file);
	    printLeaves(addressRootPtr[file], 0);
	    printf("\n");
	    printf("printing leaves of lru tree for file %d\n", file);
	    printLeavesLru(lruRootPtr, 0);
	    printf("\n");
	    */
	}
    } while (status);

    /* create the new node for the lru stack */
    lruNewNodePtr = (TWOTHREENODELRU *) malloc(sizeof(TWOTHREENODELRU));
    lruNewNodePtr->left = NULL; /* this is a leaf */
    lruNewNodePtr->start = start;
    lruNewNodePtr->end = end;
    lruNewNodePtr->file = file;
    lruNewNodePtr->id = lruNodeId++;

    /* insert the new node into the address 2-3 tree */
    newNodePtr = (TWOTHREENODE *) malloc(sizeof(TWOTHREENODE));
    newNodePtr->left = NULL; /* this is a leaf */
    newNodePtr->start = start;
    newNodePtr->end = end;
    newNodePtr->id = addressNodeId++;
    newNodePtr->lruPtr = lruNewNodePtr;

    lruNewNodePtr->addressPtr = newNodePtr;

    /*
    printf("\ninserting %d-%d (#%d) into address tree\n", start, end,
	newNodePtr->id);
    */
    addressRootPtr[file] = insertAddress(addressRootPtr[file], newNodePtr);

    /* insert at the top of the lru stack (0 bytes ahead) */
    lruRootPtr = insertLru(lruRootPtr, lruNewNodePtr, 0);

    numLeaves[file]++;
    /*
    printf("incrementing numLeavesLru: was %d", numLeavesLru);
    */
    numLeavesLru++;
    /*
    printf(", now %d\n", numLeavesLru);
    */

    /*
    printf("printing leaves of address tree for file %d\n", file);
    printLeaves(addressRootPtr[file], 0);
    printf("\n");

    printf("printing leaves of lru tree\n");
    printLeavesLru(lruRootPtr, 0);
    printf("\n");
    */

    /*
    printf("\nchecking address tree for file %d\n", file);
    checkAddressRoot(addressRootPtr[file], numLeaves[file]);
    */

    /*
    printf("\nchecking lru tree\n");
    printf("numLeavesLru = %d\n", numLeavesLru);
    checkLruRoot(lruRootPtr, numLeavesLru, maxFileNumSeen);
    */

    /* maintain locality information for this run */
    if (totalBytesMatched / length < .5) {
	/* more miss than hit */
	(*numMissPtr)++;
    } else {
	/* printf("---hit---\n"); */
	/* more hit than miss */
	/*
	printf("old averageHitDist = %lf, *numHitPtr = %d, totalDistMatched=%lf, totalBytesMatched=%lf\n", *averageHitDistPtr, *numHitPtr, totalDistMatched, totalBytesMatched);
	*/
	(*averageHitDistPtr) = (*averageHitDistPtr) * ((double)(*numHitPtr)/(*numHitPtr+1)) +
			totalDistMatched/totalBytesMatched/(*numHitPtr+1);
	(*numHitPtr)++;
	/*
	printf("hit %d bytes in file %d at %d bytes\n",
	    bytesMatched, file, (int) (totalDistMatched/totalBytesMatched));
	*/
	/*
	printf("adding hitDist %lf to average\n",
	    totalDistMatched/totalBytesMatched);
	printf("new averageHitDist = %lf, *numHitPtr=%d\n",
		    *averageHitDistPtr, *numHitPtr);
	*/
	lruFreqEntry =
		round(totalDistMatched/totalBytesMatched/LRUFREQQUANTUM);
	
	/*
	printf("putting hit into bucket %u KB\n", lruFreqEntry*LRUFREQQUANTUM/1024);
	*/
	if (lruFreqEntry >= MAXLRUNUM) {
	    printf("error: lru distance entry %d too big for lruFreq buckets\n",
		    lruFreqEntry);
	    exit(1);
	}
	lruDistFreq[lruFreqEntry]++;
    }
    (*lruDepthPtr) += length - totalBytesMatched;
}


checkLoc()
{
    /* check the consistency of all locality information */

    int file;

    /* print out and check all non-empty trees */
    /*
    printf("\nchecking lru tree\n");
    */
    checkLruRoot(lruRootPtr, numLeavesLru, maxFileNumSeen);

    for (file=0; file<MAXFILENUM; file++) {
	if (addressRootPtr[file] != NULL) {
	    /*
	    printf("printing leaves of address tree for file %d\n", file);
	    printLeaves(addressRootPtr[file], 0);
	    */

	    /*
	    printf("\nchecking address tree for file %d\n", file);
	    */
	    checkAddressRoot(addressRootPtr[file], numLeaves[file]);

	}
    }
}

printLoc(numHit, lruDepth)
    int numHit; /* how many hits there were total */
    unsigned int lruDepth;
{
    /* print the lru distance frequency graph */
    unsigned int bytes;
    int numHitCum; /* used in determining median, 2/3 median, etc. */
    int medianFlag;

    numHitCum = 0;
    medianFlag = 1;

    printf("gti Cumulative Locality Histogram of Hits (out of %d hits)\n", numHit);
    printf("xla LRU distance (KB)\n");
    printf("yla fraction of hits\n");
    printf("das locality\n");
    if (numHit>0) {
	for (bytes=0; bytes<=lruDepth+LRUFREQQUANTUM; bytes+=LRUFREQQUANTUM) {
	    /*
	    printf("%lf\t%lf\n", bytes/1024.0,
			lruDistFreq[bytes/LRUFREQQUANTUM]/(double)numHit);
	    */
	    if (numHitCum < numHit) {
		if (bytes/LRUFREQQUANTUM >= MAXLRUNUM) {
		    printf("error: in printLoc, bytes=%d still doesn't contain all the hits, MAXLRUNUM=%d, LRUFREQQUANTUM=%d\n",
			bytes, MAXLRUNUM, LRUFREQQUANTUM);
		    exit(1);
		}
		numHitCum += lruDistFreq[bytes/LRUFREQQUANTUM];
	    }
	    printf("%lf\t%lf\n", bytes/1024.0, numHitCum/(double)numHit);
	    /*
	    if ((double)numHitCum/numHit > .5 && medianFlag == 1) {
		printf(".5 median is %lf KB\n", bytes/1024.0);
		medianFlag = 2;
	    } else if ((double)numHitCum/numHit > 2/3.0 && medianFlag == 2) {
		printf(".667 median is %lf KB\n", bytes/1024.0);
		medianFlag = 3;
	    } else if ((double)numHitCum/numHit > .75 && medianFlag == 3) {
		printf(".75 median is %lf KB\n", bytes/1024.0);
		medianFlag = 4;
	    } else if ((double)numHitCum/numHit > .9 && medianFlag == 4) {
		printf(".75 median is %lf KB\n", bytes/1024.0);
		medianFlag = 5;
	    }
	    */
	}
    } else {
	printf("0 0\n");
	printf("1 1\n");
    }
    printf("dae\n");
}

int doSearch(nodePtr, addressRootPtrPtr, lruRootPtrPtr, start, end, file,
	resultBytesMatchedPtr, resultBytesAheadPtr)
    TWOTHREENODE *nodePtr;
    TWOTHREENODE **addressRootPtrPtr;
    TWOTHREENODELRU **lruRootPtrPtr;
    unsigned int start;
    unsigned int end;
    int file;
    unsigned int *resultBytesMatchedPtr;
    unsigned int *resultBytesAheadPtr;
{
    /* returns 1 if any bytes matched, 0 otherwise */

    /* search for this extent in the 2-3 tree starting at nodePtr
        if found in one branch, call doSearch on that branch
	if found in more than one branch, call doSearch on all branches it's in
	if nodePtr points to a leaf, then
	    modify the starting and ending addresses of that leaf appropriately.
	    modify the lru stack for that leaf
		change the extent info for the lru leaf
		change the byte totals from the leaf to the root
		    (return the new number of bytes ahead for that leaf)
		if the leaf was split, add a new entry for the right part of
		    the extent at the correct number of bytes ahead in the
		    lru stack (the bytes ahead for the left extent +
		    the number of bytes in the left extent)
    */

    TWOTHREENODE *newNodePtr;
    TWOTHREENODELRU *lruNewNodePtr;

    unsigned int bytesAhead;
    unsigned int bytesMatched;
    unsigned int bytesShorter;
    int status;

    TWOTHREENODE *insertAddress();
    TWOTHREENODE *deleteAddress();
    TWOTHREENODELRU *insertLru();
    TWOTHREENODELRU *deleteLru();
    unsigned int changeByte();

    if (nodePtr == NULL) {
	return(0);
    }

    /*
    printf("called doSearch on node #%d with extent [%u, %u], file=%d\n",
	nodePtr->id, start, end, file);
    */

    if (nodePtr->left == NULL) {
	/* this is a leaf */
	/* see if this extent matches this leaf */
	if (end >= nodePtr->start && start <= nodePtr->end) {
	    /*
	    printf("matched file %d, %d - %d\n", file,
		MAX(start, nodePtr->start), MIN(end, nodePtr->end));
	    printf("node #%d has extent [%d, %d]\n", nodePtr->id,
		nodePtr->start, nodePtr->end);
	    */
	    bytesMatched = MIN(end, nodePtr->end) - MAX(start, nodePtr->start)
			    + 1;
	    (*resultBytesMatchedPtr) = bytesMatched;
	    
	    if (start <= nodePtr->start && end >= nodePtr->end) {
		/* new extent is a superset of the old extent */
		/*
		printf("superset match\n");
		*/

		/* delete this extent from the address 2-3 tree */
		(*addressRootPtrPtr) = deleteAddress(*addressRootPtrPtr,
					    nodePtr->start);
		numLeaves[file]--;
		/*
		printf("decrementing numLeavesLru: was %d", numLeavesLru);
		*/
		numLeavesLru--;
		/*
		printf(", now %d\n", numLeavesLru);
		*/

		/* modify the lru stack by deleting this extent */
		/* figure out the lru distance (bytes ahead of it) in the
		    lru tree (this is done by searching UP the tree, starting
		    at this node), then delete the node with that many bytes
		    ahead of it */
		bytesAhead = changeByte(nodePtr->lruPtr, 0);
		(*resultBytesAheadPtr) = bytesAhead;

		(*lruRootPtrPtr) = deleteLru(*lruRootPtrPtr, bytesAhead);

	    } else if (start > nodePtr->start && end < nodePtr->end) {
		/* split the extent into 3 parts */
		/*
		printf("split into 3 parts type match\n");
		*/

		/* prepare the new node (right part of extent) */
		/* create the new node for the lru stack */
		lruNewNodePtr = (TWOTHREENODELRU *)
					malloc(sizeof(TWOTHREENODELRU));
		lruNewNodePtr->left = NULL; /* this is a leaf */
		lruNewNodePtr->start = end + 1;
		lruNewNodePtr->end = nodePtr->end;
		lruNewNodePtr->id = lruNodeId++;
		/* new fix */
		lruNewNodePtr->file = file;

		newNodePtr = (TWOTHREENODE *) malloc(sizeof(TWOTHREENODE));
		newNodePtr->id = addressNodeId++;
		newNodePtr->left = NULL; /* this is a leaf */
		newNodePtr->start = end + 1;
		newNodePtr->end = nodePtr->end;
		newNodePtr->lruPtr = lruNewNodePtr;

		lruNewNodePtr->addressPtr = newNodePtr;

		bytesShorter = nodePtr->end - start + 1;

		/* change the left part of the extent */
		/* note that we can blithely change this without worrying
		    about updating the nodes in the address 2-3 tree,
		    since we're only changing the end (not the starting
		    address) */
		nodePtr->end = start - 1;

		/* update the lru leaf for the left part of the extent */
		nodePtr->lruPtr->end = nodePtr->end;

		/* update the lru tree byte totals to account for change
		    to the left part of the extent */
		
		bytesAhead = changeByte(nodePtr->lruPtr, bytesShorter);

		bytesAhead += nodePtr->end - nodePtr->start + 1;

		(*resultBytesAheadPtr) = bytesAhead;

		/* insert a new address node (the right part of the extent) */
		(*addressRootPtrPtr) = insertAddress(*addressRootPtrPtr,
							newNodePtr);
		numLeaves[file]++;
		/*
		printf("incrementing numLeavesLru: was %d", numLeavesLru);
		*/
		numLeavesLru++;
		/*
		printf(", now %d\n", numLeavesLru);
		*/

		/* insert a new node (the right part of the extent) in the
		    lru tree */
		(*lruRootPtrPtr) = insertLru(*lruRootPtrPtr, lruNewNodePtr,
						bytesAhead);

	    } else {
		/* split the extent into 2 parts */
		/*
		printf("split into 2 parts type match\n");
		*/
		/* change the extent in the address 2-3 tree, but you don't
		    have to delete it or insert anything else (except for the
		    new extent, taken care of in main routine) */
		if (start <= nodePtr->start) {
		    /* no shortcuts in this one.  We have to update the
			address 2-3 tree by deleting the old node and
			inserting the new one */
		    /*
		    printf("type 1\n");
		    */
		    newNodePtr = (TWOTHREENODE *) malloc(sizeof(TWOTHREENODE));
		    newNodePtr->id = addressNodeId++;
		    newNodePtr->left = NULL; /* this is a leaf */
		    newNodePtr->start = end + 1;
		    newNodePtr->end = nodePtr->end;
		    newNodePtr->lruPtr = nodePtr->lruPtr;
		    newNodePtr->lruPtr->addressPtr = newNodePtr; /* ??? */
		    /* new fix */
		    newNodePtr->lruPtr->file = file;
		    /*nodePtr = newNodePtr; */

		    newNodePtr->lruPtr->start = newNodePtr->start;
		    newNodePtr->lruPtr->end = newNodePtr->end;

		    /* modify the byte totals in the lru stack for the
			unused part (gets smaller) */
		    bytesAhead = changeByte(nodePtr->lruPtr, bytesMatched);
		    (*resultBytesAheadPtr) = bytesAhead;

		    /* now modify the address tree (couldn't do it before,
			since we could lose the pointer to the right lru
			leaf in the shuffling around that happens in
			insertAddress (newNodePtr might no longer point to
			the right leaf)) */
		    (*addressRootPtrPtr) = deleteAddress(*addressRootPtrPtr,
						nodePtr->start);
		    (*addressRootPtrPtr) = insertAddress(*addressRootPtrPtr,
							newNodePtr);

		} else {
		    /* don't have to worry about updating the address nodes */
		    /*
		    printf("type 2\n");
		    */
		    (*resultBytesAheadPtr) = start - nodePtr->start;
		    nodePtr->end = start - 1;

		    /* the unused part stays where it was in the lru stack */
		    /* update the lru leaf */
		    nodePtr->lruPtr->end = nodePtr->end;

		    /* modify the byte totals in the lru stack for the
			unused part (gets smaller) */
		    bytesAhead = changeByte(nodePtr->lruPtr, bytesMatched);
		    (*resultBytesAheadPtr) += bytesAhead;
		}

	    }
	    return(1);
	}
	return(0);
    } else {
	/* this is a node (parent), thus it must have at least two children */
	/* as soon as you find a match, return(1), so the parent can
	    abort.  This continues up to the root, then the routine that
	    called doSearch in the first place can re-issue the call (since
	    the trees will have changed */

	if (start < nodePtr->minMiddle) {
	    /* search left branch */
	    status = doSearch(nodePtr->left, addressRootPtrPtr, lruRootPtrPtr,
				start, end, file, resultBytesMatchedPtr,
				resultBytesAheadPtr);
	    if (status) {
		return(1);
	    }
	}
	if (end >= nodePtr->minMiddle &&
	      (nodePtr->right == NULL || start <= nodePtr->minRight) ) {
	    status = doSearch(nodePtr->middle, addressRootPtrPtr, lruRootPtrPtr,
				start, end, file, resultBytesMatchedPtr,
				resultBytesAheadPtr);
	    if (status) {
		return(1);
	    }
	}
	if (nodePtr->right != NULL && end >= nodePtr->minRight) {
	    status = doSearch(nodePtr->right, addressRootPtrPtr, lruRootPtrPtr,
				start, end, file, resultBytesMatchedPtr,
				resultBytesAheadPtr);
	    if (status) {
		return(1);
	    }
	}
    }
    return(0);
}

findMissAddress(startPtr, endPtr, file, maxAddress)
    unsigned int *startPtr;
    unsigned int *endPtr;
    int file;
    unsigned int maxAddress;
{
    findMissAddressSub(startPtr, endPtr, maxAddress, addressRootPtr[file]);
    return;
}

findHitAddress(startPtr, endPtr, file, size, bytesAhead)
    unsigned int *startPtr;
    unsigned int *endPtr;
    int file;
    unsigned int size;
    unsigned int bytesAhead;
{
    findHitAddressSub(startPtr, endPtr, file, size, bytesAhead, lruRootPtr);
    return;
}

printLeavesTree(file, printAll)
    int file;
    int printAll;
{
    printLeaves(addressRootPtr[file], printAll);
}

printLeavesLruTree(printAll)
    int printAll;
{
    printLeavesLru(lruRootPtr, printAll);
}

/* --- routines for checking (but not changing) the locality of a request --- */

int describeLoc(file, start, length, hitDistPtr)
    int file;
    unsigned int start;
    unsigned int length;
    double *hitDistPtr;
{
    /* checks the locality of this request (doesn't change any trees)
	returns 1 if this request is a hit, 0 if this request is a miss
	also returns the hit distance
    */

    unsigned int end;

    double totalBytesMatched, totalDistMatched;

    int doSearch();

    if (file >= MAXFILENUM) {
	printf("error: file number %d too large for array\n", file);
	exit(1);
    }
    end = start + length - 1;
    /*
    printf("\n---- file %d: %d - %d ----\n", file, start, end);
    */

    /* search for this extent in the address 2-3 tree */
    totalBytesMatched = totalDistMatched = 0;
    describeSearch(addressRootPtr[file], &(addressRootPtr[file]),
		&lruRootPtr, start, end, file,
		&totalBytesMatched, &totalDistMatched);

    /*
    printf("\nchecking address tree for file %d\n", file);
    checkAddressRoot(addressRootPtr[file], numLeaves[file]);

    printf("\nchecking lru tree\n");
    printf("numLeavesLru = %d\n", numLeavesLru);
    checkLruRoot(lruRootPtr, numLeavesLru, maxFileNumSeen);
    */

    /* maintain locality information for this run */
    if (totalBytesMatched / length < .5) {
	/* more miss than hit */
	return(0);
    } else {
	/* more hit than miss */
	(*hitDistPtr) = totalDistMatched/totalBytesMatched;
	return(1);
    }
}

int describeSearch(nodePtr, addressRootPtrPtr, lruRootPtrPtr, start, end, file,
	totalBytesMatchedPtr, totalDistMatchedPtr)
    TWOTHREENODE *nodePtr;
    TWOTHREENODE **addressRootPtrPtr;
    TWOTHREENODELRU **lruRootPtrPtr;
    unsigned int start;
    unsigned int end;
    int file;
    double *totalBytesMatchedPtr;
    double *totalDistMatchedPtr;
{
    /* returns 1 if any bytes matched, 0 otherwise */

    /* search for this extent in the 2-3 tree starting at nodePtr
        if found in one branch, call describeSearch on that branch
	if found in more than one branch, call describeSearch on all branches
	it's in
    */

    unsigned int bytesAhead;
    unsigned int bytesMatched;
    int status;

    unsigned int changeByte();

    if (nodePtr == NULL) {
	return(0);
    }

    /*
    printf("called describeSearch on node #%d with extent [%u, %u], file=%d\n",
	nodePtr->id, start, end, file);
    */

    if (nodePtr->left == NULL) {
	/* this is a leaf */
	/* see if this extent matches this leaf */
	if (end >= nodePtr->start && start <= nodePtr->end) {
	    /*
	    printf("matched file %d, %d - %d\n", file,
		MAX(start, nodePtr->start), MIN(end, nodePtr->end));
	    printf("node #%d has extent [%d, %d]\n", nodePtr->id,
		nodePtr->start, nodePtr->end);
	    */
	    bytesMatched = MIN(end, nodePtr->end) - MAX(start, nodePtr->start)
			    + 1;
	    (*totalBytesMatchedPtr) += bytesMatched;
	    
	    if (start <= nodePtr->start && end >= nodePtr->end) {
		/* new extent is a superset of the old extent */
		/*
		printf("superset match\n");
		*/

		/* figure out the lru distance (bytes ahead of it) in the
		    lru tree (this is done by searching UP the tree, starting
		    at this node) */
		bytesAhead = changeByte(nodePtr->lruPtr, 0);

	    } else if (start > nodePtr->start && end < nodePtr->end) {
		/* split the extent into 3 parts */
		/*
		printf("split into 3 parts type match\n");
		*/

		bytesAhead = changeByte(nodePtr->lruPtr, 0);

		bytesAhead += start - nodePtr->start;

	    } else {
		/* split the extent into 2 parts */
		/*
		printf("split into 2 parts type match\n");
		*/
		if (start <= nodePtr->start) {
		    bytesAhead = changeByte(nodePtr->lruPtr, 0);
		} else {
		    /* don't have to worry about updating the address nodes */
		    /*
		    printf("type 2\n");
		    */
		    bytesAhead = start - nodePtr->start;

		    bytesAhead += changeByte(nodePtr->lruPtr, 0);
		}

	    }

	    (*totalDistMatchedPtr) +=
		    bytesMatched * ((double)bytesAhead + bytesMatched/2);
	    return(1);
	}
	return(0);
    } else {
	/* this is a node (parent), thus it must have at least two children */
	/* as soon as you find a match, return(1), so the parent can
	    abort.  This continues up to the root, then the routine that
	    called describeSearch in the first place can re-issue the call
	    (since the trees will have changed */

	if (start < nodePtr->minMiddle) {
	    /* search left branch */
	    status = describeSearch(nodePtr->left, addressRootPtrPtr,
			    lruRootPtrPtr, start, end, file,
			    totalBytesMatchedPtr, totalDistMatchedPtr);
	}
	if (end >= nodePtr->minMiddle &&
	      (nodePtr->right == NULL || start <= nodePtr->minRight) ) {
	    status |= describeSearch(nodePtr->middle, addressRootPtrPtr,
			    lruRootPtrPtr, start, end, file,
			    totalBytesMatchedPtr, totalDistMatchedPtr);
	}
	if (nodePtr->right != NULL && end >= nodePtr->minRight) {
	    status |= describeSearch(nodePtr->right, addressRootPtrPtr,
			    lruRootPtrPtr, start, end, file,
			    totalBytesMatchedPtr, totalDistMatchedPtr);
	}
    }
    return(status);
}

freeLoc()
{
    /* free all nodes in the lru and address trees */

    int file;

    freeLru(lruRootPtr);
    for (file=0; file<MAXFILENUM; file++) {
	freeAddress(addressRootPtr[file]);
    }
}
