// ************************
// trace.h
//
// DESCRIPTION:
//    Holds the class that holds the traces of temporally local disk accesses
//    and the associated counters for the traces.
//
// REVISION HISTORY:
//    0.0  12/01/00  Morris Marden   Initial Revision
// ************************

// *** GUARD ***

#ifndef TRACE_GUARD
#define TRACE_GUARD


// *** CLASS DEFINITIONS ***

// Trace: holds a trace of disk accesses
// if get the following disk accesses {23, 45, 45, 23, 33, 5}, then the data 
// structure will look like:
//   (vertical pointers are nextAltBlock, horizontal pointers are nextBlock)
//   23 - 45 - 45 - 23 - 33 - 5
//   |     |
//   |    33 - 5
//   |
//   45 - 45 - 23 - 33 - 5
//   |    | 
//   |    23 - 33 - 5
//   |
//   33 - 5
//   |
//   5  
class Trace {
 private:
  int   count;         // count of number of times seen trace
  int   blockNum;      // block that is held in trace

 public:  
  Trace *nextBlock;    // list of possible next blocks in the trace
  Trace *nextAltBlock; // next alternative block to use in parent's trace

  Trace (int theBlockNum);   // constructor
  ~Trace();                  // destructor

  void IncrCount();    // increments count of number of times seen
  int  GetCount();     // returns count of number of times seen block
  int  GetBlockNum();  // returns the block number for the block
  
  // Looks in the trace for a particular block.  If finds, returns a pointer
  //    to the trace holding the block.  Otherwise, returns NULL.
  Trace * BlockSearch(int searchBlockNum); 

  // collects stats on the trace
  void    CollectStats(int *totTraceHits, int *maxTraceHits, int *numTraces,
		       int maxTraceLen, int myLength);
};

// TracePtrList: list of pointers into the trace data structure
class TracePtrList {
 public:
  Trace        *myTracePtr;  // pointer to the element in the trace data struct
  TracePtrList *next;        // next pointer to a trace element in the list

  TracePtrList (Trace *thePtr);   // constructor
  ~TracePtrList();                // destructor
};

// StatsCollector: class that is used to collect stats on traces and hold the 
//    necessary data structures
class StatsCollector
{
 private:
  Trace        *theTraces;   // holds traces of temporally local accesses

 public:
  StatsCollector();          // constructor
  ~StatsCollector();         // destructor

  // simulate looking for traces of temporally local disk accesses from
  //    the trace of disk accesses and count how many times use each trace.
  void TraceSim(char *traceFileName, int maxTraceLen);
  // gather the stats and send them to a file
  void GatherStats(char *statsFileName, int maxTraceLen);
};

// *** END GUARD ***

#endif
