// ************************
// trace_stat.cc
//
// DESCRIPTION:
//    Calculates the statistics of traces of temporally local disk accesses
//    from a trace of disk accesses
//
// REVISION HISTORY:
//    0.0  12/02/00  Morris Marden   Initial Revision
// ************************

// *** INCLUDES ***

#include <stdlib.h>
#include <iostream.h>
#include <fstream.h>
#include <stream.h>
#include "trace.h"
#include "disk_access.h"


// *** SUBROUTINES ***

// -- StatsCollector --

// constructor for the stats collector class
StatsCollector::StatsCollector()
{
  theTraces = NULL;  // haven't collected any traces yet
}

// destructor for the stats collector class
StatsCollector::~StatsCollector()
{
  delete theTraces;
}

// TraceSim: simulate looking for traces of temporally local disk accesses from
//    the trace of disk accesses and count how many times use each trace.
void StatsCollector::TraceSim(char *traceFileName, int maxTraceLen)
{
  Trace        *currTrace = NULL; // current trace working with
  TracePtrList *actTraces = NULL; // holds list of the currently active traces
  TracePtrList *newActTraces;     // list of active traces for next iteration
  TracePtrList *endNewActTraces;  // points to last trace in new active traces
  TracePtrList *currActTrace;     // current active trace working with
  DiskAccess   *currAccess;       // holds the current disk access working with
  ifstream     *traceFile;        // file that holds trace of disk accesses
  int          traceLength;       // length of trace currently working with

  // open the disk trace file
  traceFile = new ifstream(traceFileName);
  if (!(*traceFile))
  {
    cerr << "Error, unable to open file \"" << traceFileName << "\"" << endl;
    exit(1);
  }

  // handle the disk accesses one by one until run out of accesses
  // get the first disk access
  currAccess = new DiskAccess(traceFile); 
  while (currAccess->GetBlockNum() != ACCESS_ERROR) 
  {
    // - count the length 1 trace (i.e., a trace holding the block itself) -
    // need to check if list of tracces is empty
    if (theTraces != NULL)
    {
      // first look for block in list of length 1 traces
      currTrace = theTraces->BlockSearch(currAccess->GetBlockNum());
      // if found the block, increment its count
      if (currTrace != NULL)
	currTrace->IncrCount();
      // otherwise add to the list of traces
      else
      {
	// create a new trace for the block
	currTrace = new Trace(currAccess->GetBlockNum());
	// add new trace to front of list of traces
	currTrace->nextAltBlock = theTraces;
	theTraces = currTrace;
      }
    }
    // since list of traces is empty, no trace for block, so create a new
    //   trace for
    else
    {
      theTraces = new Trace(currAccess->GetBlockNum());
      currTrace = theTraces;
    }
    // the length 1 active trace for next iteration is trace just worked with
    newActTraces = new TracePtrList(currTrace);
    endNewActTraces = newActTraces;
    
    // - count traces of length n by looking at the list of current traces -
    // work with each of the active traces in turn
    traceLength = 1;  // start with the length 1 trace
    currActTrace = actTraces;
    while (currActTrace != NULL)
    {
      // look for the block in the list of current traces inside current trace
      currTrace = currActTrace->myTracePtr->nextBlock->BlockSearch(currAccess->GetBlockNum());
      // if found the trace, then increment the count
      if (currTrace != NULL)
	currTrace->IncrCount();
      // otherwise, create a trace for the block
      else
      {
	// create a new trace for the block
	currTrace = new Trace(currAccess->GetBlockNum());
	// add to front of list of traces
	currTrace->nextAltBlock = currActTrace->myTracePtr->nextBlock;
	currActTrace->myTracePtr->nextBlock = currTrace;
      }

      // if the new trace is length < max trace length, add to list of new 
      //   active traces
      if (traceLength < maxTraceLen)
      {
	endNewActTraces->next = new TracePtrList(currTrace);
	endNewActTraces = endNewActTraces->next;
      }

      // get ready for next iteration
      traceLength++;                      // next trace has longer length
      currActTrace = currActTrace->next;  // go to next active trace
    }

    // set up for next iteration
    actTraces = newActTraces;               // set active traces for next iter
    delete currAccess;                      // finished with access, so delete
    currAccess = new DiskAccess(traceFile); // get the next disk access
  }

  // clean up
  traceFile->close();
  delete traceFile;
  delete currAccess;
  delete currTrace;
  delete actTraces;
//  delete newActTraces;
//  delete endNewActTraces;
}

// GatherStats: look at the traces and the number of times each was accessed
//   to figure out the stats for the traces.  Send the results to a file.
void StatsCollector::GatherStats(char *statsFileName, int maxTraceLen)
{
  ofstream   *statsFile;    // file to send stats to
  bool       createdFile;   // holds whether was able to create the stats file
  Trace      *currTrace;    // current trace working with
  int        loop;          // loop var

  int        *totTraceHits; // average # of hits for diff lengths of traces
  int        *maxTraceHits; // maxmimum # of hits for diff lengths of traces
  int        *numTraces;    // # of different traces for diff lengths

  // create the file
  statsFile = new ofstream(statsFileName);
  if (*statsFile)
    createdFile = true;
  else
  {
    createdFile = false;
    cerr << "Warning, unable to create file \"" << statsFileName << "\".  ";
    cerr << "Sending results to standard out instead." << endl;
  }

  // allocate space for the data structs to hold stats on each trace
  totTraceHits = new int[maxTraceLen];
  maxTraceHits = new int[maxTraceLen];
  numTraces    = new int[maxTraceLen];

  // take a look at each of the traces in turn
  currTrace = theTraces;
  while (currTrace != NULL)
  {
    // clear data structs so can collect new stats for the current trace
    for (loop = 0; loop < maxTraceLen; loop++)
    {
      totTraceHits[loop] = 0;
      maxTraceHits[loop] = 0;
      numTraces[loop]    = 0;
    }

    // collect the stats for the trace
    // first collect stats for the length 1 trace 
    totTraceHits[0] = currTrace->GetCount();  
    maxTraceHits[0] = totTraceHits[0];
    numTraces[0] = 1;
    // now collect stats for > 1 lengths
    if (currTrace->nextBlock != NULL)
      currTrace->nextBlock->CollectStats(totTraceHits, maxTraceHits,  
					 numTraces, maxTraceLen, 2);

    // send output to the file
    if (createdFile)
    {
      // start block for trace
      *statsFile << currTrace->GetBlockNum() << "  \t";  
      
      // send stats for different lengths
      for (loop = 0; loop < maxTraceLen; loop++)
      {
	// total # of hits in the traces
	*statsFile << totTraceHits[loop] << " ";
	// max number of times used each trace
	*statsFile << maxTraceHits[loop] << " ";
	// number of different traces
	*statsFile << numTraces[loop] << "\t";
      }
      *statsFile << endl;
    }
    else
    {
      cout << currTrace->GetBlockNum() << "\t";  // start block for trace

      // send stats for different lengths
      for (loop = 0; loop < maxTraceLen; loop++)
      {
	// average number of times used each trace
	cout << totTraceHits[loop] << " ";
	// max number of times used each trace
	cout << maxTraceHits[loop] << " ";
	// number of different traces
	cout << numTraces[loop] << "\t";
      }
      cout  << endl;
    }

    // set up for next iteration
//    delete totTraceHits;
//    delete maxTraceHits;
//    delete numTraces;
    currTrace = currTrace->nextAltBlock;    // go to the next trace
  }

  // clean up: 
  currTrace = NULL;  // set pointers to null so won't delete on exit
  statsFile->close();
  delete statsFile;
} 

// -- Trace --

// CollectStats: collect stats on the different length traces using a depth
//   first search
void Trace::CollectStats(int *totTraceHits, int *maxTraceHits, int *numTraces,
			 int maxTraceLen, int myLength)
{
  // if length of trace < the maximum trace length, then collect stats on the 
  //   next block in the trace
  if (myLength < maxTraceLen)
  {
    // check first that there is another block in the collected traces
    if (nextBlock != NULL)
    {
      nextBlock->CollectStats(totTraceHits, maxTraceHits, numTraces, 
			      maxTraceLen, myLength + 1);
    }
  }

  // add own stats to set of stats
  // increment count of number of traces
  numTraces[myLength - 1]++;
  // add hit count to total stats
  totTraceHits[myLength - 1] = totTraceHits[myLength - 1] + count;  
  // check if current trace has more hits than previous max
  if (count > maxTraceHits[myLength - 1])
    maxTraceHits[myLength - 1] = count;

  // if there is another trace of the same length collect stats on it
  if (nextAltBlock != NULL)
    nextAltBlock->CollectStats(totTraceHits, maxTraceHits, numTraces, 
			       maxTraceLen, myLength);
}
