/*
  BlatherUserTester

  Created by Will Benton on 2007-08-08.
  Copyright (c) 2007 Aition Technologies, LLC and Will Benton. All rights reserved.
 */

package blather.tests;

import org.junit.*;
import static org.junit.Assert.*;
import blather.model.BlatherUser;
import blather.model.EmailAddressInterface;
import blather.model.BlabInterface;
import blather.model.SimpleDate;
import blather.exceptions.BogusParameterException;

/**
 *  BlatherUserTester
 *
 * @author Will Benton
 * @version $Rev$
 */

public final class BlatherUserTester {

	private static class TestEmailAddress implements EmailAddressInterface {
		
	    private boolean validated;
	    private String email;

	    /**
	     * Determines whether or not a given String corresponds to a valid email address.  A valid email address:
	     * <ol>
	     * <li>Is of the general form <i>username</i><tt>@</tt><i>domain</i>. 
	     * That is, it has exactly one "@" sign, and valid username and domain sections.</li>
	     * <li>A valid <i>username</i> consists of at least one non-@ character.
	     * <li>A valid <i>domain</i> consists of at least one period and has at least one non-period character to the immediate left and right of the period.  
	     * </ol>
	     * Therefore, <tt>fred@bedrock.gov</tt> is a valid email address, but <tt>fred123</tt>, <tt>f@r@ed.com</tt>, and <tt>fred@bedrock</tt> are not.
	     * @param address
	     * @return true if and only if address is valid
	     */
	    public static boolean isValid(String address) {
	        if (address == null || 
	            address.equals("") || 
	            address.charAt(address.length() -1) == '.') return false;

	        String parts[] = address.split("@");
	        if(parts.length == 2 && parts[0].length() > 0) {
	            String domainparts[] = parts[1].split("\\.");
	            if (domainparts.length > 1) {
	                for (String dpart : domainparts) {
	                    if (dpart.length() == 0) return false;
	                }
	                return true;
	            }
	        }
	        return false;
	    }


	    /** Creates a new EmailAddress from the given String, which must be a 
	     *  valid email address, as defined by EmailAddress.isValid(String).  
	     *  Throws a BogusParameterException if <tt>email</tt> is not a valid 
	     *  email address.
	     * 
	     * @param email a valid email address
	     */
	    public TestEmailAddress(String email) {
	        this.setEmail(email);
	    }

	    /* (non-Javadoc)
	     * @see rsvp.model.EmailAddressInterface#toString()
	     */
	    public String toString() {
	        return email;
	    }   

	    /* (non-Javadoc)
	     * @see rsvp.model.EmailAddressInterface#setEmail(java.lang.String)
	     */
	    public void setEmail(String email) {
	        if(!isValid(email)) {
	            throw new BogusParameterException(email + "is not a valid email address");
	        }
	        this.validated = true;
	        this.email = email;
	    }

	    /* (non-Javadoc)
	     * @see rsvp.model.EmailAddressInterface#getDisplayEmail()
	     */
	    public String getDisplayEmail() {
	        if(!validated) throw new IllegalStateException("This object has not been initialized with a valid email address!");
	        String emailparts[] = email.split("@");
	        String domainparts[] = emailparts[1].split("\\."); 
	        return emailparts[0] + "@" + domainparts[0].charAt(0) + "..." + domainparts[domainparts.length-1];
	    }

	    /* (non-Javadoc)
	     * @see rsvp.model.EmailAddressInterface#getEmail()
	     */
	    public String getEmail() {
	        if(!validated) throw new IllegalStateException("This object has not been initialized with a valid email address!");
	        return email;
	    }

	}

	private static class TestBlab implements BlabInterface {
		public TestBlab(String c, String n) { }
	    public String getContent() {return null;}
	    public SimpleDate getDate() {return null;}
	    public String getUserName() {return null;}

	}

	private static final String USERNAME = "fritz";
	private static final String PASSWORD = "frotz";

	BlatherUser bu;

	@Before
	public void fixtureSetup() { bu = new BlatherUser(USERNAME, PASSWORD, new TestEmailAddress("argh@blargh.com")); }

	@After
	public void fixtureTeardown() { bu = null; }

	@Test
	public void noFriendsAtCtor() {
		assertTrue("newly-created BlatherUser should have no friends", bu.friendCount() == 0);
	}

	@Test
	public void noBlabsAtCtor() {
		assertTrue("newly-created BlatherUser should have no blabs", bu.blabCount() == 0);
	}

	@Test
	public void passwordCheckOK() { 
		assertTrue("password check doesn't succeed when called with correct password", bu.checkPassword(PASSWORD));
	}

	@Test
	public void passwordCheckEquals() { 
		assertTrue("password check appears to be using referential equality (should use .equals())", bu.checkPassword(new String(PASSWORD)));
	}

	@Test
	public void addOneFriend() { 
		bu.addFriend(bu);
		assertTrue("added one friend; friend count should be one", bu.friendCount() == 1);
	}

	@Test
	public void getOneFriend() { 
		bu.addFriend(bu);
		assertTrue("didn't get the friend we added", bu.getFriend(0) == bu);
	}


	@Test
	public void addManyFriends() { 
		final int FRIENDS = 100000;
		for (int i = 0; i < FRIENDS; i++) 
			bu.addFriend(bu);
		assertTrue("added many friends; friend count should be " + FRIENDS, bu.friendCount() == FRIENDS);
	}

	
	@Test
	public void addOneBlab() { 
		bu.addBlab(null);
		assertTrue("added one blab; blab count should be one", bu.blabCount() == 1);
	}
	
	@Test
	public void addManyBlabs() { 
		final int BLABS = 100000;
		for (int i = 0; i < BLABS; i++) 
			bu.addBlab(null);
		assertTrue("added many blabs; blab count should be " + BLABS, bu.blabCount() == BLABS);
	}
	
	private static boolean userContainsBlabs(BlatherUser b, BlabInterface[] blabs) {
		for(int i = 0; i < b.blabCount(); i++)
			if(blabs[i] != b.getBlab(i)) return false;
		return true;
	}
	
	private static boolean noNullBlabs(BlatherUser b) {
		for(int i = 0; i < b.blabCount(); i++)
			if (null == b.getBlab(i)) return false;
		return true;
	}
	
	@Test
	public void addManyBlabsAndGetBlabs() { 
		int BLABS = 3745;
		BlabInterface[] blabs = new BlabInterface[BLABS];
		for (int i = 0; i < BLABS; i++) {
			blabs[i] = new TestBlab(USERNAME, "I like to eat paste");
			bu.addBlab(blabs[i]);
		}
		assertTrue("user should contain all added blabs, in order", userContainsBlabs(bu, blabs));
	}
	
	@Test
	public void getAllBlabsShouldReturnCopy() { 
		int BLABS = 3745;
		BlabInterface[] blabs = new BlabInterface[BLABS];
		for (int i = 0; i < BLABS; i++) {
			blabs[i] = new TestBlab(USERNAME, "I like to eat paste");
			bu.addBlab(blabs[i]);
		}
		BlabInterface [] blabs2 = bu.getAllBlabs();
		for (int i = 0; i < BLABS; i++) {
			blabs2[i] = null;
		}
		assertTrue("getAllBlabs should return a copy of the blab array", noNullBlabs(bu));
	}
}